import React, { useMemo, useCallback } from "react";
import PropTypes from "prop-types";
import invariant from "invariant";
import Events from "../../../__internal__/utils/helpers/events";
import Accordion from "../accordion.component";
import { StyledAccordionGroup } from "../accordion.style";

// typescript-to-proptypes breaks on recursive type references so it has to be an interface
// eslint-disable-next-line @typescript-eslint/no-empty-interface

export const AccordionGroup = ({
  children,
  ...rest
}) => {
  const hasProperChildren = useMemo(() => {
    let hasAccordionChildren = true;
    React.Children.toArray(children).forEach(child => {
      if (typeof child === "string" || /*#__PURE__*/React.isValidElement(child) && child.type.displayName !== "Accordion") {
        hasAccordionChildren = false;
      }
    });
    return hasAccordionChildren;
  }, [children]);
  !hasProperChildren ? process.env.NODE_ENV !== "production" ? invariant(false, `AccordionGroup accepts only children of type \`${Accordion.displayName}\`.`) : invariant(false) : void 0;
  const filteredChildren = useMemo(() => React.Children.toArray(children).filter(child => {
    return /*#__PURE__*/React.isValidElement(child);
  }), [children]);
  const refs = useMemo(() => filteredChildren.map(child => child.ref || /*#__PURE__*/React.createRef()), [filteredChildren]);
  const focusAccordion = useCallback((ev, index) => {
    ev.preventDefault();
    if (index === -1) {
      refs[refs.length - 1].current?.focus();
    } else if (index === refs.length) {
      refs[0].current?.focus();
    } else {
      refs[index].current?.focus();
    }
  }, [refs]);
  const handleKeyboardAccessibility = useCallback((ev, index) => {
    if (Events.isUpKey(ev)) {
      focusAccordion(ev, index - 1);
    }
    if (Events.isDownKey(ev)) {
      focusAccordion(ev, index + 1);
    }
    if (Events.isHomeKey(ev)) {
      focusAccordion(ev, 0);
    }
    if (Events.isEndKey(ev)) {
      focusAccordion(ev, refs.length - 1);
    }
  }, [focusAccordion, refs]);
  return /*#__PURE__*/React.createElement(StyledAccordionGroup, rest, filteredChildren.map((child, index) =>
  /*#__PURE__*/
  // casted to ReactElement as there is no overload for an FunctionComponentElement in cloneElement
  React.cloneElement(child, {
    ref: refs[index],
    index,
    handleKeyboardAccessibility
  })));
};
AccordionGroup.displayName = "AccordionGroup";
export default AccordionGroup;