import React, { useContext, useEffect, useRef } from "react";
import PropTypes from "prop-types";
import ReactDOM from "react-dom";
import { flip } from "@floating-ui/dom";
import useFloating from "../../hooks/__internal__/useFloating";
import { StyledBackdrop, StyledPopoverContent } from "./popover.style";
import CarbonScopedTokensProvider from "../../style/design-tokens/carbon-scoped-tokens-provider/carbon-scoped-tokens-provider.component";
import { ModalContext } from "../../components/modal/modal.component";
const defaultMiddleware = [flip({
  fallbackStrategy: "initialPlacement"
})];
const Popover = ({
  children,
  placement,
  disablePortal,
  reference,
  middleware = defaultMiddleware,
  disableBackgroundUI,
  isOpen = true,
  animationFrame,
  popoverStrategy = "absolute",
  childRefOverride
}) => {
  const elementDOM = useRef(null);
  const {
    isInModal
  } = useContext(ModalContext);
  const candidateNode = reference.current?.closest("[role='dialog']");
  const mountNode = isInModal && candidateNode ? candidateNode : document.body;
  if (!elementDOM.current && !disablePortal) {
    elementDOM.current = document.createElement("div");
    mountNode.appendChild(elementDOM.current);
  }
  const childRef = childRefOverride || React.Children.only(children).ref;
  const innerRef = useRef(null);
  const floatingReference = childRef || innerRef;
  let content;
  if (childRef) {
    content = children;
  } else {
    content = /*#__PURE__*/React.cloneElement(children, {
      ref: floatingReference
    });
  }
  useFloating({
    isOpen,
    reference,
    floating: floatingReference,
    placement,
    middleware,
    animationFrame,
    strategy: popoverStrategy
  });
  useEffect(() => {
    return () => {
      if (!disablePortal && elementDOM.current) {
        mountNode.removeChild(elementDOM.current);
        elementDOM.current = null;
      }
    };
  }, [disablePortal, mountNode]);
  if (!disableBackgroundUI) {
    content = /*#__PURE__*/React.createElement(StyledPopoverContent, {
      isOpen: isOpen
    }, content);
  }
  if (disableBackgroundUI) {
    content = /*#__PURE__*/React.createElement(StyledPopoverContent, {
      isOpen: isOpen
    }, /*#__PURE__*/React.createElement(StyledBackdrop, null, content));
  }
  if (disablePortal) {
    return content;
  }
  return /*#__PURE__*/ReactDOM.createPortal( /*#__PURE__*/React.createElement(CarbonScopedTokensProvider, null, content), elementDOM.current);
};
export default Popover;