import React, { useCallback, useContext, useEffect, useMemo, useRef, useState } from "react";
import PropTypes from "prop-types";
import { defaultFocusableSelectors, setElementFocus, onTabGuardFocus, trapFunction } from "./focus-trap-utils";
import { ModalContext } from "../../components/modal/modal.component";
import usePrevious from "../../hooks/__internal__/usePrevious";
import TopModalContext from "../../components/carbon-provider/top-modal-context";
export const TAB_GUARD_TOP = "tab-guard-top";
export const TAB_GUARD_BOTTOM = "tab-guard-bottom";

// TODO investigate why React.RefObject<T> produces a failed prop type when current = null

const FocusTrap = ({
  children,
  autoFocus = true,
  focusableSelectors,
  focusFirstElement,
  bespokeTrap,
  wrapperRef,
  isOpen,
  additionalWrapperRefs
}) => {
  const trapRef = useRef(null);
  const [currentFocusedElement, setCurrentFocusedElement] = useState();
  const {
    isAnimationComplete = true,
    triggerRefocusFlag
  } = useContext(ModalContext);
  const {
    topModal
  } = useContext(TopModalContext);
  // we ensure that isTopModal is true if there is no TopModalContext, so that consumers who have not
  // wrapped their app in CarbonProvider do not have all FocusTrap behaviour broken
  const isTopModal = !topModal || wrapperRef.current && topModal.contains(wrapperRef.current);
  const trapWrappers = useMemo(() => additionalWrapperRefs?.length ? [wrapperRef, ...additionalWrapperRefs] : [wrapperRef], [additionalWrapperRefs, wrapperRef]);
  const onTabGuardTopFocus = useMemo(() => onTabGuardFocus(trapWrappers, focusableSelectors, "top"), [focusableSelectors, trapWrappers]);
  const onTabGuardBottomFocus = useMemo(() => onTabGuardFocus(trapWrappers, focusableSelectors, "bottom"), [focusableSelectors, trapWrappers]);
  useEffect(() => {
    additionalWrapperRefs?.forEach(ref => {
      const {
        current: containerElement
      } = ref;
      // istanbul ignore else
      if (containerElement) {
        // istanbul ignore else
        if (!containerElement.previousElementSibling?.matches(`[data-element=${TAB_GUARD_TOP}]`)) {
          const topTabGuard = document.createElement("div");
          topTabGuard.tabIndex = 0;
          topTabGuard.dataset.element = TAB_GUARD_TOP;
          containerElement.insertAdjacentElement("beforebegin", topTabGuard);
          topTabGuard.addEventListener("focus", onTabGuardTopFocus(ref));
        }
        // istanbul ignore else
        if (!containerElement.nextElementSibling?.matches(`[data-element=${TAB_GUARD_BOTTOM}]`)) {
          const bottomTabGuard = document.createElement("div");
          bottomTabGuard.tabIndex = 0;
          bottomTabGuard.dataset.element = TAB_GUARD_BOTTOM;
          containerElement.insertAdjacentElement("afterend", bottomTabGuard);
          bottomTabGuard.addEventListener("focus", onTabGuardBottomFocus(ref));
        }
      }
    });
    return () => {
      additionalWrapperRefs?.forEach(ref => {
        const previousElement = ref.current?.previousElementSibling;
        if (previousElement?.matches(`[data-element=${TAB_GUARD_TOP}]`)) {
          previousElement.remove();
        }
        const nextElement = ref.current?.nextElementSibling;
        if (nextElement?.matches(`[data-element=${TAB_GUARD_BOTTOM}]`)) {
          nextElement.remove();
        }
      });
    };
  }, [additionalWrapperRefs, onTabGuardTopFocus, onTabGuardBottomFocus]);
  const shouldSetFocus = autoFocus && isOpen && isAnimationComplete;
  const prevShouldSetFocus = usePrevious(shouldSetFocus);
  const getFocusableElements = useCallback(selector => {
    const elements = [];
    trapWrappers.forEach(ref => {
      // istanbul ignore else
      if (ref.current) {
        elements.push(...Array.from(ref.current.querySelectorAll(selector)).filter(el => Number(el.tabIndex) !== -1));
      }
    });
    return elements;
  }, [trapWrappers]);
  useEffect(() => {
    if (shouldSetFocus && !prevShouldSetFocus) {
      const candidateFirstElement = focusFirstElement && "current" in focusFirstElement ? focusFirstElement.current : focusFirstElement;
      const autoFocusedElement = getFocusableElements(defaultFocusableSelectors).find(el => el.getAttribute("data-has-autofocus") === "true");
      const elementToFocus = candidateFirstElement || autoFocusedElement || wrapperRef.current;
      // istanbul ignore else
      if (elementToFocus) {
        setElementFocus(elementToFocus);
      }
    }
  }, [shouldSetFocus, prevShouldSetFocus, getFocusableElements, focusFirstElement, wrapperRef]);
  useEffect(() => {
    const trapFn = ev => {
      // block focus trap from working if it's not the topmost trap, or is currently closed
      if (!isTopModal || !isOpen) {
        return;
      }
      const focusableElements = getFocusableElements(defaultFocusableSelectors);
      trapFunction(ev, focusableElements, document.activeElement === wrapperRef.current, focusableSelectors, bespokeTrap);
    };
    document.addEventListener("keydown", trapFn, true);
    return function cleanup() {
      document.removeEventListener("keydown", trapFn, true);
    };
  }, [bespokeTrap, wrapperRef, focusableSelectors, getFocusableElements, isTopModal, isOpen]);
  const updateCurrentFocusedElement = useCallback(() => {
    const focusableElements = getFocusableElements(focusableSelectors || defaultFocusableSelectors);
    const element = focusableElements?.find(el => el === document.activeElement);
    if (element) {
      setCurrentFocusedElement(element);
    }
  }, [getFocusableElements, focusableSelectors]);
  const refocusTrap = useCallback(() => {
    if (currentFocusedElement && !currentFocusedElement.hasAttribute("disabled")) {
      // the trap breaks if it tries to refocus a disabled element
      setElementFocus(currentFocusedElement);
    } else if (wrapperRef.current?.hasAttribute("tabindex")) {
      setElementFocus(wrapperRef.current);
    } else {
      const focusableElements = getFocusableElements(focusableSelectors || defaultFocusableSelectors);
      /* istanbul ignore else */
      if (focusableElements.length) {
        setElementFocus(focusableElements[0]);
      }
    }
  }, [currentFocusedElement, wrapperRef, focusableSelectors, getFocusableElements]);
  useEffect(() => {
    if (triggerRefocusFlag) {
      refocusTrap();
    }
  }, [triggerRefocusFlag, refocusTrap]);
  const [tabIndex, setTabIndex] = useState(0);
  useEffect(() => {
    // tabIndex is set to 0 and removed on blur.
    if (!isOpen) {
      setTabIndex(0);
    }
  }, [isOpen]);
  const onBlur = () => {
    /* istanbul ignore else */
    if (isOpen) {
      setTabIndex(undefined);
    }
  };
  const focusProps = hasNoTabIndex => ({
    ...(hasNoTabIndex && {
      tabIndex,
      onBlur
    }),
    onFocus: updateCurrentFocusedElement
  });

  // passes focusProps, sets tabIndex and onBlur if no tabIndex has been expicitly set on child
  const clonedChildren = React.Children.map(children, child => {
    const focusableChild = child;
    return /*#__PURE__*/React.cloneElement(focusableChild, focusProps(focusableChild?.props?.tabIndex === undefined));
  });
  return /*#__PURE__*/React.createElement("div", {
    ref: trapRef
  }, isOpen && /*#__PURE__*/React.createElement("div", {
    "data-element": TAB_GUARD_TOP
    // eslint-disable-next-line jsx-a11y/no-noninteractive-tabindex
    ,
    tabIndex: 0,
    onFocus: onTabGuardTopFocus(wrapperRef)
  }), clonedChildren, isOpen && /*#__PURE__*/React.createElement("div", {
    "data-element": TAB_GUARD_BOTTOM
    // eslint-disable-next-line jsx-a11y/no-noninteractive-tabindex
    ,
    tabIndex: 0,
    onFocus: onTabGuardBottomFocus(wrapperRef)
  }));
};
export default FocusTrap;