import { merge } from 'lodash';
import { isUserClientCustomizationDisabled } from '../utils/feature-toggles';
import { isDevMode } from '../utils/window';
export const USER_CLIENT_CUSTOMIZATION_KEY = 'userClientCustomizations';
const USER_CLIENT_CUSTOMIZATION_DEVMODE_WARNING = 'User client customization is disabled in development mode.';
// INFO: Using promises here as eventually these configurations may be handled by the server instead
//       of being stored/fetched from local storage.
const fetchUserClientCustomizations = () => {
    if (isUserClientCustomizationDisabled()) {
        return Promise.reject();
    }
    try {
        const serialized = localStorage.getItem(USER_CLIENT_CUSTOMIZATION_KEY);
        if (serialized) {
            const customizations = JSON.parse(serialized);
            return Promise.resolve(customizations);
        }
        return Promise.reject();
    }
    catch {
        return Promise.reject();
    }
};
// INFO: Using promises here as eventually these configurations may be handled by the server instead
//       of being stored/fetched from local storage.
const storeUserClientCustomizations = (customizations) => {
    if (isUserClientCustomizationDisabled()) {
        return Promise.reject();
    }
    try {
        const serialized = JSON.stringify(customizations);
        localStorage.setItem(USER_CLIENT_CUSTOMIZATION_KEY, serialized);
        return Promise.resolve();
    }
    catch {
        return Promise.reject();
    }
};
export const getUserClientCustomization = async (username, screenId, elementId, type) => {
    if (isUserClientCustomizationDisabled()) {
        if (isDevMode()) {
            // eslint-disable-next-line no-console
            console.warn(USER_CLIENT_CUSTOMIZATION_DEVMODE_WARNING);
        }
        return null;
    }
    try {
        const customizations = await fetchUserClientCustomizations();
        const customization = customizations?.[username]?.[screenId]?.[elementId]?.[type];
        if (customization) {
            return customization;
        }
        return null;
    }
    catch {
        // eslint-disable-next-line no-console
        console.warn('An error occurred while fetching from local storage. Unable to store user client customization.');
        return null;
    }
};
export const setUserClientCustomization = async (username, screenId, elementId, type, data) => {
    if (isUserClientCustomizationDisabled()) {
        if (isDevMode()) {
            // eslint-disable-next-line no-console
            console.warn(USER_CLIENT_CUSTOMIZATION_DEVMODE_WARNING);
        }
        return null;
    }
    const customization = { data, elementId, screenId, type };
    let storedCustomizations;
    try {
        storedCustomizations = await fetchUserClientCustomizations();
    }
    catch {
        storedCustomizations = undefined;
    }
    const customizationStorage = {
        [username]: { [screenId]: { [elementId]: { [type]: customization } } },
    };
    const customizations = storedCustomizations
        ? merge(storedCustomizations, customizationStorage)
        : customizationStorage;
    // INFO: While customization storage structure should be merged, the data of the current customization should not
    //       merge with, but rather override the previous customization.
    customizations[username][screenId][elementId][type] = customization;
    try {
        await storeUserClientCustomizations(customizations);
        return customization;
    }
    catch {
        // eslint-disable-next-line no-console
        console.warn('An error occurred while saving to local storage. Unable to store user client customization.');
        return null;
    }
};
export const clearUserClientCustomization = async (username, screenId, elementId, type) => {
    if (isUserClientCustomizationDisabled()) {
        if (isDevMode()) {
            // eslint-disable-next-line no-console
            console.warn(USER_CLIENT_CUSTOMIZATION_DEVMODE_WARNING);
        }
        return;
    }
    const customizations = await fetchUserClientCustomizations();
    if (customizations?.[username]?.[screenId]?.[elementId]?.[type]) {
        delete customizations[username][screenId][elementId][type];
    }
    try {
        await storeUserClientCustomizations(customizations);
    }
    catch {
        // eslint-disable-next-line no-console
        console.warn('An error occurred while saving to local storage. Unable to store user client customization.');
    }
};
//# sourceMappingURL=user-client-customization-service.js.map