import { checkIfPageIsLoaded } from '../utils/state-utils';
import { mapValues } from 'lodash';
import { getStore } from '../redux/store';
import { xtremConsole } from '../utils/console';
import * as transactionActions from '../redux/actions/transaction-actions';
import { NEW_PAGE } from '../utils/constants';
const transactionCommitReferences = {};
class ReadOnlyTransactionError extends Error {
}
let isReadOnly = false;
const transactions = {};
export const commitTransaction = (screenId) => {
    const transaction = transactions[screenId];
    if (transaction) {
        const store = getStore();
        store.dispatch(transactionActions.commitTransaction(screenId, transaction));
        delete transactions[screenId];
        delete transactionCommitReferences[screenId];
    }
};
export const rollbackTransaction = (screenId) => {
    if (transactionCommitReferences[screenId]) {
        clearTimeout(transactionCommitReferences[screenId]);
        delete transactionCommitReferences[screenId];
    }
    if (transactions[screenId]) {
        delete transactions[screenId];
    }
};
export const isTransactionInProgress = (screenId) => !!transactions[screenId];
const transactionize = (from) => mapValues(from, value => ({
    hasChangedInTransaction: false,
    value,
}));
const startTransaction = (screenId) => {
    if (!transactions[screenId]) {
        const store = getStore();
        const state = store.getState();
        checkIfPageIsLoaded(screenId, state);
        transactions[screenId] = {
            values: transactionize(state.screenDefinitions[screenId].values),
            uiComponentProperties: transactionize(state.screenDefinitions[screenId].metadata.uiComponentProperties),
        };
        transactionCommitReferences[screenId] = window.setTimeout(() => commitTransaction(screenId));
    }
};
/* State setters and getters */
export const getFieldValue = (screenId, elementId) => {
    const state = getStore().getState();
    checkIfPageIsLoaded(screenId, state);
    let fieldValue;
    if (transactions[screenId]) {
        fieldValue = !transactions[screenId].values[elementId]
            ? undefined
            : transactions[screenId].values[elementId].value;
    }
    else {
        fieldValue = state.screenDefinitions[screenId].values[elementId];
    }
    if (elementId === '_id' && fieldValue === NEW_PAGE) {
        return null;
    }
    return fieldValue;
};
export const setFieldValue = (screenId, elementId, newValue) => {
    if (isReadOnly) {
        throw new ReadOnlyTransactionError(`It is forbidden to set a value of a field from a callback property. Updating other parts of your screen is only allowed by using event handlers. Screen ID: ${screenId}, element ID: ${elementId}`);
    }
    startTransaction(screenId);
    transactions[screenId].values[elementId] = {
        value: newValue,
        hasChangedInTransaction: true,
    };
};
export const getUiComponentProperties = (screenId, elementId) => {
    checkIfPageIsLoaded(screenId);
    let uiComponentProperties;
    if (transactions[screenId]) {
        uiComponentProperties = !transactions[screenId].uiComponentProperties[elementId]
            ? undefined
            : transactions[screenId].uiComponentProperties[elementId].value;
    }
    else {
        uiComponentProperties = getStore().getState().screenDefinitions[screenId].metadata.uiComponentProperties[elementId];
    }
    return uiComponentProperties;
};
export const setUiComponentProperties = (screenId, elementId, newValue) => {
    if (isReadOnly) {
        throw new ReadOnlyTransactionError(`It is forbidden to set a property of a field from a callback property. Updating other parts of your screen is only allowed by using event handlers. Screen ID: ${screenId}, element ID: ${elementId}`);
    }
    startTransaction(screenId);
    transactions[screenId].uiComponentProperties[elementId] = {
        value: newValue,
        hasChangedInTransaction: true,
    };
};
export const executeInReadOnlyTransaction = (callback) => {
    isReadOnly = true;
    let result = null;
    try {
        result = callback();
        isReadOnly = false;
        return result;
    }
    catch (e) {
        isReadOnly = false;
        if (e instanceof ReadOnlyTransactionError) {
            xtremConsole.error(e);
            return result;
        }
        throw e;
    }
    finally {
        isReadOnly = false;
    }
};
//# sourceMappingURL=transactions-service.js.map