import { flat } from '@sage/xtrem-shared';
import axios from 'axios';
import { jsonToGraphQLQuery } from 'json-to-graphql-query';
import { camelCase, escapeRegExp, isString, set } from 'lodash';
import { NestedGridControlObject, PodCollectionControlObject, TableControlObject, VitalPodControlObject, } from '../component/control-objects';
import * as xtremRedux from '../redux';
import { xtremConsole } from '../utils/console';
import { isDevMode } from '../utils/window';
import { CollectionValue } from './collection-data-service';
import { PromiseTracker } from './promise-tracker';
import { PLUGIN_CACHE } from '../utils/constants';
import { purgeMetaDataCache } from './metadata-service';
import { cacheArtifact, getArtifactCacheDatabase, getArtifactCachedEntry } from '../utils/artifact-cache-utils';
export const queryToGraphQuery = (query) => jsonToGraphQLQuery(Object.keys(query)[0] !== 'mutation' ? { query } : query, { pretty: isDevMode() });
export function escapeStringRegexp(input) {
    return escapeRegExp(input);
}
/**
 * The function that executes all API calls. It adds request headers based on the application context.
 * If `cacheSettings` provided and the request is successful, the response is put into the cache.
 */
const MAX_ATTEMPTS = 3;
const waitForMs = (ms) => new Promise(resolve => setTimeout(resolve, ms));
const axiosPostWithRetry = (url, query, config) => {
    // eslint-disable-next-line no-async-promise-executor
    return new Promise(async (resolve, reject) => {
        for (let i = 0; i < MAX_ATTEMPTS; i += 1) {
            try {
                const result = await axios.post(url, query, config);
                resolve(result);
                break;
            }
            catch (err) {
                const error = err;
                // If the response is a 503, and we have a retry timeout and it is not the last attempt, we try again
                if (error.response &&
                    error.response.status === 503 &&
                    error.response.headers['retry-after'] &&
                    i < MAX_ATTEMPTS - 1) {
                    const ms = parseInt(error.response.headers['retry-after'], 10);
                    await waitForMs(ms);
                }
                else {
                    // otherwise we fail
                    reject(error);
                    break;
                }
            }
        }
    });
};
export const untrackedExecuteGraphqlQuery = async ({ query, endpoint = '/api', cacheSettings, axiosCancelToken, forceRefetch, }) => {
    const stringifiedQuery = isString(query) ? query.toString() : queryToGraphQuery(query);
    const shouldUseCache = window.location.protocol === 'https:' && !isDevMode() && !!cacheSettings;
    const applicationContext = xtremRedux.getStore().getState().applicationContext;
    let db = null;
    const passphrase = applicationContext?.cacheEncryptionKey;
    if (shouldUseCache) {
        // Open database connection to IndexedDB for caching operations
        db = await getArtifactCacheDatabase();
    }
    if (isDevMode()) {
        xtremConsole.log(stringifiedQuery);
    }
    // If the cache is enabled, we search the artifact object store instead of sending a request to the server.
    if (db && shouldUseCache && !forceRefetch) {
        const value = await getArtifactCachedEntry({ db, cacheSettings, passphrase });
        if (value) {
            return {
                ...value,
                key: value.key,
                version: value.version,
                locale: value.locale,
                shouldFetchPlatformLiterals: value.shouldFetchPlatformLiterals,
                cachedAt: value.cachedAt,
            };
        }
    }
    // Configure the request
    const headers = getHeadersFromApplicationContext(applicationContext);
    const url = applicationContext && applicationContext.path ? applicationContext.path + endpoint : endpoint;
    try {
        const response = await axiosPostWithRetry(url, { query: stringifiedQuery }, { headers, ...(axiosCancelToken ? { cancelToken: axiosCancelToken } : {}) });
        // Check if cache should be used and the query was executed successfully
        if (db &&
            shouldUseCache &&
            cacheSettings &&
            response.status === 200 &&
            !response.data.errors &&
            response.data.data) {
            await cacheArtifact({ db, cacheSettings, data: response.data, passphrase });
        }
        if (isDevMode()) {
            xtremConsole.log(JSON.stringify(response.data, null, 4));
        }
        return response.data;
    }
    catch (err) {
        if (applicationContext?.onApiRequestError) {
            applicationContext?.onApiRequestError(err.response || err);
        }
        throw err;
    }
};
export const executeGraphqlQuery = async ({ query, endpoint = '/api', cacheSettings, axiosCancelToken, forceRefetch, }) => {
    return PromiseTracker.withTracker(async () => untrackedExecuteGraphqlQuery({ query, endpoint, cacheSettings, axiosCancelToken, forceRefetch }));
};
export const covertPageToStringKey = (path) => {
    const components = path.split('/');
    return `${components[0]}/${components[1]}`;
};
export const getHeadersFromApplicationContext = (applicationContext) => {
    let headers = {
        'Accept-Language': applicationContext?.locale || 'en-US',
    };
    if (applicationContext?.requestHeaders) {
        headers = { ...headers, ...applicationContext.requestHeaders };
    }
    return headers;
};
// TODO: this will fail if we have properties called 'edges' in our model
export const removeEdges = (target, keepPageInfo = false, keepCursor = false) => {
    if (!target || typeof target !== 'object') {
        return target;
    }
    if (target.query && target.query.edges && Array.isArray(target.query.edges)) {
        const data = target.query.edges.map((e) => {
            const nextTarget = keepCursor ? { ...e.node, __cursor: e.cursor } : e.node;
            return removeEdges(nextTarget, keepPageInfo, keepCursor);
        });
        return keepPageInfo
            ? { data, pageInfo: target.query.pageInfo || {}, totalCount: target.query.totalCount }
            : data;
    }
    return Object.keys(target).reduce((r, k) => {
        r[k] = Array.isArray(target[k]) ? target[k] : removeEdges(target[k], keepPageInfo, keepCursor);
        return r;
    }, {});
};
// TODO: this will fail if we have properties called 'edges' in our model
export const removeEdgesDeep = (target, keepPageInfo = false, keepCursor = false) => {
    if (!target || typeof target !== 'object') {
        return target;
    }
    if (target.edges && Array.isArray(target.edges)) {
        const data = target.edges.map((e) => {
            const nextTarget = keepCursor ? { ...e.node, __cursor: e.cursor } : e.node;
            return removeEdgesDeep(nextTarget, keepPageInfo, keepCursor);
        });
        return keepPageInfo ? { data, pageInfo: target.pageInfo || {}, totalCount: target.totalCount } : data;
    }
    return Object.keys(target).reduce((r, k) => {
        r[k] = Array.isArray(target[k]) ? target[k] : removeEdgesDeep(target[k], keepPageInfo, keepCursor);
        return r;
    }, {});
};
/**
 * Gets all groups along with their aggregations.
 *
 * Returns all groups along with their group key.
 *
 * @export
 * @param {*} target
 * @param {string} [k]
 * @returns {*}
 */
export function getGroups(target, groupKey) {
    return getGroupsInternal(groupKey)(target);
}
const getGroupsInternal = (groupKey) => (target, k) => {
    if (target && typeof target === 'object') {
        if (target.queryAggregate) {
            return {
                groups: target.queryAggregate.edges.map(queryAggregateMapper(groupKey)),
                key: k,
            };
        }
        if (!Array.isArray(target)) {
            // eslint-disable-next-line no-restricted-syntax
            for (const key of Object.keys(target)) {
                const result = getGroupsInternal(groupKey)(target[key], key);
                if (result) {
                    return result;
                }
            }
        }
        else {
            // eslint-disable-next-line no-restricted-syntax
            for (const element of target) {
                const result = getGroupsInternal(groupKey)(element);
                if (result) {
                    return result;
                }
            }
        }
    }
    return undefined;
};
const queryAggregateMapper = (key) => (e) => {
    const { group, values } = e.node;
    const flatGroup = flat(group);
    const [groupByIdKey, firstKey] = Object.keys(flatGroup).sort();
    const groupKey = key ?? firstKey ?? groupByIdKey;
    const groupValue = flatGroup[groupKey];
    const aggregations = Object.entries(flat(values)).reduce((acc, [k, value]) => {
        const newKey = k.split('.').slice(0, -1).join('.');
        set(acc, newKey, value);
        return acc;
    }, {});
    const node = {
        ...aggregations,
        ...group,
        // always assign an _id
        _id: `__group-${groupValue}`,
        __isGroup: true,
        __groupKey: groupKey ?? groupByIdKey,
        __groupCount: values?._id?.distinctCount,
    };
    return {
        node,
        cursor: e.cursor,
    };
};
export function getAllGroups(input) {
    const result = {};
    function getAllGroupsInternal(target, k) {
        if (target && typeof target === 'object') {
            if (target.queryAggregate && k) {
                result[k] = {
                    groups: target.queryAggregate.edges.map(queryAggregateMapper()),
                    key: k,
                };
            }
            if (!Array.isArray(target)) {
                // eslint-disable-next-line no-restricted-syntax
                for (const key of Object.keys(target)) {
                    getAllGroupsInternal(target[key], key);
                }
            }
            else {
                // eslint-disable-next-line no-restricted-syntax
                for (const element of target) {
                    getAllGroupsInternal(element);
                }
            }
        }
        return undefined;
    }
    getAllGroupsInternal(input);
    return result;
}
export const flattenEdges = (edges) => edges.map((e) => (e.node ? e.node : e));
const buildNestedQuery = (queryObject, nameParts, alias) => {
    const [firstPart, ...missingParts] = nameParts;
    const nestedQueryContent = missingParts.length > 0 ? buildNestedQuery(queryObject, missingParts) : queryObject;
    const nestedQuery = { [alias || firstPart]: nestedQueryContent };
    if (alias) {
        nestedQueryContent.__aliasFor = firstPart;
    }
    return nestedQuery;
};
export const splitNodeName = (nodeName) => String(nodeName)
    .replace(/^@\w*\//, '')
    .split('/')
    .map(camelCase);
export const wrapQuery = (nodeName, queryObject, alias) => {
    const nameParts = splitNodeName(nodeName);
    return buildNestedQuery(queryObject, nameParts, alias);
};
export const unwrapQueryResponse = (nodeName, queryResponse, alias) => {
    const nameParts = splitNodeName(nodeName);
    const queryKeys = alias ? [alias, ...nameParts.slice(1)] : nameParts;
    return queryKeys.reduce((reduced, key) => reduced[key], queryResponse);
};
export const getNodeForLookupOperation = ({ parentElementId, screenDefinition, level, }) => {
    const screenId = screenDefinition.metadata.screenId;
    if (parentElementId) {
        const value = screenDefinition.values[parentElementId];
        const parentElement = screenDefinition.metadata.controlObjects[parentElementId];
        if (parentElement instanceof VitalPodControlObject ||
            parentElement instanceof TableControlObject ||
            parentElement instanceof PodCollectionControlObject) {
            return parentElement.node;
        }
        if (parentElement instanceof NestedGridControlObject && value instanceof CollectionValue) {
            const targetLevel = level || 0;
            return value.nodes[targetLevel];
        }
    }
    return screenDefinition.metadata.uiComponentProperties[screenId]
        .node;
};
export const wipeCache = async () => {
    window.localStorage.clear();
    await purgeMetaDataCache();
    await window.caches.delete(PLUGIN_CACHE);
};
//# sourceMappingURL=graphql-utils.js.map