import { get, isEmpty, set } from 'lodash';
import { navigationPanelId } from '../component/container/navigation-panel/navigation-panel-types';
import { getReferenceOrderBy } from '../component/field/reference/reference-utils';
import { FieldKey } from '../component/types';
import * as xtremRedux from '../redux';
import { GraphQLKind } from '../types';
import { filterFields, getNestedFieldElementId } from '../utils/abstract-fields-utils';
import { NEW_PAGE, QUERY_ALIAS_NAV_PANEL, QUERY_ALIAS_PAGE_DATA } from '../utils/constants';
import { convertDeepBindToPathNotNull, getNestedFieldsFromProperties } from '../utils/nested-field-utils';
import { findDeepPropertyDetails } from '../utils/node-utils';
import { resolveByValue } from '../utils/resolve-value-utils';
import { CollectionValue } from './collection-data-service';
import { CollectionFieldTypes } from './collection-data-types';
import { getGroupKey } from './collection-data-utils';
import { buildDefaultsQuery, buildFieldQuery, buildNestedAggregationQuery, buildNestedDefaultsQuery, buildReferenceFieldQuery, buildRootNodeQuery, buildRowDefaultQuery, buildSearchBoxFilter, buildTableQuery, buildTableTotalCountQuery, mergeFilters, nestedFieldsToQueryProperties, queryBuilder, } from './graphql-query-builder';
import { executeGraphqlQuery, getAllGroups, getGroups, removeEdges, splitNodeName, unwrapQueryResponse, wrapQuery, } from './graphql-utils';
import { localize } from './i18n-service';
import { createNavigationTableProperties, getInitialOptionMenuItem } from './navigation-panel-service';
import { showToast } from './toast-service';
import { formatScreenValues } from './value-formatter-service';
export const fetchCollectionDataCount = async ({ rootNode, filter, }) => {
    const query = buildTableTotalCountQuery({
        filter,
    });
    const wrappedQuery = wrapQuery(rootNode, query);
    const response = await executeGraphqlQuery({ query: wrappedQuery });
    const unwrappedResponse = unwrapQueryResponse(rootNode, response.data);
    return unwrappedResponse.query.totalCount;
};
export const fetchCollectionData = async ({ bind, elementId, group, nestedFields, node, nodeTypes, queryArguments, rootNode, rootNodeId, screenDefinition, totalCount = false, treeProperties, }) => {
    const query = buildTableQuery({
        rootNodeId,
        elementId,
        columns: nestedFields,
        screenDefinition,
        queryArguments,
        bind: treeProperties?.sublevelProperty ? convertDeepBindToPathNotNull(treeProperties?.sublevelProperty) : bind,
        group,
        treeProperties,
        totalCount,
        node,
        nodeTypes,
    });
    const nodeToUse = treeProperties?.sublevelProperty && node ? node : rootNode;
    const wrappedQuery = wrapQuery(nodeToUse, query);
    const response = await executeGraphqlQuery({ query: wrappedQuery });
    const groups = getGroups(response.data, getGroupKey(group?.key));
    const unwrappedResponse = unwrapQueryResponse(nodeToUse, response.data);
    const nodeData = removeEdges(unwrappedResponse, true, true);
    if (!nodeData || !nodeData.data || !nodeData.data[0] || !nodeData.data[0][elementId]) {
        return {};
    }
    const { data = [], pageInfo = {} } = get(nodeData.data[0], treeProperties?.sublevelProperty
        ? [elementId, ...convertDeepBindToPathNotNull(treeProperties?.sublevelProperty).split('.').slice(1)]
        : [
            elementId,
            ...convertDeepBindToPathNotNull(bind || elementId)
                .split('.')
                .slice(1),
        ]);
    if (groups) {
        // Groups are appended to the data array with an '__isGroup' flag
        data.push(...groups.groups.map((g) => ({ ...g.node, __cursor: g.cursor })));
    }
    return { data, pageInfo, totalCount: nodeData.totalCount };
};
export const fetchCollectionRecord = async (screenDefinition, nestedFields, nodeName, recordId) => {
    const properties = nestedFieldsToQueryProperties({ screenDefinition, nestedFields });
    const query = queryBuilder(nodeName, {
        properties,
        queryArguments: { filter: JSON.stringify({ _id: recordId }) },
        noPageInfo: true,
    });
    const wrappedQuery = wrapQuery(nodeName, query);
    const response = await executeGraphqlQuery({ query: wrappedQuery });
    const unwrappedResponse = unwrapQueryResponse(nodeName, response.data);
    return unwrappedResponse?.query?.edges?.[0]?.node || null;
};
export const formatNavigationPanelResponse = (navigationPanelResponse) => {
    if (!navigationPanelResponse.query || !navigationPanelResponse.query.edges) {
        showToast(localize('@sage/xtrem-ui/navigation-panel-failed', 'An error ocurred loading the navigation panel items'), { type: 'error' });
    }
    return removeEdges(navigationPanelResponse, true, true).data;
};
export const fetchNavigationPanelData = async (screenId, screenDefinition, tableDecoratorProperties, orderBy, filter, first = 30, nodeTypes, 
/**
 * If it is an initial page load query for the navigation panel, the option menu is added to the query filter by this method.
 * In other cases (e.g. filter change, sorting change), the collection data service provided filter already contains the currently selected
 * option item.
 *  */
optionMenuItem) => {
    const { query } = buildReferenceFieldQuery({
        fieldProperties: tableDecoratorProperties,
        elementId: navigationPanelId,
        screenDefinition,
        screenId,
        orderBy,
        filter,
        valueField: navigationPanelId,
        pageSize: first,
        nodeTypes,
        optionMenuItem,
    });
    const response = await executeGraphqlQuery({ query });
    const navigationPanelResponse = unwrapQueryResponse(tableDecoratorProperties.node, response.data);
    return formatNavigationPanelResponse(navigationPanelResponse);
};
export const fetchReferenceFieldSuggestions = async ({ after, contextNode, exactMatch, fieldId, fieldProperties, filterValue, isFilterLimitedToDataset = false, level, locale, parentElementId, recordContext, screenId, }) => {
    const allFilters = [fieldProperties.filter];
    const state = xtremRedux.getStore().getState();
    const nodeTypes = state.nodeTypes;
    let isArray = false;
    const bind = fieldProperties.bind;
    const valueField = typeof fieldProperties.valueField === 'object'
        ? Object.keys(fieldProperties.valueField)[0]
        : fieldProperties.valueField;
    if (typeof fieldProperties.valueField === 'object') {
        const type = findDeepPropertyDetails(fieldProperties.node, fieldProperties.valueField, nodeTypes);
        isArray = type?.kind === GraphQLKind.List;
    }
    const searchBoxGraphQLFilter = buildSearchBoxFilter(fieldProperties, nodeTypes, locale, CollectionFieldTypes.LOOKUP_DIALOG, filterValue, exactMatch, isArray);
    if (searchBoxGraphQLFilter) {
        allFilters.push(searchBoxGraphQLFilter);
    }
    const screen = state.screenDefinitions[screenId];
    const filter = mergeFilters(allFilters, screen, recordContext);
    const orderBy = getReferenceOrderBy(fieldProperties);
    let filterLimitedToDataset;
    let orderByLimitedToDataSet;
    if (isFilterLimitedToDataset && filter) {
        filterLimitedToDataset = { [String(fieldProperties.bind)]: filter };
    }
    if (isFilterLimitedToDataset && orderBy) {
        orderByLimitedToDataSet = { [String(fieldProperties.bind)]: orderBy };
    }
    const queryResult = await fetchReferenceFieldData({
        after,
        contextNode,
        elementId: fieldId,
        fieldProperties,
        filter: isFilterLimitedToDataset ? filterLimitedToDataset : filter,
        isFilterLimitedToDataset,
        level,
        orderBy: isFilterLimitedToDataset ? orderByLimitedToDataSet : orderBy,
        parentElementId,
        recordContext,
        screenId,
        valueField: fieldProperties.valueField,
    });
    let initialValues = removeEdges(queryResult, true, true)?.data ?? [];
    if (isFilterLimitedToDataset) {
        initialValues = (removeEdges(queryResult, true, true)?.data ?? []).map((node) => ({
            _id: isFilterLimitedToDataset ? node[String(bind)]._id : node._id,
            [String(valueField)]: isFilterLimitedToDataset ? node[String(bind)][valueField] : node[valueField],
        }));
    }
    const collectionValue = new CollectionValue({
        bind: fieldProperties.bind,
        screenId,
        elementId: fieldId,
        nodeTypes,
        isTransient: !!fieldProperties.isTransient,
        hasNextPage: !!queryResult.query?.pageInfo?.hasNextPage,
        orderBy: [orderBy],
        columnDefinitions: [fieldProperties.columns || []],
        nodes: [String(fieldProperties.node)],
        filter: [filter],
        initialValues,
        fieldType: CollectionFieldTypes.LOOKUP_DIALOG,
        parentElementId,
        recordContext,
        dbKey: 'lookup-temp',
    });
    return collectionValue.getData({
        cleanMetadata: false,
        inGroup: isFilterLimitedToDataset,
        temporaryRecords: resolveByValue({
            skipHexFormat: true,
            screenId,
            propertyValue: fieldProperties.additionalLookupRecords,
            rowValue: null,
            fieldValue: null,
        }),
    });
};
export const fetchReferenceFieldData = async ({ after, axiosCancelToken, contextNode, elementId, fieldProperties, filter, group, isFilterLimitedToDataset, level, orderBy, pageSize, parentElementId, recordContext, screenId, valueField, }) => {
    const state = xtremRedux.getStore().getState();
    // We check if we are in a nested collection context
    if (parentElementId !== '$navigationPanel' &&
        contextNode &&
        state.screenDefinitions[screenId].values[String(parentElementId)] &&
        isFilterLimitedToDataset) {
        const value = state.screenDefinitions[screenId].values[String(parentElementId)];
        if (value instanceof CollectionValue) {
            const { parentNode, parentId, childLevel, parentBind } = value.getParentNodeAndParentIdFromChildNode({
                node: String(contextNode),
            });
            /*
            If we have the filter limited to the dataset we need to group and do a aggregation query.
            To be sure, we also we check if we have all the ids and nodes needed and if the level it's correct in the collection value
        */
            if (parentNode && parentId && childLevel > 0) {
                const referenceProperties = fieldProperties;
                const parsedValueField = typeof referenceProperties.valueField === 'object'
                    ? Object.keys(referenceProperties.valueField)[0]
                    : referenceProperties.valueField;
                const groupBy = { key: `${referenceProperties.bind}.${parsedValueField}` };
                const aggregationQuery = buildNestedAggregationQuery({
                    elementId,
                    group: groupBy,
                    rootNodeId: parentId,
                    bind: parentBind,
                });
                const res = await executeGraphqlQuery({
                    query: wrapQuery(parentNode, aggregationQuery),
                    axiosCancelToken,
                });
                const groups = getGroups(res.data, getGroupKey(group?.key));
                const edgesPath = `${splitNodeName(parentNode).join('.')}.query.edges`;
                // Groups are appended to the data array with an '__isGroup' flag
                set(res.data, edgesPath, groups.groups);
                return unwrapQueryResponse(parentNode, res.data);
            }
        }
    }
    const { query, bind, node, fieldNode } = buildReferenceFieldQuery({
        after,
        contextNode,
        elementId,
        fieldProperties,
        filter,
        group,
        isFilterLimitedToDataset,
        level,
        nodeTypes: state.nodeTypes,
        orderBy,
        pageSize,
        parentElementId,
        recordContext,
        screenDefinition: state.screenDefinitions[screenId],
        screenId,
        valueField,
    });
    const response = await executeGraphqlQuery({ query, axiosCancelToken });
    let data;
    if (!bind) {
        // Not using lookup API
        data = response?.data;
        const groups = getGroups(response.data, getGroupKey(group?.key));
        if (groups) {
            const edgesPath = `${splitNodeName(fieldNode).join('.')}.query.edges`;
            // Groups are appended to the data array with an '__isGroup' flag
            set(data, edgesPath, [...groups.groups, ...get(data, edgesPath, [])]);
        }
    }
    else {
        /**
         * Using lookup API => build object to be unwrapped
         *
         * {
                "xtremShowCase": {
                    "showCaseProduct": {
                    "lookups": {
                        "provider": {
                        "edges": [
                            {
                            "node": {
                                "textField": "Ali Express",
                                "_id": "1"
                            },
                            "cursor": "[1]#78"
                            },
                            {
                            "node": {
                                "textField": "Amazon",
                                "_id": "2"
                            },
                            "cursor": "[2]#37"
                            }
                        ],
                        "pageInfo": {
                            "startCursor": "[1]#78",
                            "endCursor": "[2]#37",
                            "hasPreviousPage": false,
                            "hasNextPage": false
                        }
                        }
                    }
                    }
                }
            }
            becomes
            {
                "xtremShowCase": {
                    "showCaseProvider": {
                    "query": {
                        "edges": [
                        {
                            "node": {
                            "textField": "Ali Express",
                            "_id": "1"
                            },
                            "cursor": "[1]#78"
                        },
                        {
                            "node": {
                            "textField": "Amazon",
                            "_id": "2"
                            },
                            "cursor": "[2]#37"
                        }
                        ],
                        "pageInfo": {
                        "startCursor": "[1]#78",
                        "endCursor": "[2]#37",
                        "hasPreviousPage": false,
                        "hasNextPage": false
                        }
                    }
                    }
                }
            }
         */
        const lookupKeyAccessor = splitNodeName(node).join('.');
        const edges = get(response.data, `${lookupKeyAccessor}.lookups.${bind}`);
        const setKey = splitNodeName(fieldNode).join('.');
        data = set({}, `${setKey}.query`, edges);
    }
    return unwrapQueryResponse(fieldNode, data);
};
export const fetchReferenceFieldDataById = async ({ _id, contextNode, elementId, fieldProperties, level, parentElementId, recordContext, screenId, valueField, }) => {
    const result = await fetchReferenceFieldData({
        filter: { _id },
        contextNode,
        elementId,
        fieldProperties,
        level,
        parentElementId,
        recordContext,
        screenId,
        valueField,
    });
    return result.query.edges[0]?.node;
};
export const fetchRecordData = async (screenId, recordId, screenDefinition) => {
    const controlObjects = screenDefinition.metadata.controlObjects;
    const uiComponentProperties = screenDefinition.metadata.uiComponentProperties;
    const pageProperties = uiComponentProperties[screenId];
    const nodeName = pageProperties.node;
    const rootNodeQuery = buildRootNodeQuery(nodeName, filterFields(controlObjects), screenDefinition, recordId);
    const response = await executeGraphqlQuery({ query: rootNodeQuery });
    const unwrapped = removeEdges(unwrapQueryResponse(nodeName, response.data), true, true);
    const result = unwrapped.read;
    const groups = getGroups(response.data);
    if (groups) {
        if (!result[groups.key].data) {
            result[groups.key].data = [];
        }
        // Groups are appended to the data array with an '__isGroup' flag
        result[groups.key].data.push(...groups.groups.map((g) => ({ ...g.node, __cursor: g.cursor })));
    }
    return result;
};
export const fetchField = async (screenDefinition, fieldProperties, recordId, elementId, plugins, nodeTypes, nodeName, keepPageInfo = false) => {
    const pageMetadata = screenDefinition.metadata;
    const pageProperties = pageMetadata.uiComponentProperties[pageMetadata.screenId];
    const nodeQuery = buildFieldQuery(pageProperties.node, pageMetadata.controlObjects[elementId], screenDefinition, fieldProperties, elementId, recordId, nodeTypes);
    const query = wrapQuery(pageProperties.node, nodeQuery);
    const response = await executeGraphqlQuery({ query });
    const unwrapped = unwrapQueryResponse(pageProperties.node, response.data);
    const noEdges = removeEdges(unwrapped, keepPageInfo, true);
    const nodeData = keepPageInfo ? (noEdges.data && noEdges.data[0]) || {} : noEdges[0] || {};
    const formattedValues = formatScreenValues({
        screenId: pageMetadata.screenId,
        controlObjects: pageMetadata.controlObjects,
        plugins,
        nodeTypes,
        values: nodeData,
        parentNode: pageProperties.node,
        onlyElementIds: [elementId],
    });
    return formattedValues[nodeName || elementId];
};
export const fetchInitialData = async (pageDefinition, recordId, path, applicationContext, nodeTypes, dataTypes, 
/**
 * If the values are provided, they are not fetched from the server.
 * This feature is used to pre-populate page dialogs with data.
 *  */
values, isDuplicate = false) => {
    const pageMetadata = pageDefinition.metadata;
    const { screenId } = pageMetadata;
    const pageProperties = pageMetadata.uiComponentProperties[screenId];
    const rootNode = pageProperties.node;
    let query = {};
    // If no root node is provided, the page is handled as it would be transient because it can't fetch data without a target node.
    const isPageTransient = pageProperties.isTransient || !rootNode || values;
    if (!isPageTransient && recordId) {
        // If the page is not transient and there is a record ID, query the record data
        const nodeQuery = buildRootNodeQuery(rootNode, filterFields(pageMetadata.controlObjects), pageDefinition, recordId, QUERY_ALIAS_PAGE_DATA, nodeTypes, isDuplicate);
        query = {
            ...query,
            ...nodeQuery,
        };
    }
    else if (!isPageTransient) {
        // If no record ID provided, but the page is not transient, query the default values for the new record
        query = {
            ...query,
            ...buildDefaultsQuery({
                screenDefinition: pageDefinition,
                alias: QUERY_ALIAS_PAGE_DATA,
                node: rootNode,
                nodeTypes,
            }),
        };
    }
    if (pageProperties.navigationPanel && rootNode && nodeTypes && dataTypes && !isDuplicate) {
        const navigationTableProperties = await createNavigationTableProperties(screenId, rootNode, pageDefinition, nodeTypes, dataTypes, path, applicationContext);
        const optionMenuItem = getInitialOptionMenuItem(pageDefinition, navigationTableProperties);
        /**
         * The navigation panel query is identical to a transient reference field suggestions lookup query, so we reuse
         * the same function. In both cases we query the root list of the node type with a given criteria.
         *  */
        const { query: navPanelQuery } = buildReferenceFieldQuery({
            fieldProperties: navigationTableProperties,
            elementId: navigationPanelId,
            nodeTypes,
            screenDefinition: pageDefinition,
            screenId,
            orderBy: navigationTableProperties.orderBy,
            filter: navigationTableProperties.filter,
            valueField: navigationPanelId,
            queryAlias: QUERY_ALIAS_NAV_PANEL,
            optionMenuItem,
        });
        query = {
            ...query,
            ...navPanelQuery,
        };
    }
    const initialData = {
        values: values || {},
        navigationPanel: [],
    };
    if (isEmpty(query)) {
        return initialData;
    }
    const response = await executeGraphqlQuery({ query });
    if (rootNode && response.data) {
        initialData.values = {};
        const unwrappedRootNode = response.data[QUERY_ALIAS_PAGE_DATA]
            ? unwrapQueryResponse(rootNode, response.data, QUERY_ALIAS_PAGE_DATA)
            : false;
        if (unwrappedRootNode?.getDefaults) {
            // Handling default values
            initialData.values = unwrappedRootNode.getDefaults;
        }
        else if (unwrappedRootNode) {
            // Handling real record data
            initialData.values = removeEdges(unwrappedRootNode[isDuplicate ? 'getDuplicate' : 'read'] || {}, true, true);
        }
        else if (values) {
            initialData.values = values;
        }
        const groups = getAllGroups(response.data.rootNode);
        Object.keys(groups).forEach(fieldKey => {
            const group = groups[fieldKey];
            if (!initialData.values[group.key].data) {
                initialData.values[group.key].data = [];
            }
            // Groups are appended to the data array with an '__isGroup' flag
            initialData.values[group.key].data.push(...group.groups.map((g) => ({ ...g.node, __cursor: g.cursor })));
        });
        if (recordId && response.data[QUERY_ALIAS_PAGE_DATA] && Object.keys(initialData.values).length === 0) {
            showToast(`No record found with id ${recordId}`, { type: 'error' });
        }
    }
    if (response.data[QUERY_ALIAS_NAV_PANEL] && rootNode) {
        const navigationPanelResponse = unwrapQueryResponse(rootNode, response.data, QUERY_ALIAS_NAV_PANEL);
        initialData.navigationPanel = formatNavigationPanelResponse(navigationPanelResponse);
    }
    return initialData;
};
export const fetchNestedDefaultValues = async ({ screenId, elementId, isNewRow = true, recordId, nestedElementId, level = 0, data, }) => {
    const pageDefaults = 'pageDefaults';
    const nestedDefaults = 'nestedDefaults';
    const state = xtremRedux.getStore().getState();
    const screenDefinition = state.screenDefinitions[screenId];
    const screenProperties = screenDefinition.metadata.uiComponentProperties[screenId];
    const fieldProperties = screenDefinition.metadata.uiComponentProperties[elementId];
    const isNestedGrid = Object.prototype.hasOwnProperty.call(fieldProperties, 'levels');
    const nestedField = isNestedGrid
        ? fieldProperties.levels[level].columns.find(column => getNestedFieldElementId(column) === nestedElementId)
        : getNestedFieldsFromProperties(fieldProperties).find(column => getNestedFieldElementId(column) === nestedElementId);
    const nestedFieldProperties = nestedField?.properties;
    const shouldFetchNestedDefaults = (isNewRow && fieldProperties.node && !fieldProperties.isTransient) ||
        (nestedFieldProperties?.fetchesDefaults && fieldProperties.node && !fieldProperties.isTransient);
    const result = {};
    const queryParameters = screenDefinition.queryParameters;
    const shouldFetchPageDefaults = !isNewRow &&
        fieldProperties.fetchesDefaults &&
        screenProperties.node &&
        !screenProperties.isTransient &&
        screenDefinition.type === 'page' &&
        (!queryParameters?._id || queryParameters?._id === NEW_PAGE);
    if (shouldFetchPageDefaults) {
        const pageQueryResult = await executeGraphqlQuery({
            query: buildDefaultsQuery({ screenDefinition, alias: pageDefaults }),
        });
        result[pageDefaults] = unwrapQueryResponse(String(screenProperties.node), pageQueryResult.data, pageDefaults).getDefaults;
    }
    if (shouldFetchNestedDefaults &&
        (isNestedGrid || fieldProperties._controlObjectType !== FieldKey.Table)) {
        const nestedQueryResult = await executeGraphqlQuery({
            query: buildNestedDefaultsQuery({
                screenDefinition,
                elementId,
                isNewRow,
                alias: nestedDefaults,
                recordId,
                level,
                data,
                nodeTypes: state.nodeTypes,
            }),
        });
        const rowNode = isNestedGrid
            ? String(fieldProperties.levels[level].node)
            : String(fieldProperties.node);
        result[nestedDefaults] = unwrapQueryResponse(rowNode, nestedQueryResult.data, nestedDefaults).getDefaults;
    }
    else if (shouldFetchNestedDefaults) {
        const nestedQueryResult = await executeGraphqlQuery({
            query: buildRowDefaultQuery({
                screenDefinition,
                elementId,
                isNewRow,
                alias: nestedDefaults,
                recordId,
                data,
                nodeTypes: state.nodeTypes,
            }),
        });
        const node = String(screenProperties.node);
        result[nestedDefaults] = removeEdges(unwrapQueryResponse(node, nestedQueryResult.data, nestedDefaults).getDefaults)[elementId][0];
    }
    return result;
};
export const fetchDefaultValues = async (screenDefinition, rootNode, requestedFieldIds, clean, nodeTypes) => {
    const query = buildDefaultsQuery({ screenDefinition, requestedFieldIds, clean, node: rootNode, nodeTypes });
    if (isEmpty(query)) {
        return {};
    }
    const result = await executeGraphqlQuery({ query });
    const unwrapped = unwrapQueryResponse(rootNode, result.data);
    return unwrapped.getDefaults || {};
};
export const fetchWorkflowNodes = async () => {
    const query = {
        workflow: {
            getAvailableWorkflowNodes: {
                key: true,
                title: true,
                description: true,
                color: true,
                icon: true,
                configurationPage: true,
                shape: true,
                type: true,
            },
        },
    };
    const result = await executeGraphqlQuery({ query, endpoint: '/metadata' });
    return result.data.workflow.getAvailableWorkflowNodes || [];
};
//# sourceMappingURL=graphql-service.js.map