import { date as xtremDate } from '@sage/xtrem-date-time';
import { geFilterQuery } from '@sage/xtrem-filter-utils';
import { isEmpty, isEqual, set, uniqWith } from 'lodash';
import { getFieldTitle } from '../component/field/carbon-helpers';
import { getReferencePath } from '../component/field/reference/reference-utils';
import { FieldKey, RANGE_DIVIDER } from '../component/types';
import { findColumnDefinitionByBind, getPropertyScalarType, normalizeUnderscoreBind, } from '../utils/abstract-fields-utils';
import { convertDeepBindToPathNotNull } from '../utils/nested-field-utils';
import { findDeepPropertyType } from '../utils/node-utils';
import { schemaTypeNameFromNodeName } from '../utils/transformers';
const agGridFilterMidnightDateToStringDate = (value) => {
    try {
        const asDate = xtremDate.parse(value, 'base', 'YYYY-MM-DD 00:00:00');
        return asDate.format('YYYY-MM-DD');
    }
    catch {
        return value;
    }
};
const generate = (filter, column) => {
    const columnType = column?.type || '';
    const isDate = columnType === FieldKey.Date;
    const filterType = filter.value[0].filterType;
    const filterValue = isDate
        ? filter.value[0].filterValue
            .toString()
            .split(RANGE_DIVIDER)
            .map(agGridFilterMidnightDateToStringDate)
            .join(RANGE_DIVIDER)
        : filter.value[0].filterValue;
    if (filterValue !== undefined && filterValue !== null) {
        const filterQuery = geFilterQuery({ filterType: filterType.value, filterValue });
        const isReferenceObject = columnType === FieldKey.Reference;
        if (isReferenceObject) {
            const valueFieldKey = filter.value[0]?.filterById
                ? '_id'
                : getReferencePath(column.properties.valueField);
            return set({}, valueFieldKey, filterQuery);
        }
        return filterQuery;
    }
    return {};
};
const transformFilterToGraphQLFilter = (filter, column) => {
    const bind = normalizeUnderscoreBind(filter.id);
    const queryFilter = {};
    const { filterValue } = filter.value[0];
    if (filterValue !== undefined && filterValue !== null) {
        if (filter.value.length > 1) {
            // Combined Filter (_or)
            queryFilter._or = filter.value.map(value => set({}, bind, generate({ id: bind, value: [value] }, column)));
        }
        else {
            // Single Filter
            set(queryFilter, bind, generate(filter, column));
        }
    }
    return queryFilter;
};
export const getGraphQLFilter = (filters, nestedFields) => {
    let queryFilter = {};
    if (filters.length === 1) {
        const rtFilter = filters[0];
        queryFilter = transformFilterToGraphQLFilter(rtFilter, findColumnDefinitionByBind(nestedFields, rtFilter.id));
    }
    else if (filters.length > 1) {
        queryFilter._and = filters.map(rtFilter => transformFilterToGraphQLFilter(rtFilter, findColumnDefinitionByBind(nestedFields, rtFilter.id)));
    }
    return queryFilter;
};
export const getGraphQLFilterAsString = (queryFilter) => {
    return isEmpty(queryFilter) ? undefined : JSON.stringify(queryFilter);
};
export const getFilterAsGraphQLFilterString = (filters, nestedFields) => {
    const queryFilter = getGraphQLFilter(filters, nestedFields);
    return getGraphQLFilterAsString(queryFilter);
};
export const mergeGraphQLFilters = (graphQLFilters) => {
    let filters = uniqWith(graphQLFilters, isEqual);
    filters = filters.filter(filter => !isEmpty(filter));
    return isEmpty(filters) ? undefined : filters.length === 1 ? filters[0] : { _and: filters };
};
export const getTypedNestedFields = (screenId, node, nestedFields, knownNodeTypes) => {
    return nestedFields.map(nestedField => {
        const bind = convertDeepBindToPathNotNull(nestedField.properties.bind);
        const propertyType = findDeepPropertyType(schemaTypeNameFromNodeName(node), nestedField.properties.bind, knownNodeTypes, true)
            ?.type || FieldKey.Text;
        const fieldType = nestedField.type;
        return {
            type: fieldType,
            properties: {
                ...nestedField.properties,
                bind,
                title: getFieldTitle(screenId, nestedField.properties, null),
                type: getPropertyScalarType(knownNodeTypes, propertyType, nestedField.type, nestedField.properties.valueField),
            },
        };
    });
};
//# sourceMappingURL=filter-service.js.map