import { getFieldTitle } from '../component/field/carbon-helpers';
import * as xtremRedux from '../redux';
import { xtremConsole } from '../utils/console';
import { triggerScreenEvent } from '../utils/events';
import { isDevMode } from '../utils/window';
import { localize } from './i18n-service';
import { PromiseTracker } from './promise-tracker';
let nextDialogId = 0;
/** For unit tests only */
export const resetDialogIds = () => {
    nextDialogId = 0;
};
const triggerOnCloseActions = async (dialogId, isWizardFinished) => {
    const store = xtremRedux.getStore();
    const state = store.getState();
    const activeDialog = state.activeDialogs[dialogId];
    const thunkDispatch = store.dispatch;
    if (activeDialog &&
        activeDialog.type === 'page' &&
        activeDialog.screenId &&
        state.screenDefinitions[activeDialog.screenId]) {
        const screenId = activeDialog.screenId;
        await triggerScreenEvent(screenId, 'onClose', isWizardFinished);
    }
    thunkDispatch(xtremRedux.actions.closeDialog(dialogId));
};
export class DialogControl {
    static { Symbol.toStringTag; }
    constructor({ dialogId, screenId, level, title, content, getButtons, options, isSticker = false, isDirtyCheck = false, subtitle, type, }) {
        const thunkDispatch = xtremRedux.getStore().dispatch;
        this.dialogPromise = new Promise((resolve, reject) => {
            this.rejectPromise = reject;
            this.resolvePromise = resolve;
            const dialog = {
                isSticker,
                isDirtyCheck,
                dialogId,
                buttons: getButtons(dialogId, resolve, reject),
                content,
                level,
                options: {
                    fullScreen: options?.fullScreen || false,
                    resolveOnCancel: options?.resolveOnCancel || false,
                    reverseButtons: options?.reverseButtons || false,
                    rightAligned: options?.rightAligned || false,
                    size: options?.size,
                    skipDirtyCheck: options?.skipDirtyCheck || false,
                    dialogTitle: options?.dialogTitle || undefined,
                    mdContent: options?.mdContent || false,
                    isDuplicate: options?.isDuplicate || false,
                    height: options?.height,
                },
                screenId,
                title,
                subtitle,
                dialogControl: this,
                type,
            };
            thunkDispatch(xtremRedux.actions.openDialog(dialogId, dialog));
        });
        this.id = dialogId;
        PromiseTracker.pushDialog();
    }
    cancel() {
        if (this.rejectPromise) {
            this.rejectPromise();
            triggerOnCloseActions(this.id);
        }
        else {
            // This branch is only reached if we call cancel right after creating the class instance;
            // in this case, the promise callback might not have been executed yet, so the rejectPromise
            // attribute might not be defined yet. In that case, we set an immediate to cancel the promise
            // right after the promise callback has been executed
            // eslint-disable-next-line @typescript-eslint/no-implied-eval
            setTimeout(this.rejectPromise, 0);
        }
    }
    catch(callback) {
        return this.dialogPromise.catch(callback);
    }
    then(onFulfilled, onRejected) {
        return this.dialogPromise.then(onFulfilled, onRejected);
    }
    finally(onFinally) {
        return this.dialogPromise.finally(onFinally);
    }
    resolve(value) {
        PromiseTracker.popDialog();
        if (this.resolvePromise) {
            this.resolvePromise(value);
        }
        const thunkDispatch = xtremRedux.getStore().dispatch;
        thunkDispatch(xtremRedux.actions.closeDialog(this.id));
    }
}
export class PageDialogControl extends DialogControl {
    static { Symbol.toStringTag; }
    constructor(dialogId, options) {
        super({
            dialogId,
            screenId: null,
            level: 'info',
            title: localize('@sage/xtrem-ui/dialog-loading', 'Loading...'),
            content: null,
            getButtons: () => ({}),
            options,
            isSticker: false,
            isDirtyCheck: false,
            subtitle: '',
            type: 'page',
        });
    }
    resolve(value) {
        super.resolve(value);
    }
    finally(onFinally) {
        return this.promise.finally(onFinally);
    }
    get promise() {
        return this.dialogPromise;
    }
}
export const closeDialog = async (dialogId, result, isWizardCompleted) => {
    const serializedResult = result ? JSON.parse(JSON.stringify(result)) : undefined;
    const activeDialog = xtremRedux.getStore().getState().activeDialogs[dialogId];
    if (activeDialog && activeDialog.screenId) {
        await triggerOnCloseActions(dialogId, isWizardCompleted);
        activeDialog.dialogControl.resolve(serializedResult);
    }
    else {
        throw new Error(`Cannot close dialog with invalid id: ${dialogId}`);
    }
};
export const createDialog = (screenId, level, title, content, getButtons, options, isSticker = false, type) => {
    nextDialogId += 1;
    return new DialogControl({
        dialogId: nextDialogId,
        screenId,
        level,
        title,
        content,
        getButtons,
        options,
        isSticker,
        isDirtyCheck: options?.isDirtyCheck || false,
        subtitle: undefined,
        type,
    });
};
export const createStickerDialog = (store, screenId, level, title, content, options) => {
    const dialogControl = createDialog(screenId, level, title, content, () => ({}), options, true);
    const onFinish = (result) => {
        closeDialog(dialogControl.id, result);
    };
    store.dispatch(xtremRedux.actions.setStickerDialogId(screenId, dialogControl.id, onFinish));
    return dialogControl;
};
export const getDialogButton = (dialogId, callback, defaultText, buttonConfiguration) => ({
    className: buttonConfiguration?.className,
    isDisabled: buttonConfiguration?.isDisabled,
    isHidden: buttonConfiguration?.isHidden,
    isNegative: buttonConfiguration?.isNegative,
    onClick: (value) => {
        triggerOnCloseActions(dialogId);
        callback(value);
    },
    text: buttonConfiguration?.text || defaultText,
});
const DEFAULT_CONFIRMATION_DIALOG_BUTTON_OPTION = {
    className: 'e-ui-confirmation-dialog-button',
};
export const getAcceptButton = (dialogId, resolve, buttonConfiguration) => getDialogButton(dialogId, resolve, localize('@sage/xtrem-ui/ok', 'OK'), {
    ...DEFAULT_CONFIRMATION_DIALOG_BUTTON_OPTION,
    ...buttonConfiguration,
});
export const getCancelButton = (dialogId, callback, buttonConfiguration) => getDialogButton(dialogId, () => callback(), localize('@sage/xtrem-ui/cancel', 'Cancel'), {
    ...DEFAULT_CONFIRMATION_DIALOG_BUTTON_OPTION,
    ...buttonConfiguration,
    isNegative: true,
});
export const getConfirmationButtons = (options) => (dialogId, resolve, reject) => ({
    accept: getAcceptButton(dialogId, resolve, options?.acceptButton),
    cancel: getCancelButton(dialogId, options?.resolveOnCancel ? resolve : reject, options?.cancelButton),
});
export const errorDialog = (screenId, title, error) => {
    const getButtons = (dialogId, resolve) => {
        return {
            accept: getAcceptButton(dialogId, resolve),
        };
    };
    createDialog(screenId, 'error', title, error, getButtons, undefined, false, 'error').catch((dialogError) => {
        if (isDevMode()) {
            xtremConsole.log('An error ocurred creating the dialog', dialogError);
        }
    });
};
export const confirmationDialog = (screenId, level, title, message, options) => createDialog(screenId, level, title, message, getConfirmationButtons(options), options, false, 'confirmation');
export const messageDialog = (screenId, level, title, message, options) => {
    const getButtons = (dialogId, resolve) => {
        return {
            accept: getAcceptButton(dialogId, resolve, options && options.acceptButton),
        };
    };
    return createDialog(screenId, level, title, message, getButtons, options, false, 'message');
};
export const customDialog = (screenId, level, content, options) => {
    const sections = content instanceof Array ? content : [content];
    const sectionTitle = options?.dialogTitle || getFieldTitle(screenId, sections[0], null) || '';
    return createDialog(screenId, level, sectionTitle, sections, getConfirmationButtons(options), options, false, 'custom');
};
export function resolvePageDialog(resolutionValue) {
    const activeDialogs = Object.keys(xtremRedux.getStore().getState().activeDialogs);
    const lastActiveDialog = activeDialogs[activeDialogs.length - 1];
    xtremRedux.getStore().getState().activeDialogs[lastActiveDialog].dialogControl.resolve(resolutionValue);
}
export const pageDialog = async (store, path, queryParameters = {}, options) => {
    nextDialogId += 1;
    const dialogId = nextDialogId;
    const control = new PageDialogControl(dialogId, options);
    const onFinish = (result) => {
        closeDialog(dialogId, result);
    };
    const screenId = await xtremRedux.actions.loadPageDialogContent(path, queryParameters, options?.values, onFinish, options?.isDuplicate)(store.dispatch, store.getState);
    if (!screenId) {
        throw new Error(localize('@sage/xtrem-ui/page-failed-to-load', 'Failed to load page'));
    }
    const state = store.getState();
    const screenDefinition = state.screenDefinitions[screenId];
    const pageControlObject = screenDefinition.metadata.controlObjects[screenId];
    // Check if the dialog is still open, it could be closed while the page definition was being retrieved
    if (state.activeDialogs[control.id]) {
        store.dispatch(xtremRedux.actions.setScreenDefinitionDialogId(screenId, dialogId, pageControlObject, options?.title || getFieldTitle(screenId, pageControlObject, null) || '', options?.subtitle || pageControlObject.subtitle || ''));
        // Here a fresh copy of the state is needed.
        state.applicationContext?.updateMenu(store.getState().menuItems);
    }
    return control.promise;
};
export const openPageDialog = async (path, queryParameters = {}, options) => pageDialog(xtremRedux.getStore(), path, queryParameters, options);
//# sourceMappingURL=dialog-service.js.map