import { DateRange } from '@sage/xtrem-date-time';
import * as xtremDecimal from '@sage/xtrem-decimal';
import { flat } from '@sage/xtrem-shared';
import { chain, cloneDeep, difference, get, intersection, isEmpty, isNil, set, sortBy, take, uniq, unset, } from 'lodash';
import { FieldKey } from '../component/types';
import { GraphQLKind } from '../types';
import { findColumnDefinitionByBind, getNestedFieldElementId } from '../utils/abstract-fields-utils';
import { xtremConsole } from '../utils/console';
import { cleanMetadataFromRecord, removeNullOnlyLeaves, schemaTypeNameFromNodeName, splitValueToMergedValue, } from '../utils/transformers';
import { isDevMode } from '../utils/window';
import { RecordActionType } from './collection-data-types';
import { filterReservedWords } from './graphql-query-builder';
import { formatInputProperty } from './value-serializer-service';
const numericRegex = /^-?[\d.]+$/;
export const isNumeric = (value) => !!numericRegex.test(String(value).trim());
export const formatValueForSorting = (object, path, type) => {
    const value = get(object, path);
    if (value === null || value === undefined) {
        return { value: '', isDecimal: false };
    }
    if (path.endsWith('_sortValue') && isNumeric(value)) {
        return { value: xtremDecimal.Decimal.make(String(value)), isDecimal: true };
    }
    if ((path.endsWith('_id') || type === FieldKey.Numeric) && isNumeric(value)) {
        return { value: xtremDecimal.Decimal.make(String(value)), isDecimal: true };
    }
    return { value: String(value), isDecimal: false };
};
export const caseInsensitiveSorting = (sortProperties, columnDefinitions) => (obj1, obj2) => {
    for (let i = 0; i < sortProperties.length; i += 1) {
        const sortCondition = sortProperties[i];
        const property = sortCondition[0];
        const bind = property;
        const sortingOrder = sortCondition[1];
        const { type } = findColumnDefinitionByBind(columnDefinitions, bind) ?? {
            type: FieldKey.Text,
            properties: {},
        };
        const { value: value1, isDecimal: isValue1Decimal } = formatValueForSorting(obj1, property, type);
        const { value: value2, isDecimal: isValue2Decimal } = formatValueForSorting(obj2, property, type);
        if (value1 === null || (value1 === undefined && value2 !== null && value2 !== undefined)) {
            return sortingOrder ? -1 : 1;
        }
        if (value2 === null || (value2 === undefined && value1 !== null && value1 !== undefined)) {
            return !sortingOrder ? -1 : 1;
        }
        if (value1 === null || (value1 === undefined && value2 === null && value2 === undefined)) {
            return 0;
        }
        if (isValue1Decimal && isValue2Decimal && bind) {
            const result = xtremDecimal.Decimal.make(isValue1Decimal ? value1 : 0).comparedTo(xtremDecimal.Decimal.make(isValue2Decimal ? value2 : 0));
            if (result !== 0) {
                return sortingOrder ? result * -1 : result;
            }
        }
        else {
            // TODO: switch to numeric: true when Basalt changes their default collation (XT-24930)
            const collator = new Intl.Collator(undefined, { numeric: false, sensitivity: 'base' });
            const result = collator.compare(String(value1), String(value2));
            if (result !== 0) {
                return sortingOrder ? result * -1 : result;
            }
        }
    }
    return 0;
};
export function transformFilterForCollectionValue(outerInput) {
    function getParent({ parent, key, parentKey }) {
        if (parent === '') {
            return key;
        }
        const isOperator = filterReservedWords.includes(key) || filterReservedWords.includes(parentKey);
        if (isOperator || parent.slice(-1) === ']') {
            return `${parent}.${key}`;
        }
        return `${parent}.${key}`;
    }
    const aggregationFilterKeys = [
        '_containedBy',
        '_atLeast',
        '_atMost',
        '_every',
        '_none',
    ];
    function nested({ input, parent = '', parentKey = '', realPath = '', obj = {}, }) {
        if (typeof input !== 'object' || !input) {
            set(obj, parent, input);
            return obj;
        }
        if (Array.isArray(input)) {
            input.forEach((i, index) => nested({
                input: i,
                realPath: `${realPath}[${index}]`,
                parentKey,
                obj,
                parent: `${realPath}[${index}]`,
            }));
        }
        else {
            const inputKeys = Object.keys(input);
            const nonAggregationFilterKeys = inputKeys.filter(k => !aggregationFilterKeys.includes(k));
            if (nonAggregationFilterKeys.length !== inputKeys.length) {
                // eslint-disable-next-line no-restricted-syntax
                for (const key of nonAggregationFilterKeys) {
                    // unset all keys except aggregation as they are not store in the collection value
                    unset(input, key);
                }
            }
            // eslint-disable-next-line no-restricted-syntax
            for (const key of Object.keys(input)) {
                nested({
                    input: input[key],
                    parent: getParent({ key, parent, parentKey }),
                    realPath: realPath === '' ? key : `${realPath}.${key}`,
                    parentKey: key,
                    obj,
                });
            }
        }
        return obj;
    }
    return nested({ input: cloneDeep(outerInput) });
}
/**
 * Sorts a referenced ResultSet based on the received orderBy sequence
 * @param resultSet - lokijs ResultSet instance
 * @param [orderBy] - sequence of property name / sorting type (1 for asc, -1 for desc) pairs
 * @returns ResultSet
 */
export const sortResultSet = ({ resultSet, orderBy, columnDefinitions, }) => {
    if (orderBy && !isEmpty(orderBy)) {
        const flattenedOrderBy = flat(orderBy);
        const sortProperties = Object.keys(flattenedOrderBy).map(key => {
            const isDescending = get(orderBy, key) === -1;
            return [key, isDescending];
        });
        return resultSet.sort(caseInsensitiveSorting([['__level', false], ...sortProperties, ['_id', false]], columnDefinitions));
    }
    return resultSet.compoundsort([
        ['__level', false],
        ['_id', false],
    ]);
};
export function getGroupKey(groupKey) {
    // eslint-disable-next-line redos/no-vulnerable
    const nestedValueMatch = groupKey?.match(/(.*)\./);
    return nestedValueMatch ? `${nestedValueMatch[1]}._id` : (groupKey ?? '');
}
function getNullGroupByCondition({ mode, groupKey }) {
    const split = getGroupKey(groupKey).split('.');
    const prefix = mode === 'client' ? '$' : '_';
    return chain(Array.from({ length: split.length - 1 }, (_, i) => i + 1))
        .map(t => take(split, t).join('.'))
        .map(key => (prefix === '$' ? { [key]: { [`${prefix}eq`]: null } } : set({}, key, { [`${prefix}eq`]: null })))
        .thru(clause => ({
        [`${prefix}or`]: clause.concat(prefix === '$'
            ? { [groupKey]: { [`${prefix}eq`]: null } }
            : set({}, groupKey, { [`${prefix}eq`]: null })),
    }))
        .value();
}
export const getGroupFilterValue = ({ group, mode, }) => {
    if (group?.value == null) {
        return null;
    }
    if (group.value === '') {
        return getNullGroupByCondition({ groupKey: group.key, mode });
    }
    const prefix = mode === 'client' ? '$' : '_';
    if (group.aggFunc) {
        const { start, end } = DateRange.getDateRange({ date: group.value, range: `same-${group.aggFunc}` });
        const filterValue = {
            [`${prefix}and`]: [{ [`${prefix}gte`]: start?.toString() }, { [`${prefix}lte`]: end?.toString() }],
        };
        return mode === 'client' ? { [group.key]: filterValue } : set({}, group.key, filterValue);
    }
    return mode === 'client'
        ? { [getGroupKey(group.key)]: group.value }
        : set({}, getGroupKey(group.key), { _eq: group.value });
};
export class CollectionDataRow {
    constructor(input, columnDefinitions, nodes, nodeTypes, options) {
        this.input = input;
        this.columnDefinitions = columnDefinitions;
        this.nodes = nodes;
        this.nodeTypes = nodeTypes;
        this.options = options;
        this.getNode = (level) => {
            if (this.options?.isTree) {
                return this.nodes[0];
            }
            const targetLevel = isNil(level) ? 0 : level;
            return this.nodes[targetLevel];
        };
        this.getColumnDefinitions = (level) => {
            if (this.options?.isTree) {
                return this.columnDefinitions?.[0] || [];
            }
            const targetLevel = isNil(level) ? 0 : level;
            return this.columnDefinitions?.[targetLevel] || [];
        };
        this.result = input;
        this.includeActions = this.options?.includeActions ?? true;
        this.removeNegativeId = this.options?.removeNegativeId ?? false;
        this.isRequestingDefaults = this.options?.isRequestingDefaults ?? false;
        this.cleanInputTypes = this.options?.cleanInputTypes ?? true;
    }
    getLevel() {
        return this.input.__level ?? 0;
    }
    getDirtyColumns() {
        return this.input.__dirtyColumns;
    }
    getAction() {
        return this.input.__action;
    }
    isRemoved() {
        return this.getAction() === RecordActionType.REMOVED;
    }
    removeTransientColumns() {
        if (!this.isRemoved()) {
            const level = this.getLevel();
            this.result = Object.keys(this.result).reduce((acc, key) => {
                const colDef = this.getColumnDefinitions(level)?.find(d => getNestedFieldElementId(d) === key);
                if (!colDef?.properties?.isTransient) {
                    acc[key] = this.result[key];
                }
                return acc;
            }, {});
        }
        return this;
    }
    // This function should remove all computed columns like Aggregation from the record.
    removeComputedColumns() {
        if (!this.isRemoved()) {
            const level = this.getLevel();
            this.result = Object.keys(this.result).reduce((acc, key) => {
                const colDef = this.getColumnDefinitions(level)?.find(d => getNestedFieldElementId(d) === key);
                const isComputed = colDef?.properties._controlObjectType === FieldKey.Aggregate ||
                    colDef?.properties._controlObjectType === FieldKey.Count;
                if (!isComputed) {
                    acc[key] = this.result[key];
                }
                return acc;
            }, {});
        }
        return this;
    }
    cleanupRemoved() {
        if (this.getAction() === RecordActionType.REMOVED) {
            this.result = { _id: this.result._id, _action: RecordActionType.REMOVED };
        }
        return this;
    }
    cleanMetadataFromRecord() {
        if (!this.isRemoved()) {
            this.result = cleanMetadataFromRecord(this.result, true, this.isRequestingDefaults);
        }
        return this;
    }
    removeNonInputTypes() {
        if (!this.isRemoved() && this.cleanInputTypes) {
            const level = this.getLevel();
            const node = this.nodeTypes[schemaTypeNameFromNodeName(this.getNode(level))];
            if (node) {
                const inputProperties = Object.keys(node.properties).filter(k => !!node.properties[k].isOnInputType);
                const inputKeys = sortBy(uniq([...inputProperties, '_id', '_action', '_sortValue']));
                this.result = intersection(inputKeys, Object.keys(this.result)).reduce((acc, key) => {
                    acc[key] = this.result[key];
                    return acc;
                }, {});
                // Print warning in dev mode in case any key is removed
                if (isDevMode()) {
                    const removedKeys = difference(Object.keys(this.result), inputKeys);
                    if (removedKeys.length > 0) {
                        xtremConsole.warn(`The following keys were removed from ${JSON.stringify(this.result)}: ${JSON.stringify(removedKeys)}, are you sure you are using the correct node type ('${this.getNode(level)}')?`);
                    }
                }
            }
        }
        return this;
    }
    extractDirty() {
        if (!this.isRemoved() && this.isRequestingDefaults) {
            const dirtyColumns = this.getDirtyColumns();
            this.result = Object.keys(this.result).reduce((acc, key) => {
                if (dirtyColumns?.has(key)) {
                    acc[key] = this.result[key];
                }
                return acc;
            }, { _id: this.result._id, _action: this.result._action });
        }
        return this;
    }
    cloneAction() {
        if (this.includeActions) {
            this.result._action = this.input.__action || 'update';
        }
        return this;
    }
    deleteNegativeId() {
        if (this.removeNegativeId) {
            const { _id, ...rest } = this.result;
            this.result = {
                ...(_id !== undefined && _id !== null && Number.parseInt(_id, 10) > 0 && { _id }),
                ...rest,
            };
        }
        return this;
    }
    removeNullOnlyLeaves() {
        removeNullOnlyLeaves(this.result);
        return this;
    }
    splitValueToMergedValue() {
        this.result = splitValueToMergedValue(this.result);
        return this;
    }
    getChangeset() {
        const result = this.cleanupRemoved()
            .cloneAction()
            .removeTransientColumns()
            .removeComputedColumns()
            .cleanMetadataFromRecord()
            .splitValueToMergedValue()
            .removeNonInputTypes()
            .extractDirty()
            .removeNullOnlyLeaves()
            .deleteNegativeId().result;
        if (!this.cleanInputTypes) {
            return result;
        }
        const level = this.getLevel();
        return formatInputProperty({
            value: result,
            nodeTypes: this.nodeTypes,
            isTopLevel: true,
            property: {
                type: schemaTypeNameFromNodeName(this.getNode(level)),
                kind: GraphQLKind.Object,
                isMutable: true,
            },
        });
    }
}
//# sourceMappingURL=collection-data-utils.js.map