import { fetchCollectionData, fetchDefaultValues } from '../../service/graphql-service';
import { getScreenElement } from '../../service/screen-base-definition';
import { triggerScreenEvent } from '../../utils/events';
import { ActionType } from '../action-types';
import { getPageDefinitionFromState, getPagePropertiesFromPageDefinition, getPagePropertiesFromState, hasAnyDirtyScreenDefinitions, isScreenDefinitionDirty, } from '../../utils/state-utils';
import { formatScreenValues } from '../../service/value-formatter-service';
import { getStore } from '../store';
import { notifyConsumerAboutDirtyStatus, onPageDirtyChange } from '../../service/dirty-state-service';
import { setComponentLoading } from './common-actions';
import { xtremConsole } from '../../utils/console';
import { NEW_PAGE } from '../../utils/constants';
import { FieldKey } from '../../component/types';
export const setFieldValue = (screenId, elementId, fieldValue, isOrganicChange = false) => async (dispatch, getState) => {
    let state = getState();
    let screenDefinition = state.screenDefinitions[screenId];
    const originalPageDirtyState = isScreenDefinitionDirty(screenDefinition);
    const originalAppDirtyState = hasAnyDirtyScreenDefinitions(state);
    const page = getScreenElement(screenDefinition);
    if (page.$.isTransactionInProgress()) {
        /**
         * If the dirty event listeners set properties or values, it can cause a conflict with the SetValueField below. The internal set value field action
         * does not use the transactions, so it can happen that the values are overwritten by the transaction which is committed after the values are set.
         *  */
        await page.$.commitValueAndPropertyChanges();
    }
    // If it is a user initiated, organic change and the application is currently clean, we tell the consumer listener that the application has just become dirty.
    if (isOrganicChange &&
        !originalAppDirtyState &&
        !screenDefinition.metadata.uiComponentProperties[screenId].skipDirtyCheck) {
        notifyConsumerAboutDirtyStatus(state, true);
    }
    dispatch({
        type: ActionType.SetFieldValue,
        value: {
            screenId,
            elementId,
            fieldValue,
            isOrganicChange,
        },
    });
    if (isOrganicChange) {
        state = getState();
        state.applicationContext?.onTelemetryEvent?.(`fieldValueModified-${elementId}`, { screenId, elementId });
        screenDefinition = state.screenDefinitions[screenId];
        const fieldProperties = screenDefinition.metadata.uiComponentProperties[elementId];
        await applyDefaultValues({
            dispatch,
            screenDefinition,
            fieldProperties,
            plugins: state.plugins,
            nodeTypes: state.nodeTypes,
        });
    }
    // If it is a user initiated, organic change and the page is currently clean, we tell the screen listener that the screen has just become dirty.
    if (isOrganicChange && !originalPageDirtyState) {
        onPageDirtyChange(state, screenId, true);
        triggerScreenEvent(screenId, 'onDirtyStateUpdated', true);
    }
};
export const applyDefaultValues = async ({ dispatch, screenDefinition, fieldProperties, plugins, nodeTypes, requestedFieldIds, skipDispatch = false, }) => {
    const screenProperties = getPagePropertiesFromPageDefinition(screenDefinition);
    const queryParameters = screenDefinition?.queryParameters;
    // This block will check for tables with phantom rows and apply default values to it.
    if ((requestedFieldIds || fieldProperties?.fetchesDefaults) &&
        screenProperties.node &&
        !screenProperties.isTransient &&
        screenDefinition.type === 'page') {
        // Function that should retrieve tables with phantomRow from uiComponentProperties
        const getPhantomRowTableIds = (uiComponentProperties) => {
            return Object.keys(uiComponentProperties).filter(value => uiComponentProperties[value]._controlObjectType === FieldKey.Table &&
                !uiComponentProperties[value].isTransient &&
                uiComponentProperties[value].canAddNewLine);
        };
        getPhantomRowTableIds(screenDefinition.metadata.uiComponentProperties).forEach(tableId => {
            screenDefinition.values[tableId].resetPhantomRowToDefault();
        });
    }
    // This block will be executed only if the field has fetchesDefaults set to true AND is a new record after that will update other non dirty fields.
    if ((requestedFieldIds || fieldProperties?.fetchesDefaults) &&
        screenProperties.node &&
        !screenProperties.isTransient &&
        screenDefinition.type === 'page' &&
        (requestedFieldIds || !queryParameters?._id || queryParameters?._id === NEW_PAGE) &&
        !queryParameters.__duplicate // We don't want to apply default values to duplicate pages
    ) {
        const defaultValues = await fetchDefaultValues(screenDefinition, String(screenProperties.node), requestedFieldIds, false, nodeTypes);
        dispatchDefaultValuesSet({ screenDefinition, plugins, nodeTypes, defaultValues, dispatch, skipDispatch });
        return defaultValues;
    }
    return {};
};
export const dispatchDefaultValuesSet = ({ screenDefinition, plugins, nodeTypes, defaultValues, dispatch = getStore().dispatch, skipDispatch = false, }) => {
    const keys = Object.keys(defaultValues);
    const screenId = screenDefinition.metadata.screenId;
    const formattedDefaultValues = formatScreenValues({
        screenId,
        controlObjects: screenDefinition.metadata.controlObjects,
        plugins,
        nodeTypes,
        values: defaultValues,
        parentNode: undefined,
        onlyElementIds: keys,
    });
    delete formattedDefaultValues._id;
    if (!skipDispatch) {
        dispatch({
            type: ActionType.SetValues,
            value: {
                screenId: screenDefinition.metadata.screenId,
                values: { ...screenDefinition.values, ...formattedDefaultValues },
                preserveValidationState: true,
            },
        });
    }
    return formattedDefaultValues;
};
export const loadCollectionData = (screenId, elementId, nestedFields, queryArguments, bind, keepAllRecords = false) => (dispatch, getState) => {
    // eslint-disable-next-line no-async-promise-executor
    return new Promise(async (res, rej) => {
        try {
            const state = getState();
            const pageDefinition = getPageDefinitionFromState(screenId, state);
            const pageProperties = getPagePropertiesFromState(screenId, state);
            const rootNode = pageProperties.node;
            const selectedRecordId = pageDefinition.selectedRecordId || '';
            dispatch(setComponentLoading(screenId, elementId, true));
            const result = await fetchCollectionData({
                screenDefinition: pageDefinition,
                rootNode,
                rootNodeId: selectedRecordId,
                elementId,
                nestedFields,
                queryArguments,
                bind,
            });
            dispatch(setComponentLoading(screenId, elementId, false));
            const currentValue = pageDefinition.values[elementId];
            const updatedValue = keepAllRecords
                ? { ...result, data: currentValue.data.concat(result.data) }
                : result;
            await dispatch(setFieldValue(screenId, elementId, updatedValue, false));
            res(updatedValue);
        }
        catch (err) {
            dispatch(setComponentLoading(screenId, elementId, false));
            xtremConsole.error('Error filtering collection items', err);
            rej(err);
        }
    });
};
//# sourceMappingURL=field-value-actions.js.map