import { getFieldTitle, getPageSubtitle } from '../../component/field/carbon-helpers';
import { messageDialog } from '../../service/dialog-service';
import { notifyConsumerAboutDirtyStatus } from '../../service/dirty-state-service';
import { localize } from '../../service/i18n-service';
import { fetchPageDefinition } from '../../service/screen-loader-service';
import { commitTransaction, setUiComponentProperties } from '../../service/transactions-service';
import { NEW_PAGE } from '../../utils/constants';
import { triggerScreenEvent } from '../../utils/events';
import { getMainPageDefinitionFromState, getPagePropertiesFromState } from '../../utils/state-utils';
import { ActionType } from '../action-types';
import { addScreenDefinition, removeScreenDefinition, setScreenDefinitionReady } from './common-actions';
import { addMenuItem, removeMenuItem } from './menu-actions';
import { closeDialogMenuItemId, getCloseDialogMenuItem } from './menu-items';
import { getPageTitlesFromPageDefinition } from '../../utils/page-utils';
import { xtremConsole } from '../../utils/console';
const closeOpenedDialogs = (screenDefinitions, screenId) => {
    Object.keys(screenDefinitions[screenId].metadata.uiComponentProperties).forEach(fieldId => {
        if (screenDefinitions[screenId].metadata.uiComponentProperties[fieldId]
            .isReferenceDialogOpen === true) {
            setUiComponentProperties(screenId, fieldId, {
                ...screenDefinitions[screenId].metadata.uiComponentProperties[fieldId],
                isReferenceDialogOpen: false,
            });
        }
    });
};
export const closeDialog = (dialogId) => async (dispatch, getState) => {
    const state = getState();
    const activeDialogs = state.activeDialogs;
    const dialogDescription = activeDialogs[dialogId];
    const screenDefinitions = state.screenDefinitions;
    if (dialogDescription && dialogDescription.isSticker && dialogDescription.screenId) {
        const screenId = dialogDescription.screenId;
        // If the dialog we close is a sticker, then we need to close all reference look up dialogs on the sticker
        closeOpenedDialogs(screenDefinitions, screenId);
    }
    dispatch(removeMenuItem(closeDialogMenuItemId));
    dispatch({ type: ActionType.CloseDialog, value: dialogId });
    if (dialogDescription &&
        dialogDescription.screenId &&
        !dialogDescription.isSticker &&
        dialogDescription.type === 'page') {
        // We need to call the onClose event of the page dialog before destroying the page definition
        await triggerScreenEvent(dialogDescription.screenId, 'onClose');
        dispatch(removeScreenDefinition(dialogDescription.screenId));
    }
    const applicationContext = getState().applicationContext;
    if (applicationContext?.onPageTitleChange) {
        const mainPage = getMainPageDefinitionFromState(state);
        if (mainPage) {
            const screenId = mainPage.metadata.screenId;
            const pageProperties = getPagePropertiesFromState(screenId, state);
            const pageTitle = pageProperties && getFieldTitle(screenId, pageProperties, null);
            const pageSubtitle = pageProperties && getPageSubtitle(screenId, pageProperties, null);
            applicationContext.onPageTitleChange(pageTitle || null, pageSubtitle || null);
        }
        else {
            applicationContext.onPageTitleChange(null, null);
        }
    }
    // If we already in a dirtyCheck dialog we will not call again the notifyConsumerAboutDirtyStatus
    if (!dialogDescription?.isDirtyCheck) {
        // We need a fresh copy of the state here.
        notifyConsumerAboutDirtyStatus(getState());
    }
};
export const openDialog = (dialogId, dialog) => (dispatch, getState) => {
    dispatch(addMenuItem(getCloseDialogMenuItem(dialogId, dialog.isSticker, dialog.title)));
    dispatch({ type: ActionType.OpenDialog, value: { dialog, dialogId } });
    const applicationContext = getState().applicationContext;
    if (applicationContext && applicationContext.onPageTitleChange) {
        applicationContext.onPageTitleChange(dialog.title || null, dialog.subtitle || null);
    }
};
export const loadPageDialogContent = (path, queryParameters = {}, 
/**
 * If the values are provided, they are not fetched from the server.
 * This feature is used to pre-populate page dialogs with data.
 *  */
values, onFinish, isDuplicate = false) => async (dispatch, getState) => {
    const pageDefinition = await fetchPageDefinition({
        getState,
        dispatch,
        path,
        isMainPage: false,
        queryParameters,
        values,
        onFinish,
        isDuplicate,
    });
    if (!pageDefinition) {
        return null;
    }
    pageDefinition.isInDialog = true;
    dispatch(addScreenDefinition(pageDefinition));
    const screenId = pageDefinition.metadata.screenId;
    try {
        pageDefinition.page.$standardDeleteAction.isDisabled =
            !queryParameters._id || queryParameters._id === NEW_PAGE;
        pageDefinition.page.$standardDuplicateAction.isDisabled =
            !queryParameters._id || queryParameters._id === NEW_PAGE;
        await triggerScreenEvent(screenId, 'onLoad');
    }
    catch (e) {
        xtremConsole.error(e);
        messageDialog(screenId, 'error', 'Error', localize('@sage/xtrem-ui/dialog-actions-onload-unhandled-error', 'An unhandled error ocurred'));
    }
    commitTransaction(screenId);
    dispatch(setScreenDefinitionReady(screenId));
    const state = getState();
    const { title, subtitle } = getPageTitlesFromPageDefinition(pageDefinition, state.applicationContext?.locale || 'en-US');
    state.applicationContext?.onPageTitleChange?.(title, subtitle || null);
    return screenId;
};
//# sourceMappingURL=dialog-actions.js.map