import React from 'react';
import * as wrapperService from '../../../service/wrapper-service';
import { getFieldTitle } from '../../field/carbon-helpers';
import { localize } from '../../../service/i18n-service';
import { Icon } from '../icon/icon-component';
import IconButton from 'carbon-react/esm/components/icon-button';
import Textbox from 'carbon-react/esm/components/textbox';
import { debounce } from 'lodash';
export const getSearchTextboxId = (screenId, elementId) => screenId && elementId ? `e-table-field-mobile-search-input-${screenId}-${elementId}`.replace('$', '') : undefined;
export function TableSearchBox({ disableAutoFocus, elementId, fieldProperties, filterText = '', isDisabled, onSearchBoxValueChange, screenId, }) {
    const [searchText, setSearchText] = React.useState(filterText);
    const inputRef = React.useRef(null);
    React.useEffect(() => {
        setSearchText(filterText);
    }, [filterText]);
    const debouncedSearch = React.useMemo(() => debounce((filter) => {
        onSearchBoxValueChange(filter);
    }, 500), [onSearchBoxValueChange]);
    // Debounce the actual filter value update so we don't refetch on every key stroke
    const debouncedTextFilterUpdate = React.useCallback(debouncedSearch, [onSearchBoxValueChange, debouncedSearch]);
    const onSearchTextChanged = React.useCallback((event) => {
        setSearchText(event.target.value);
        debouncedTextFilterUpdate(event.target.value);
    }, [setSearchText, debouncedTextFilterUpdate]);
    const onSearchTextFocused = React.useCallback((event) => {
        if (fieldProperties && elementId && screenId) {
            wrapperService.onFocus({
                screenId,
                elementId,
                title: getFieldTitle(screenId, fieldProperties, null),
            }, (updatedSearchText) => {
                setSearchText(updatedSearchText);
                onSearchBoxValueChange(updatedSearchText);
            });
            if (event.target) {
                // Sets selection to the end of the current input.
                setTimeout(() => {
                    event.target.setSelectionRange(event.target.value.length, event.target.value.length);
                }, 10);
            }
        }
    }, [setSearchText, onSearchBoxValueChange, elementId, screenId, fieldProperties]);
    const onClearButtonClicked = React.useCallback(() => {
        setSearchText('');
        onSearchBoxValueChange('');
        if (inputRef?.current) {
            inputRef.current.focus();
        }
    }, [setSearchText, onSearchBoxValueChange, inputRef]);
    const onBlur = React.useCallback(() => {
        if (screenId && elementId) {
            wrapperService.onBlur({
                screenId,
                elementId,
            });
        }
    }, [screenId, elementId]);
    return (React.createElement("div", { className: "e-table-field-mobile-search", "data-testid": "e-table-field-mobile-search" },
        React.createElement(Textbox, { id: getSearchTextboxId(screenId, elementId), ref: inputRef, autoFocus: !disableAutoFocus, onKeyDown: (e) => {
                // On hitting the enter key the component reloads the page for some reason
                if (e.key === 'Enter') {
                    e.preventDefault();
                }
            }, onChange: onSearchTextChanged, onFocus: onSearchTextFocused, placeholder: localize('@sage/xtrem-ui/reference-lookup-dialog-search-placeholder', 'Search...'), "aria-label": localize('@sage/xtrem-ui/reference-lookup-dialog-search-placeholder', 'Search...'), leftChildren: React.createElement(Icon, { type: "search", bgSize: "large" }), value: searchText, disabled: isDisabled, onBlur: onBlur }, !!searchText && (React.createElement(IconButton, { onAction: onClearButtonClicked, "aria-label": localize('@sage/xtrem-ui/clear-filter-text', 'Clear filter text') },
            React.createElement(Icon, { type: "cross" }))))));
}
//# sourceMappingURL=table-search-box-component.js.map