import React from 'react';
import { fetchCollectionDataCount } from '../../../service/graphql-service';
import { getDataTestIdAttribute } from '../../../utils/dom';
import { getGraphFilter, getTableContext, setTableContext } from '../../../utils/table-component-utils';
export function TableRowSelectionHeader({ api, screenId, elementId, eGridHeader, node, fieldProperties, activeOptionsMenuItem, }) {
    const [checked, setChecked] = React.useState(Boolean(getTableContext(api).isSelectAllEnabled));
    const [prevChecked, setPrevChecked] = React.useState(checked);
    const [partial, setPartial] = React.useState(false);
    const inputRef = React.useRef(null);
    // Uncheck checkbox when no rows are selected.
    React.useEffect(() => {
        const listener = ({ api: gridApi }) => {
            const selectedCount = gridApi.getSelectedNodes().length;
            if (selectedCount === 0) {
                setChecked(false);
                setPartial(false);
            }
            if (selectedCount > 0) {
                let isPartial = false;
                gridApi.forEachNode(n => {
                    if (!n.isSelected()) {
                        isPartial = true;
                    }
                });
                setPartial(isPartial);
            }
        };
        api.addEventListener('rowSelected', listener);
        return () => api.removeEventListener('rowSelected', listener);
    }, [api]);
    // Focus on checkbox instead of ag-grid wrapper.
    React.useEffect(() => {
        const listener = () => {
            setTimeout(() => {
                inputRef.current?.focus();
            }, 0);
        };
        eGridHeader.addEventListener('focus', listener);
        return () => eGridHeader.removeEventListener('focus', listener);
    }, [eGridHeader]);
    // Fetch count when checkbox is checked
    React.useEffect(() => {
        // We only want to execute this piece of code when the user manually clicks on the checkbox and not when other hook dependencies are updated
        if (checked !== prevChecked) {
            //  When we re-render due to ungroup from the main component the checkbox is not checked but the state is not updated to reflect it.
            if (inputRef.current?.checked !== checked) {
                setChecked(inputRef.current?.checked || false);
            }
            setTableContext(api, c => {
                c.isSelectAllEnabled = checked;
            });
            setPartial(false);
            if (checked) {
                const filter = getGraphFilter({
                    filterModel: api.getFilterModel(),
                    screenId,
                    tableFieldProperties: fieldProperties,
                    activeOptionsMenuItem,
                });
                fetchCollectionDataCount({
                    rootNode: node,
                    filter: JSON.stringify(filter),
                }).then(totalRowCount => {
                    setTableContext(api, c => {
                        c.totalRowCount = totalRowCount;
                    });
                    api.forEachNode(n => n.setSelected(true, false));
                    // call 'onSelectAll' provided by desktop-table through ag-grid's context
                    getTableContext(api).onSelectAll?.(totalRowCount);
                });
            }
            else {
                setTableContext(api, c => {
                    c.totalRowCount = 0;
                });
                api.getSelectedNodes().forEach(n => n.setSelected(false, false));
                // call 'onUnselectAll' provided by desktop-table through ag-grid's context
                getTableContext(api).onUnselectAll?.();
            }
            setPrevChecked(checked);
        }
    }, [prevChecked, checked, api, screenId, fieldProperties, node, activeOptionsMenuItem]);
    const handleChange = React.useCallback(() => {
        setChecked((currentChecked) => !currentChecked);
    }, []);
    return (React.createElement("div", { role: "presentation", className: "ag-labeled ag-label-align-right ag-checkbox ag-input-field" },
        React.createElement("div", { className: `ag-wrapper ag-input-wrapper ag-checkbox-input-wrapper ${checked && 'ag-checked'} ${partial && 'ag-partial'}`, role: "presentation" },
            React.createElement("input", { ref: inputRef, "data-testid": getDataTestIdAttribute('table', 'select-all', elementId), className: "ag-input-field-input ag-checkbox-input", type: "checkbox", "aria-label": "Press Space to toggle row selection", tabIndex: -1, onChange: handleChange, checked: checked }))));
}
//# sourceMappingURL=table-row-selection-header.js.map