import CheckboxComponent from 'carbon-react/esm/components/checkbox/checkbox.component';
import Link from 'carbon-react/esm/components/link';
import Loader from 'carbon-react/esm/components/loader';
import Textbox from 'carbon-react/esm/components/textbox';
import { camelCase, debounce, set, uniqBy } from 'lodash';
import * as React from 'react';
import { fetchReferenceFieldSuggestions } from '../../../../../service/graphql-service';
import { localize } from '../../../../../service/i18n-service';
import { xtremConsole } from '../../../../../utils/console';
import { convertDeepBindToPathNotNull } from '../../../../../utils/nested-field-utils';
import { nodesToSelectItems } from '../../../../field/reference/reference-utils';
import { SET } from '../../../../types';
const getItems = async (filterValue, columnDefinition, after) => {
    const valueField = columnDefinition.cellRendererParams.fieldProperties.valueField;
    const isFilterLimitedToDataset = columnDefinition.cellRendererParams.fieldProperties.isFilterLimitedToDataset;
    const nodes = await fetchReferenceFieldSuggestions({
        after,
        contextNode: columnDefinition.cellRendererParams.contextNode,
        fieldProperties: {
            ...columnDefinition.cellRendererParams.fieldProperties,
            isTransient: true,
            columns: undefined,
            helperTextField: undefined,
            minLookupCharacters: 0,
            orderBy: set({}, convertDeepBindToPathNotNull(valueField), 1),
        },
        fieldId: columnDefinition.cellRendererParams.columnId,
        filterValue: filterValue || undefined,
        isFilterLimitedToDataset,
        parentElementId: columnDefinition.cellRendererParams.tableElementId,
        recordContext: {},
        screenId: columnDefinition.screenId,
    });
    return nodesToSelectItems({
        nodes,
        fieldProperties: columnDefinition.cellRendererParams.fieldProperties,
    });
};
const ReferenceFilterItem = React.memo(({ textFilter: searchCriteria, itemValue }) => {
    if (!itemValue) {
        return React.createElement("span", { className: "e-reference-custom-filter-item-value" });
    }
    if (!searchCriteria) {
        return React.createElement("span", { className: "e-reference-custom-filter-item-value" }, itemValue);
    }
    const pos = itemValue.toLowerCase().indexOf(searchCriteria.toLowerCase());
    const firstPart = itemValue.substr(0, pos);
    const highlightedPart = itemValue.substr(pos, searchCriteria.length);
    const lastPart = itemValue.substr(pos + searchCriteria.length);
    return (React.createElement("span", { className: "e-reference-custom-filter-item-value" },
        firstPart,
        React.createElement("span", { className: "e-reference-custom-filter-item-value-highlighted" }, highlightedPart),
        lastPart));
});
ReferenceFilterItem.displayName = 'ReferenceFilterItem';
/**
 * Custom filter component for AG-Grid, handling references.
 *
 * Not implemented:
 * - Support for nested grids
 * - "Select All" functionality
 */
export function ReferenceFilter({ colDef, onModelChange, model }) {
    const [textFilter, setTextFilter] = React.useState('');
    const [textFilterDisplayValue, setTextFilterDisplayValue] = React.useState('');
    const [hasNextPage, setHasNextPage] = React.useState(true);
    const [isFetching, setFetching] = React.useState(true);
    const [availableItems, setAvailableItems] = React.useState([]);
    const debouncedSearch = React.useMemo(() => debounce((filter) => {
        setTextFilter(filter);
    }, 500), []);
    // Debounce the actual filter value update so we don't refetch on every key stroke
    const debouncedTextFilterUpdate = React.useCallback(debouncedSearch, [debouncedSearch]);
    React.useEffect(() => {
        getItems(textFilter, colDef)
            .then(items => {
            setAvailableItems(items);
        })
            .catch(e => {
            xtremConsole.error(e);
            setAvailableItems([]);
        })
            .finally(() => {
            setFetching(false);
        });
    }, [colDef, textFilter]);
    const onTextFilterChange = React.useCallback((ev) => {
        setHasNextPage(true);
        setTextFilterDisplayValue(ev.target.value);
        debouncedTextFilterUpdate(ev.target.value);
    }, [debouncedTextFilterUpdate]);
    const onSelectionChange = React.useCallback((isChecked, selectedItem) => {
        const newFilter = [...(model?.values || [])];
        if (isChecked) {
            newFilter.push(selectedItem.id);
        }
        else {
            const index = newFilter.findIndex(i => i === selectedItem.id);
            if (index !== -1) {
                newFilter.splice(index, 1);
            }
        }
        onModelChange(newFilter.length > 0
            ? {
                filterType: SET,
                values: newFilter,
                /**
                 * By default, the filtering applied to the value field text value, so we need to pass this flag to indicate that
                 * the records should be filtered by their ID values
                 *  */
                filterById: true,
                // This value is used by the ReferenceFloatingFilter component to display the selection summary in the header
                displayValues: newFilter.map(r => {
                    const item = availableItems.find(i => i.id === r);
                    return item?.displayedAs || item?.value || r;
                }),
            }
            : null);
    }, [availableItems, model?.values, onModelChange]);
    const onScroll = React.useCallback(async (event) => {
        const listBody = event.target;
        // If the scrollbar is 30 pixels away from the bottom, then we initiate the fetch of the next set of records
        const shouldLoadNextPage = listBody.scrollHeight - listBody.scrollTop <= listBody.clientHeight + 30;
        if (!isFetching && shouldLoadNextPage && hasNextPage) {
            const lastItem = availableItems[availableItems.length - 1];
            const newItems = await getItems(textFilter, colDef, lastItem.__collectionItem?.__cursor);
            setHasNextPage(newItems.length > 0);
            setAvailableItems(uniqBy([...availableItems, ...newItems], i => i.id));
            setFetching(false);
        }
    }, [availableItems, colDef, hasNextPage, isFetching, textFilter]);
    const onClearSelection = React.useCallback((ev) => {
        ev.preventDefault();
        ev.stopPropagation();
        onModelChange(null);
    }, [onModelChange]);
    const debouncedOnScroll = React.useMemo(() => debounce(onScroll, 1000), [onScroll]);
    const isClearSelectedDisabled = React.useMemo(() => (model?.values ?? []).length === 0, [model?.values]);
    const onChange = React.useCallback(item => (event) => onSelectionChange(!!event.target.checked, item), [onSelectionChange]);
    return (React.createElement("div", { className: "e-reference-custom-filter" },
        React.createElement("div", { className: "e-reference-custom-filter-clear-container", "data-testid": "e-reference-custom-filter-clear-button" },
            React.createElement(Link, { onClick: onClearSelection, disabled: isClearSelectedDisabled }, localize('@sage/xtrem-ui/reference-filter-clear-selection', 'Clear selected items'))),
        React.createElement(Textbox, { "data-testid": "e-reference-custom-filter-input", placeholder: localize('@sage/xtrem-ui/filter-search-placeholder', 'Search'), value: textFilterDisplayValue, onChange: onTextFilterChange, size: "small", mx: "8px", my: "4px" }),
        React.createElement("div", { className: "e-reference-custom-filter-body", onScroll: debouncedOnScroll },
            availableItems.map(item => (React.createElement("div", { className: "e-reference-custom-filter-item", "data-testid": "e-reference-custom-filter-item", key: item.id },
                React.createElement(CheckboxComponent, { onChange: onChange(item), key: item.id, value: item.id, checked: (model?.values ?? []).includes(item.id), "data-testid": `e-reference-custom-filter-checkbox-label-${camelCase(String(item.value))} e-reference-custom-filter-checkbox-bind-${item.id}` }),
                React.createElement(ReferenceFilterItem, { itemValue: String(item.value), textFilter: textFilter })))),
            isFetching && (React.createElement("div", { className: "e-reference-custom-filter-loader" },
                React.createElement(Loader, null))))));
}
ReferenceFilter.displayName = 'ReferenceFilter';
export default ReferenceFilter;
//# sourceMappingURL=reference-filter.js.map