import { Checkbox } from 'carbon-react/esm/components/checkbox';
import * as React from 'react';
import { ConnectedNestedFieldWrapper } from '../../../render/nested-field-wrapper';
import { localize } from '../../../service/i18n-service';
import { getNestedFieldElementId, normalizeUnderscoreBind } from '../../../utils/abstract-fields-utils';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { cleanMetadataFromRecord, splitValueToMergedValue } from '../../../utils/transformers';
import ConnectedTableDropdownActions from '../table-shared/table-dropdown-actions/table-dropdown-actions';
import { getImagePlaceholderValue } from '../../../utils/nested-field-utils';
import { get, isNil } from 'lodash';
import ProgressTracker from 'carbon-react/esm/components/progress-tracker';
import IconButton from 'carbon-react/esm/components/icon-button';
import Icon from 'carbon-react/esm/components/icon';
import * as tokens from '@sage/design-tokens/js/base/common';
import ConnectedTableInlineActions from '../table-shared/table-inline-actions';
export const getNestedValue = (value, bind) => {
    if (!(value instanceof Object) || !bind) {
        return null;
    }
    const accessor = normalizeUnderscoreBind(bind);
    const result = get(value, accessor, null);
    if (isNil(result)) {
        return null;
    }
    return result;
};
const isHidden = (screenId, value, field, fieldValue) => !!field &&
    !!resolveByValue({
        screenId,
        propertyValue: field.properties.isHidden,
        rowValue: splitValueToMergedValue(value),
        fieldValue,
        skipHexFormat: true,
    });
const renderCard = ({ canSelect, cardDefinition, contextType, isDisabled, isNewItem, isSelected, onSelect, onCardClick, parentElementId, inlineActions, dropdownActions, screenId, value, isClickIndicatorHidden, }) => {
    if (!value) {
        return null;
    }
    const _id = value._id;
    const level = value.__level || 0;
    const cleanValue = cleanMetadataFromRecord(value);
    const titleLineValueId = getNestedFieldElementId(cardDefinition.title);
    const titleRightValueId = getNestedFieldElementId(cardDefinition.titleRight);
    const line2ValueId = getNestedFieldElementId(cardDefinition.line2);
    const line2RightValueId = getNestedFieldElementId(cardDefinition.line2Right);
    const line3ValueId = getNestedFieldElementId(cardDefinition.line3);
    const line3RightValueId = getNestedFieldElementId(cardDefinition.line3Right);
    const imageValueId = getNestedFieldElementId(cardDefinition.image);
    const progressValueId = getNestedFieldElementId(cardDefinition.progressBar);
    const titleLineValue = getNestedValue(value, titleLineValueId);
    const titleRightValue = getNestedValue(value, titleRightValueId);
    const line2Value = getNestedValue(value, line2ValueId);
    const line2RightValue = getNestedValue(value, line2RightValueId);
    const line3Value = getNestedValue(value, line3ValueId);
    const line3RightValue = getNestedValue(value, line3RightValueId);
    const imageValue = getNestedValue(value, imageValueId);
    const progressValue = getNestedValue(value, progressValueId);
    const titleRightHidden = isHidden(screenId, value, cardDefinition.titleRight, titleRightValue);
    const line2Hidden = isHidden(screenId, value, cardDefinition.line2, line2Value);
    const line2RightHidden = isHidden(screenId, value, cardDefinition.line2Right, line2RightValue);
    const line3Hidden = isHidden(screenId, value, cardDefinition.line3, line3Value);
    const line3RightHidden = isHidden(screenId, value, cardDefinition.line3Right, line3RightValue);
    const imageHidden = isHidden(screenId, value, cardDefinition.image, imageValue);
    const progressHidden = isHidden(screenId, value, cardDefinition.progressBar, progressValue);
    const commonProperties = {
        _id,
        contextType,
        handlersArguments: {
            onClick: [_id, cleanValue],
            rowValue: splitValueToMergedValue(cleanValue),
        },
        screenId,
        parentElementId,
        setFieldValue: () => Promise.resolve(),
    };
    const titleLineDisplayValue = isNewItem
        ? `${titleLineValue} (${localize('@sage/xtrem-ui/new', 'New')})`
        : titleLineValue;
    return (React.createElement(React.Fragment, null,
        React.createElement("div", { className: "e-card-body" },
            cardDefinition.image && !imageHidden && (React.createElement("span", { className: "e-card-image" },
                React.createElement(ConnectedNestedFieldWrapper, { ...commonProperties, columnDefinition: cardDefinition.image, columnName: imageValueId, columnProperties: {
                        ...cardDefinition.image.properties,
                        isReadOnly: true,
                        size: 'medium',
                    }, value: imageValue, nestedReadOnlyField: true }))),
            React.createElement("div", { className: "e-card-content" },
                canSelect && (React.createElement("div", { className: "e-card-check" },
                    React.createElement(Checkbox, { checked: isSelected, value: "checked", onChange: () => onSelect?.(value), disabled: isDisabled }))),
                React.createElement("div", { className: "e-card-row-container" },
                    React.createElement("div", { className: "e-card-row e-card-row-line-title" },
                        React.createElement("span", { className: "e-card-title" },
                            React.createElement(ConnectedNestedFieldWrapper, { ...commonProperties, nestedReadOnlyField: true, columnDefinition: cardDefinition.title, columnName: titleLineValueId, columnProperties: cardDefinition.title.properties, value: titleLineDisplayValue })),
                        cardDefinition.titleRight && !titleRightHidden && (React.createElement("span", { className: "e-card-row-right" },
                            React.createElement(ConnectedNestedFieldWrapper, { ...commonProperties, nestedReadOnlyField: true, columnDefinition: cardDefinition.titleRight, columnName: titleRightValueId, columnProperties: cardDefinition.titleRight.properties, value: titleRightValue })))),
                    React.createElement("div", { className: "e-card-row e-card-row-line-2" },
                        cardDefinition.line2 && !line2Hidden && (React.createElement("div", { className: "e-card-line-2" },
                            React.createElement(ConnectedNestedFieldWrapper, { ...commonProperties, nestedReadOnlyField: true, columnDefinition: cardDefinition.line2, columnName: line2ValueId, columnProperties: cardDefinition.line2.properties, value: line2Value }))),
                        cardDefinition.line2Right && !line2RightHidden && (React.createElement("span", { className: "e-card-row-right" },
                            React.createElement(ConnectedNestedFieldWrapper, { ...commonProperties, nestedReadOnlyField: true, columnDefinition: cardDefinition.line2Right, columnName: line2RightValueId, columnProperties: cardDefinition.line2Right.properties, value: line2RightValue })))),
                    React.createElement("div", { className: "e-card-row e-card-row-line-3" },
                        cardDefinition.line3 && !line3Hidden && (React.createElement("div", { className: "e-card-line-3" },
                            React.createElement(ConnectedNestedFieldWrapper, { ...commonProperties, nestedReadOnlyField: true, columnDefinition: cardDefinition.line3, columnName: line3ValueId, columnProperties: cardDefinition.line3.properties, value: line3Value }))),
                        cardDefinition.line3Right && !line3RightHidden && (React.createElement("span", { className: "e-card-row-right" },
                            React.createElement(ConnectedNestedFieldWrapper, { ...commonProperties, nestedReadOnlyField: true, columnDefinition: cardDefinition.line3Right, columnName: line3RightValueId, columnProperties: cardDefinition.line3Right.properties, value: line3RightValue }))))),
                inlineActions && inlineActions.length > 0 && (React.createElement(ConnectedTableInlineActions, { screenId: screenId, fieldId: parentElementId, recordId: value._id, level: level, rowValue: value, actions: inlineActions })),
                dropdownActions && dropdownActions.length > 0 && (React.createElement(ConnectedTableDropdownActions, { screenId: screenId, fieldId: parentElementId, recordId: value._id, level: level, rowValue: value })),
                onCardClick && parentElementId !== '$navigationPanel' && !isClickIndicatorHidden && (React.createElement("div", { className: "e-card-row-chevron-container" },
                    React.createElement(IconButton, { "aria-label": localize('@sage/xtrem-ui/select-record', 'Select record'), "data-testid": "e-card-row-chevron-right", onAction: () => { } },
                        React.createElement(Icon, { type: "chevron_right", color: tokens.colorsYin090, tooltipMessage: localize('@sage/xtrem-ui/select-record', 'Select record') })))))),
        progressValueId && !progressHidden && cardDefinition.progressBar && (React.createElement("div", { className: "e-card-row-progress-bar-container" },
            React.createElement(ProgressTracker, { length: "100%", progress: progressValue || 0, showDefaultLabels: !cardDefinition.progressBar.properties.areProgressLabelsHidden, currentProgressLabel: cardDefinition.progressBar.properties.areProgressLabelsHidden
                    ? undefined
                    : cardDefinition.progressBar.properties.currentProgressLabel, maxProgressLabel: cardDefinition.progressBar.properties.areProgressLabelsHidden
                    ? undefined
                    : cardDefinition.progressBar.properties.maxProgressLabel })))));
};
export const CardComponent = React.memo((props) => {
    const { buttonRef, canSelect, cardDefinition, contextType, dropdownActions, inlineActions, isClickIndicatorHidden, isDisabled, isInFocus, isNewItem, isSelected, id, onCardClick, onSelect, parentElementId, screenId, value, } = props;
    const wrapperClassName = ['e-card'];
    if (isSelected) {
        wrapperClassName.push('e-card-selected');
    }
    if (cardDefinition.image) {
        wrapperClassName.push('e-card-has-image');
    }
    if (contextType) {
        wrapperClassName.push(`e-context-${contextType}`);
    }
    // We need to set the tab index to -1 to prevent some items from attracting focus.
    const focusTabIndex = isInFocus ? 0 : -1;
    const tabIndex = isInFocus === undefined ? undefined : focusTabIndex;
    const onClickHandler = (event) => {
        if (event.target.type !== 'checkbox' && onCardClick && !isDisabled) {
            onCardClick(value, event.ctrlKey || event.metaKey);
        }
    };
    const sharedProps = {
        className: wrapperClassName.join(' '),
        id,
    };
    if (props.canDrag) {
        sharedProps.draggable = true;
        sharedProps.unselectable = 'on';
        sharedProps.onDragStart = (e) => {
            e.dataTransfer.setData('application/json', JSON.stringify(value));
        };
    }
    const wrapperElement = onCardClick ? (React.createElement("button", { ...sharedProps, onClick: onClickHandler, type: "button", ref: buttonRef, tabIndex: tabIndex, disabled: isDisabled }, renderCard({
        canSelect,
        cardDefinition,
        contextType,
        isDisabled,
        isNewItem,
        isSelected,
        onCardClick,
        onSelect,
        parentElementId,
        inlineActions,
        dropdownActions,
        screenId,
        value,
        isClickIndicatorHidden,
    }))) : (React.createElement("div", { ...sharedProps }, renderCard({
        canSelect,
        cardDefinition,
        contextType,
        isDisabled,
        isNewItem,
        isSelected,
        onSelect,
        onCardClick,
        parentElementId,
        inlineActions,
        dropdownActions,
        screenId,
        value,
        isClickIndicatorHidden,
    })));
    const titleTextValue = getImagePlaceholderValue(value, cardDefinition.title);
    const firstChar = titleTextValue ? String(titleTextValue).charAt(0).toUpperCase() : undefined;
    const separatorContent = firstChar && firstChar !== ' ' ? firstChar : '_';
    return (React.createElement("div", { "data-label": `${titleTextValue}`, "data-testid": "e-card" },
        props.shouldDisplayAlphabeticalDivider ? (React.createElement("div", { className: "e-horizontal-separator" }, separatorContent)) : null,
        wrapperElement));
});
CardComponent.displayName = 'CardComponent';
//# sourceMappingURL=card-component.js.map