import * as React from 'react';
import { connect, useDispatch, useSelector } from 'react-redux';
import { mapDispatchToProps, mapStateToProps } from '../field-base-component';
import ReactFlow, { useReactFlow, ReactFlowProvider, Controls, Background, useNodesState, useEdgesState, } from 'reactflow';
import { CustomNode } from './custom-node';
import { CarbonWrapper } from '../carbon-wrapper';
import { isEqual } from 'lodash';
import { handleChange } from '../../../utils/abstract-fields-utils';
import { useCompare } from '../../../utils/hooks/effects/use-compare';
import { allocateId, changeEventHandler, edgeStyle, removeChildNodesAndEdgesFromStartPoint, removeTransientEdgeDataPropertiesFromEdges, removeTransientNodeDataPropertiesFromNodes, } from './workflow-component-utils';
import { AddWorkflowNodeDialog } from './add-workflow-node-dialog';
import { loadWorkflowNodes } from '../../../redux/actions/workflow-nodes-actions';
import { CustomEdge } from './custom-edge';
import ButtonMinor from 'carbon-react/esm/components/button-minor';
import Typography from 'carbon-react/esm/components/typography';
import Box from 'carbon-react/esm/components/box';
import { isFieldDisabled, isFieldReadOnly } from '../carbon-helpers';
import { WorkflowContext } from './workflow-context-provider';
import { localize } from '../../../service/i18n-service';
import { findAncestorDatasetProperty } from '../../../utils/dom';
import { confirmationDialog } from '../../../service/dialog-service';
const nodeTypes = {
    default: CustomNode,
};
const edgeTypes = {
    default: CustomEdge,
};
export function WorkflowComponent(props) {
    return (React.createElement(ReactFlowProvider, null,
        React.createElement(WorkflowInnerComponent, { ...props })));
}
export function WorkflowInnerComponent(props) {
    const { elementId, fieldProperties, isParentDisabled, screenId, setFieldValue, fixedHeight, validate, value } = props;
    const isDragging = React.useRef(false);
    const [history, setHistory] = React.useState([]);
    const [historyIndex, setHistoryIndex] = React.useState(0);
    const componentRef = React.useRef(null);
    const dispatch = useDispatch();
    const workflowNodes = useSelector(s => s.workflowNodes);
    const hasExternalValueChanged = useCompare(value);
    const connectingNodeId = React.useRef(null);
    const nextNodePosition = React.useRef({});
    const [addDialogFilters, setAddDialogFilters] = React.useState(null);
    const { screenToFlowPosition } = useReactFlow();
    const [nodes, setNodes, onNodesChange] = useNodesState([]);
    const [edges, setEdges, onEdgesChange] = useEdgesState([]);
    const takeHistorySnapshot = React.useCallback(() => {
        const newHistory = history.slice(0, historyIndex);
        setHistory([...newHistory, { nodes, edges }]);
        setHistoryIndex(newHistory.length + 1);
    }, [edges, history, historyIndex, nodes]);
    const onUndo = React.useCallback(() => {
        const historyItem = history[historyIndex - 1];
        if (!historyItem)
            return;
        setNodes(historyItem.nodes);
        setEdges(historyItem.edges);
        setHistoryIndex(historyIndex - 1);
    }, [history, historyIndex, setEdges, setNodes]);
    const onRedo = React.useCallback(() => {
        const historyItem = history[historyIndex];
        if (!historyItem)
            return;
        setNodes(historyItem.nodes);
        setEdges(historyItem.edges);
        setHistoryIndex(historyIndex + 1);
        setHistory([...history].slice(0, historyIndex + 1));
    }, [history, historyIndex, setEdges, setNodes]);
    const isDisabled = React.useMemo(() => {
        return Boolean(isParentDisabled) || isFieldDisabled(screenId, fieldProperties, value, {});
    }, [fieldProperties, isParentDisabled, screenId, value]);
    const isReadOnly = React.useMemo(() => {
        return isFieldReadOnly(screenId, fieldProperties, value, {});
    }, [fieldProperties, screenId, value]);
    React.useEffect(() => {
        if (!workflowNodes) {
            dispatch(loadWorkflowNodes());
        }
    }, [dispatch, workflowNodes]);
    React.useEffect(() => {
        const areNodesDifferent = !isEqual(value?.nodes, removeTransientNodeDataPropertiesFromNodes(nodes));
        if (hasExternalValueChanged && areNodesDifferent) {
            setNodes((value?.nodes || []).map(n => ({
                ...n,
                data: {
                    ...n.data,
                    type: n.type,
                },
            })));
        }
        const areEdgesDifferent = !isEqual(value?.edges, removeTransientEdgeDataPropertiesFromEdges(edges));
        if (hasExternalValueChanged && areEdgesDifferent) {
            setEdges((value?.edges || []).map(e => ({
                ...e,
                ...edgeStyle,
                data: { screenId, elementId },
            })));
        }
        if (hasExternalValueChanged && (areEdgesDifferent || areNodesDifferent)) {
            setHistory([]);
            setHistoryIndex(0);
        }
    }, [value, setNodes, setEdges, elementId, screenId, nodes, edges, hasExternalValueChanged, isReadOnly, isDisabled]);
    const createNewEdge = React.useCallback((targetId, currentEdges) => {
        if (!connectingNodeId.current) {
            throw new Error('No source node found.');
        }
        return {
            id: allocateId(`${connectingNodeId.current.nodeId}--${connectingNodeId.current.handleId}`, currentEdges),
            source: connectingNodeId.current.nodeId,
            sourceHandle: connectingNodeId.current.handleId,
            target: targetId,
            data: { screenId, elementId },
            ...edgeStyle,
        };
    }, [elementId, screenId]);
    const onAddNodeDialogClose = React.useCallback(() => {
        connectingNodeId.current = null;
        setAddDialogFilters(null);
    }, []);
    const onChange = React.useCallback((newValue = { edges, nodes }) => {
        if (hasExternalValueChanged) {
            return;
        }
        const cleanNodes = removeTransientNodeDataPropertiesFromNodes(newValue.nodes);
        const cleanEdges = removeTransientEdgeDataPropertiesFromEdges(newValue.edges);
        if (!isEqual(cleanNodes, value?.nodes) || !isEqual(cleanEdges, value?.edges)) {
            handleChange(elementId, { nodes: cleanNodes, edges: cleanEdges }, setFieldValue, validate, changeEventHandler(screenId, elementId));
        }
    }, [
        edges,
        nodes,
        hasExternalValueChanged,
        value?.nodes,
        value?.edges,
        elementId,
        setFieldValue,
        validate,
        screenId,
    ]);
    const onNodeDataChange = React.useCallback((nodeId, data) => {
        takeHistorySnapshot();
        const newNodes = [...nodes];
        const index = newNodes.findIndex(n => n.id === nodeId);
        newNodes[index] = {
            ...newNodes[index],
            data: { ...data, outputVariables: data.outputVariables || [] },
        };
        setNodes(newNodes);
    }, [nodes, setNodes, takeHistorySnapshot]);
    const onClick = React.useCallback(() => {
        if (componentRef.current && componentRef.current.contains(document.activeElement)) {
            return;
        }
        onChange();
    }, [onChange]);
    const onConnectStart = React.useCallback((_, { nodeId, handleId }) => {
        if (nodeId && handleId) {
            connectingNodeId.current = { nodeId, handleId };
        }
    }, []);
    const onConnectEnd = React.useCallback((event) => {
        if (!connectingNodeId.current)
            return;
        const target = event.target;
        const targetIsPane = target?.classList?.contains('react-flow__pane');
        if (targetIsPane) {
            nextNodePosition.current = {
                clientX: event.clientX,
                clientY: event.clientY,
            };
            setAddDialogFilters(['condition', 'action']);
            return;
        }
        const nodeId = findAncestorDatasetProperty(target, 'nodeid');
        if (nodeId) {
            takeHistorySnapshot();
            onChange({
                edges: [...edges, createNewEdge(nodeId, edges)],
                nodes,
            });
        }
    }, [createNewEdge, edges, nodes, onChange, takeHistorySnapshot]);
    const onEdgesChangeInternal = change => {
        if (!isDisabled && !isReadOnly) {
            onEdgesChange(change.filter(e => e.type !== 'remove'));
        }
    };
    const handleRemovalChange = React.useCallback(async (change) => {
        const connectingEdges = edges.filter(e => e.source === change.id);
        if (connectingEdges.length > 0) {
            try {
                await confirmationDialog(props.screenId, 'info', localize('@sage/xtrem-ui/workflow-delete-node-chain-title', 'Delete step flow'), localize('@sage/xtrem-ui/workflow-delete-node-chain-message', 'If you remove this step, any subsequent steps with no other links are also removed.'));
            }
            catch (e) {
                // eslint-disable-next-line no-useless-return
                return;
            }
        }
        takeHistorySnapshot();
        let newEdges = [...edges];
        const newNodes = [...nodes];
        removeChildNodesAndEdgesFromStartPoint(change.id, newEdges, newNodes);
        const remainingIds = newNodes.map(n => n.id);
        newEdges = newEdges.filter(e => remainingIds.includes(e.source) && remainingIds.includes(e.target));
        setNodes(newNodes);
        setEdges(newEdges);
    }, [edges, nodes, props.screenId, setEdges, setNodes, takeHistorySnapshot]);
    const handlePositionChange = React.useCallback((change) => {
        // We need to take a history snapshot only when the user starts dragging a node
        if (!isDragging.current && change.dragging) {
            isDragging.current = true;
            takeHistorySnapshot();
        }
        else if (isDragging.current && !change.dragging) {
            isDragging.current = false;
        }
        onNodesChange([change]);
    }, [onNodesChange, takeHistorySnapshot]);
    const onNodesChangeInternal = React.useCallback(async (change) => {
        if (!isDisabled && !isReadOnly) {
            const removalChange = change.find(c => c.type === 'remove');
            if (removalChange) {
                await handleRemovalChange(removalChange);
                return;
            }
            const positionChange = change.find(c => c.type === 'position');
            if (positionChange) {
                await handlePositionChange(positionChange);
                return;
            }
            onNodesChange(change);
        }
    }, [handlePositionChange, handleRemovalChange, isDisabled, isReadOnly, onNodesChange]);
    const onAddFirstElement = React.useCallback(() => {
        setAddDialogFilters(['event']);
    }, []);
    const onNewNodeAdded = React.useCallback(({ selectedNodeType, values }) => {
        const newNode = {
            id: allocateId(selectedNodeType, nodes),
            position: nextNodePosition.current
                ? screenToFlowPosition({
                    x: (nextNodePosition.current.clientX ?? 170) - 150,
                    y: nextNodePosition.current.clientY ?? 20,
                })
                : {
                    x: 20,
                    y: 20,
                },
            type: selectedNodeType,
            data: {
                type: selectedNodeType,
                ...values,
            },
            origin: [0.5, 0.0],
        };
        const newNodes = [...nodes, newNode];
        const newEdges = [...edges];
        if (connectingNodeId.current) {
            newEdges.push(createNewEdge(newNode.id, edges));
        }
        setNodes(newNodes);
        setEdges(newEdges);
        setAddDialogFilters(null);
    }, [nodes, screenToFlowPosition, edges, setNodes, setEdges, createNewEdge]);
    React.useEffect(() => {
        window.addEventListener('click', onClick);
        return () => {
            window.removeEventListener('click', onClick);
        };
    }, [onClick]);
    const workflowContext = React.useMemo(() => ({ screenId, elementId, onNodeDataChange, isReadOnly: isReadOnly || isDisabled }), [screenId, elementId, onNodeDataChange, isReadOnly, isDisabled]);
    const height = fixedHeight ? fixedHeight - 80 : '400px';
    return (React.createElement(React.Fragment, null,
        React.createElement(CarbonWrapper, { ...props, className: "e-workflow-field", componentName: "workflow", noReadOnlySupport: true, value: !!value || false },
            nodes.length === 0 && (React.createElement("div", { className: "e-workflow-empty", style: { height } },
                React.createElement(Box, { paddingY: 80, gap: 1, display: "flex", flexDirection: "column", alignItems: "center", justifyContent: "center" },
                    React.createElement("div", null,
                        React.createElement(Typography, { color: "--colorsUtilityMajor100", lineHeight: "30px", variant: "h3" }, localize('@sage/xtrem-ui/workflow-empty', 'This workflow is currently empty'))),
                    !isReadOnly && !isDisabled && (React.createElement(ButtonMinor, { mt: 2, iconType: "add", onClick: onAddFirstElement, "data-testid": "add-item-button" }, localize('@sage/xtrem-ui/workflow-add-trigger-event', 'Add a trigger event')))))),
            nodes.length !== 0 && (React.createElement("div", { className: "e-workflow-wrapper", style: { height }, ref: componentRef },
                React.createElement(WorkflowContext.Provider, { value: workflowContext },
                    React.createElement(ReactFlow, { nodesDraggable: !isReadOnly && !isDisabled, nodesConnectable: !isReadOnly && !isDisabled, elementsSelectable: !isReadOnly && !isDisabled, nodes: nodes, edges: edges, snapToGrid: true, snapGrid: [20, 20], onConnectStart: onConnectStart, onConnectEnd: onConnectEnd, onNodesChange: onNodesChangeInternal, onEdgesChange: onEdgesChangeInternal, nodeTypes: nodeTypes, edgeTypes: edgeTypes, deleteKeyCode: ['Backspace', 'Delete'] },
                        React.createElement(Background, { color: "#004455" }),
                        React.createElement(Controls, { showInteractive: false },
                            React.createElement("button", { onClick: onUndo, type: "button", disabled: historyIndex === 0, className: "react-flow__controls-button", title: localize('@sage/xtrem-ui/workflow-undo', 'Undo'), "aria-label": localize('@sage/xtrem-ui/workflow-undo', 'Undo'), "data-testid": "e-workflow-undo-button" },
                                React.createElement("svg", { xmlns: "http://www.w3.org/2000/svg", viewBox: "0 0 32 30" },
                                    React.createElement("g", { transform: "scale(0.031 0.031)" },
                                        React.createElement("path", { d: "M312.435 641.319c9.706 9.706 22.391 14.547 35.101 14.547s25.395-4.841 35.101-14.547c19.388-19.388 19.388-50.815 0-70.203l-163.492-163.492h401.458c123.203 0 223.418 100.215 223.418 223.418s-100.215 223.418-223.418 223.418h-198.594c-27.406 0-49.648 22.243-49.648 49.648s22.243 49.648 49.648 49.648h198.594c177.94 0 322.715-144.775 322.715-322.715s-144.775-322.715-322.715-322.715h-401.458l163.492-163.492c19.388-19.388 19.388-50.815 0-70.203s-50.815-19.388-70.203 0l-248.242 248.242c-19.388 19.388-19.388 50.815 0 70.203l248.242 248.242z" })))),
                            React.createElement("button", { type: "button", onClick: onRedo, disabled: historyIndex === history.length, className: "react-flow__controls-button", title: localize('@sage/xtrem-ui/workflow-redo', 'Redo'), "aria-label": localize('@sage/xtrem-ui/workflow-redo', 'Redo'), "data-testid": "e-workflow-redo-button" },
                                React.createElement("svg", { xmlns: "http://www.w3.org/2000/svg", viewBox: "0 0 32 30" },
                                    React.createElement("g", { transform: "scale(0.031 0.031)" },
                                        React.createElement("path", { d: "M372.367 953.757h186.182c27.406 0 49.648-22.243 49.648-49.648s-22.243-49.648-49.648-49.648h-186.182c-123.203 0-223.418-100.215-223.418-223.418s100.215-223.418 223.418-223.418h401.458l-163.492 163.492c-19.388 19.388-19.388 50.815 0 70.203 9.706 9.706 22.391 14.547 35.101 14.547s25.395-4.841 35.101-14.547l248.242-248.242c19.388-19.388 19.388-50.815 0-70.203l-248.242-248.242c-19.388-19.388-50.815-19.388-70.203 0s-19.388 50.815 0 70.203l163.492 163.492h-401.458c-177.94 0-322.715 144.774-322.715 322.715s144.775 322.715 322.715 322.715z" })))))))))),
        React.createElement(AddWorkflowNodeDialog, { isOpen: !!addDialogFilters, onClose: onAddNodeDialogClose, onConfirm: onNewNodeAdded, elementId: props.elementId, screenId: props.screenId, previousNodeId: connectingNodeId.current?.nodeId, filterType: addDialogFilters || [] })));
}
export const ConnectedFormDesignerComponent = connect(mapStateToProps(), mapDispatchToProps())(WorkflowComponent);
export default ConnectedFormDesignerComponent;
//# sourceMappingURL=workflow-component.js.map