import { AgGridReact } from '@ag-grid-community/react';
import { EnterpriseCoreModule } from '@ag-grid-enterprise/core';
import { ExcelExportModule } from '@ag-grid-enterprise/excel-export';
import { CsvExportModule } from '@ag-grid-community/csv-export';
import { MenuModule } from '@ag-grid-enterprise/menu';
import { RichSelectModule } from '@ag-grid-enterprise/rich-select';
import { RowGroupingModule } from '@ag-grid-enterprise/row-grouping';
import { ServerSideRowModelModule } from '@ag-grid-enterprise/server-side-row-model';
import { SetFilterModule } from '@ag-grid-enterprise/set-filter';
import { isValidDatePropertyValue } from '@sage/xtrem-date-time';
import { flat } from '@sage/xtrem-shared';
import { difference, differenceBy, get, isEqual, isNil, set, sortBy, uniq, xorBy } from 'lodash';
import * as React from 'react';
import { renderToStaticMarkup } from 'react-dom/server';
import { connect } from 'react-redux';
import * as xtremRedux from '../../../redux';
import { ActionType } from '../../../redux';
import { subscribeToActions } from '../../../redux/middleware/action-subscription-middleware';
import { setAgGridLicence } from '../../../service/ag-grid-licence-service';
import { RecordActionType } from '../../../service/collection-data-types';
import { getGroupKey } from '../../../service/collection-data-utils';
import { localize } from '../../../service/i18n-service';
import { getScreenElement } from '../../../service/screen-base-definition';
import { ContextType } from '../../../types';
import { findColumnDefinitionByBind, getNestedFieldElementId } from '../../../utils/abstract-fields-utils';
import { COLUMN_ID_AUTO_COLUMN, COLUMN_ID_ROW_ACTIONS, COLUMN_ID_ROW_SELECTION, COLUMN_ID_VALIDATIONS, frameworkComponents, } from '../../../utils/ag-grid/ag-grid-column-config';
import { isNoConsole, xtremConsole } from '../../../utils/console';
import { isRowValueInDateGroup } from '../../../utils/date-utils';
import { isChildOfElementWithAttributeValue, isChildOfElementWithClass } from '../../../utils/dom';
import { triggerFieldEvent } from '../../../utils/events';
import { convertDeepBindToPath, convertDeepBindToPathNotNull } from '../../../utils/nested-field-utils';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { getPageDefinitionFromState, getPagePropertiesFromPageDefinition } from '../../../utils/state-utils';
import { splitValueToMergedValue } from '../../../utils/transformers';
import { navigationPanelId } from '../../container/navigation-panel/navigation-panel-types';
import { FieldKey, SET } from '../../types';
import { AsyncCalendarBodyComponent } from '../../ui/calendar-body/async-calendar-body-component';
import { Icon } from '../../ui/icon/icon-component';
import { DesktopTableHeaderComponent } from '../../ui/table-shared/desktop-table-header';
import { generateFieldId, getFieldTitle, isFieldDisabled, isFieldReadOnly } from '../carbon-helpers';
import { getReferenceValueField } from '../reference/reference-utils';
import { GroupState } from './table-component-types';
import { AUTO_COLUMN_ID, getActiveOptionsMenu, getCardDefinitionFromColumns, getOrderByFromSortModel, getTableContext, getTableViewColumnHidden, getTableViewFilter, getTableViewGrouping, getTableViewSortOrder, isColumnBindVariant, setTableContext, } from '../../../utils/table-component-utils';
import { TableConfigurationDialog } from '../../ui/table-shared/table-configuration-dialog/table-configuration-dialog';
import { changeGroupIcon, processGroupSelection, updateGroupState } from '../../ui/table-shared/table-selection-helper';
import { DesktopTableBulkActionBar } from '../../ui/table-shared/desktop-table-bulk-action-bar';
import { TableLoadingCellRenderer } from '../../ui/table-shared/cell/table-loading-cell-renderer';
import { callGridMethod, getColumnStatesForColumnPanel, getFilterModel, getFirstEditableColumn, getSafeGridApiContext, mapAgGridFilterToXtremFilters, ROW_HEIGHT, tryToCommitPhantomRow, } from '../../../utils/ag-grid/ag-grid-table-utils';
import { getColumns } from '../../../utils/ag-grid/ag-grid-service';
import { onRowClick } from '../../../utils/ag-grid/ag-grid-event-handlers';
import { cellExportFormatter, getColumnsToExport, getRowClassRules, } from '../../../utils/ag-grid/ag-grid-cell-editor-utils';
import localeText from '../../../utils/ag-grid/ag-grid-strings';
import { isDevMode } from '../../../utils/window';
const activatedAgGridModules = [
    ServerSideRowModelModule,
    EnterpriseCoreModule,
    MenuModule,
    RichSelectModule,
    SetFilterModule,
    RowGroupingModule,
    ExcelExportModule,
    CsvExportModule,
];
var DataOperationMode;
(function (DataOperationMode) {
    DataOperationMode[DataOperationMode["NONE"] = 0] = "NONE";
    DataOperationMode[DataOperationMode["FETCH_PAGE"] = 1] = "FETCH_PAGE";
    DataOperationMode[DataOperationMode["REMOVE_RECORD"] = 2] = "REMOVE_RECORD";
    DataOperationMode[DataOperationMode["RESET_TABLE"] = 3] = "RESET_TABLE";
    DataOperationMode[DataOperationMode["FILTER_ERRORS"] = 4] = "FILTER_ERRORS";
})(DataOperationMode || (DataOperationMode = {}));
export class DesktopTableComponent extends React.Component {
    constructor(props) {
        super(props);
        this.gridApi = null;
        this.gridId = generateFieldId({
            screenId: this.props.screenId,
            elementId: this.props.elementId,
            fieldProperties: this.props.fieldProperties,
            isNested: false,
        });
        this.dataOperationMode = DataOperationMode.NONE;
        this.autoSizeStrategy = {
            type: 'fitCellContents',
        };
        this.isFetching = false;
        this.isFilteringErrors = false;
        this.newDataSelected = false;
        this.containerRef = React.createRef();
        this.defaultColDef = { flex: 1, resizable: false };
        this.loadingCellRendererParams = {
            elementId: this.props.elementId,
        };
        this.loadingOverlayComponentParams = { size: 'large' };
        this.getRowNode = ({ id, gridApi = this.gridApi, }) => {
            let node = callGridMethod(gridApi, 'getRowNode', id);
            if (node) {
                return node;
            }
            node = gridApi?.getPinnedTopRow(0);
            if (node?.data?._id === id) {
                return node;
            }
            return undefined;
        };
        this.focusPhantomRowAndStartEditing = () => {
            setTimeout(async () => {
                if (!this.gridApi) {
                    return;
                }
                const phantomRecords = await this.props.value.getPhantomRecords();
                const colKey = getFirstEditableColumn(this.gridApi, phantomRecords?.[0]);
                if (colKey === undefined) {
                    return;
                }
                callGridMethod(this.gridApi, 'setFocusedCell', 0, colKey, 'top');
                callGridMethod(this.gridApi, 'ensureColumnVisible', colKey);
                callGridMethod(this.gridApi, 'startEditingCell', {
                    colKey,
                    rowIndex: 0,
                    key: 'Enter',
                    rowPinned: 'top',
                });
            }, 50);
        };
        this.onCollectionUpdated = (type, rowValue) => {
            if (this.gridApi) {
                if (type === RecordActionType.MODIFIED) {
                    if (rowValue?.__phantom && rowValue?.__forceRowUpdate) {
                        delete rowValue.__forceRowUpdate;
                        this.setState({ phantomRows: [rowValue] });
                    }
                    else {
                        // In this case we target one specific row in order to reduce the amplitude of rendering
                        this.getRowNode({ id: rowValue._id })?.setData(rowValue);
                    }
                }
                else if (rowValue?.__phantom !== undefined && type === RecordActionType.ADDED) {
                    this.setState({ phantomRows: [rowValue] }, this.focusPhantomRowAndStartEditing);
                }
                else {
                    this.dataOperationMode =
                        type === RecordActionType.ADDED ? DataOperationMode.FETCH_PAGE : DataOperationMode.REMOVE_RECORD;
                    const gridApi = this.gridApi;
                    setTimeout(() => callGridMethod(gridApi, 'refreshServerSide', { purge: true }), 0);
                }
            }
        };
        this.resetOptionsMenu = () => {
            const optionsMenu = resolveByValue({
                propertyValue: this.props.fieldProperties.optionsMenu,
                rowValue: null,
                screenId: this.props.screenId,
                fieldValue: null,
                skipHexFormat: true,
            });
            this.props.setTableViewOptionsMenuItemAndViewFilter?.(0, optionsMenu?.[0], {});
        };
        /** Marks items selected on the table, this function is triggered by changes initiated by a pagination or reset
         * event
         **/
        this.selectTableItems = ({ api, data }) => {
            // Marking user selected rows, selected in the new dataset
            const selectedRecords = this.props.fieldProperties.selectedRecords || [];
            const selectedRows = [];
            selectedRecords.forEach(id => {
                const rowNode = this.getRowNode({ id, gridApi: api });
                if (rowNode) {
                    rowNode.setSelected(true);
                    selectedRows.push(rowNode.data);
                    if (this.props.lookupSelectionMode === 'single' && this.gridApi) {
                        setTimeout(() => {
                            const columns = callGridMethod(this.gridApi, 'getColumns');
                            if (columns) {
                                // Move focus to first visible cell
                                callGridMethod(api, 'setFocusedCell', rowNode.rowIndex, columns.filter(c => c.isVisible())[0], rowNode.data?.__phantom ? 'top' : undefined);
                            }
                        }, 500);
                    }
                }
            });
            this.updateSelectedRows(selectedRows);
            setTimeout(() => {
                let someSelected = false;
                if (data !== undefined && getTableContext(api)?.isSelectAllEnabled) {
                    data.forEach(row => {
                        const node = this.getRowNode({ id: row._id, gridApi: api });
                        if (node) {
                            node.setSelected(true, false, 'checkboxSelected');
                            someSelected = true;
                        }
                    });
                    if (someSelected) {
                        this.newDataSelected = true;
                    }
                }
            }, 10);
        };
        this.insertDataSourceIntoTable = (getRowsParams, data, lastRow) => {
            getRowsParams.success({ rowData: data, rowCount: lastRow });
            callGridMethod(getRowsParams.api, 'hideOverlay');
            const isEmptyTable = getRowsParams.request.groupKeys?.length === 0 && lastRow === 0;
            setTimeout(() => {
                if (isEmptyTable) {
                    callGridMethod(getRowsParams.api, 'showNoRowsOverlay');
                }
                else {
                    callGridMethod(getRowsParams.api, 'hideOverlay');
                }
            }, 100);
            if (lastRow && lastRow !== -1) {
                triggerFieldEvent(this.props.screenId, this.props.elementId, 'onAllDataLoaded');
            }
            else {
                triggerFieldEvent(this.props.screenId, this.props.elementId, 'onDataLoaded');
            }
        };
        this.updateColumnValidation = (globalValidationState) => {
            const ctx = this.gridApi ? getTableContext(this.gridApi) : undefined;
            if (this.gridApi && ctx) {
                // Refresh column's header
                const invalidColumns = Object.keys(globalValidationState?.[0] || {});
                const headerClasses = invalidColumns.reduce((acc, column) => {
                    acc[column] = false;
                    return acc;
                }, {});
                setTableContext(this.gridApi, c => {
                    c.headerClasses = headerClasses;
                });
                this.gridApi
                    .getAllGridColumns()
                    .map(c => c.getColDef().field)
                    .forEach(k => {
                    if (!k) {
                        return;
                    }
                    if (invalidColumns.includes(k)) {
                        document
                            .querySelectorAll(`[grid-id="${this.gridId}"] .ag-header-cell[col-id="${k}"]`)
                            .forEach(element => {
                            element.classList.add('e-table-field-header-column-error');
                        });
                    }
                    else {
                        document
                            .querySelectorAll(`[grid-id="${this.gridId}"] .ag-header-cell[col-id="${k}"]`)
                            .forEach(element => {
                            element.classList.remove('e-table-field-header-column-error');
                        });
                    }
                });
                // Display/hide validation column
                getSafeGridApiContext(column => {
                    const isValidationColumnVisible = Boolean(column?.isVisible());
                    const shouldValidationColumnBeVisible = invalidColumns.length > 0;
                    if (isValidationColumnVisible !== shouldValidationColumnBeVisible) {
                        callGridMethod(this.gridApi, 'setColumnVisible', COLUMN_ID_VALIDATIONS, shouldValidationColumnBeVisible);
                        this.autoSizeAllColumns();
                    }
                }, this.gridApi, 'getColumn', COLUMN_ID_VALIDATIONS);
            }
        };
        this.getUnknownLastRow = () => {
            return this.isInfiniteScroll() ? -1 : undefined;
        };
        this.getRowCount = ({ data, getRowsParams, }) => {
            return data.length < (getRowsParams.request.endRow || 0) - (getRowsParams.request.startRow || 0)
                ? (getRowsParams.request.startRow || 0) + data.length
                : this.getUnknownLastRow();
        };
        this.isNavigationPanel = () => this.props.elementId === navigationPanelId;
        /**
         * Ag-grid callback which is triggered whenever the table component needs new data from our framework
         * @param getRowsParams
         */
        this.getRows = async (getRowsParams) => {
            const hasGroups = getRowsParams.request.rowGroupCols.length > 0;
            /**
             * Set flag to disable row field selection so events will not be triggered when rows selected
             * programmatically.
             */
            const pageSize = this.props.fieldProperties.pageSize || 20;
            const pageNumber = Math.max(0, Math.ceil((getRowsParams.request.endRow || 0) / pageSize) - 1);
            callGridMethod(getRowsParams.api, 'hideOverlay');
            if (hasGroups) {
                callGridMethod(getRowsParams.api, 'showLoadingOverlay');
                const colId = getRowsParams.request.rowGroupCols[0].id;
                const column = findColumnDefinitionByBind(this.props.fieldProperties.columns || [], colId);
                const groupKey = column?.type === FieldKey.Reference
                    ? `${convertDeepBindToPathNotNull(column?.properties.bind)}.${getReferenceValueField(column?.properties)}`
                    : convertDeepBindToPathNotNull(column?.properties.bind);
                const groupValue = column?.type === FieldKey.Reference
                    ? (get(getRowsParams.parentNode.data, getGroupKey(groupKey)) ??
                        getRowsParams.request.groupKeys?.[0])
                    : getRowsParams.request.groupKeys?.[0];
                let cursor;
                const nodes = [];
                if (!isNil(groupValue)) {
                    const belongsToGroup = getRowsParams.request.rowGroupCols[0]?.aggFunc === undefined
                        ? (n) => get(n.data, getGroupKey(groupKey), null) === (groupValue === '' ? null : groupValue)
                        : (n) => {
                            const aggFunc = getRowsParams.request.rowGroupCols[0]?.aggFunc;
                            const rowValue = get(n.data, groupKey);
                            if (groupValue === '') {
                                return rowValue == null;
                            }
                            if (!isValidDatePropertyValue(rowValue)) {
                                return false;
                            }
                            return isRowValueInDateGroup(rowValue, groupValue, aggFunc);
                        };
                    callGridMethod(getRowsParams.api, 'forEachNode', n => {
                        if (!n.group && belongsToGroup(n)) {
                            nodes.push(n);
                        }
                    });
                    cursor = nodes[(getRowsParams.request.startRow ?? 1) - 1]?.data?.__cursor;
                    if (cursor === undefined && getRowsParams.request.startRow !== 0) {
                        getRowsParams.success({
                            rowData: [],
                            rowCount: nodes.length,
                        });
                        callGridMethod(getRowsParams.api, 'showNoRowsOverlay');
                        return;
                    }
                }
                else {
                    callGridMethod(getRowsParams.api, 'forEachNode', n => {
                        if (n.group && n.data) {
                            nodes.push(n);
                        }
                    });
                    cursor = nodes[(getRowsParams.request.startRow ?? 1) - 1]?.data?.__cursor;
                    if (cursor === undefined && getRowsParams.request.startRow !== 0) {
                        getRowsParams.success({
                            rowData: nodes,
                            rowCount: nodes.length,
                        });
                        return;
                    }
                }
                const sortOrder = getOrderByFromSortModel(getRowsParams.request.sortModel, this.props.fieldProperties.columns || [], this.state.autoGroupColumnDef);
                const orderBy = groupValue !== undefined
                    ? sortOrder
                    : Object.keys(flat(sortOrder)).reduce((acc, key) => {
                        if (key === groupKey) {
                            set(acc, key, get(sortOrder, key));
                        }
                        return acc;
                    }, {});
                // Map Ag-grid filter objects into GraphQL/MongoDB friendly filter data structure
                const userFilters = this.getFilters(getFilterModel(getRowsParams.api, this.state.groupByColumn?.field));
                const gridPageSize = (getRowsParams.request.endRow || 0) - (getRowsParams.request.startRow || 0);
                const groupedData = await this.props.value.getPage({
                    tableFieldProperties: this.props.fieldProperties,
                    filters: userFilters,
                    orderBy,
                    pageSize: gridPageSize,
                    group: {
                        key: groupKey,
                        value: groupValue,
                        aggFunc: getRowsParams.request.rowGroupCols[0]?.aggFunc,
                    },
                    cursor,
                    cleanMetadata: false,
                    pageNumber: cursor ? pageNumber : undefined,
                    selectedOptionsMenuItem: this.props.selectedOptionsMenuItem,
                });
                if (groupedData[0]?.__groupCount) {
                    const selectedGroupState = groupedData.reduce((acc, data) => {
                        const group = {
                            currentSelected: 0,
                            state: GroupState.NOT_SELECTED,
                            total: data.__groupCount,
                            aggFunc: getRowsParams.request.rowGroupCols[0]?.aggFunc,
                        };
                        if (isNil(acc[data.__groupKey])) {
                            acc[data.__groupKey] = {};
                        }
                        acc[data.__groupKey][get(data, data.__groupKey)] = group;
                        return acc;
                    }, {});
                    this.setState({ selectedGroupState });
                }
                const rowCount = this.getRowCount({ data: groupedData, getRowsParams });
                getRowsParams.success({
                    rowData: groupedData,
                    rowCount,
                });
                const selectedRows = [];
                callGridMethod(getRowsParams.api, 'forEachNode', (node) => {
                    if (node.parent?.isSelected() && groupedData.find(d => d._id === node.data._id)) {
                        node.setSelected(true);
                        selectedRows.push(node.data);
                    }
                });
                callGridMethod(getRowsParams.api, 'hideOverlay');
                this.updateSelectedRows(selectedRows);
                return;
            }
            if (!this.props.value) {
                this.insertDataSourceIntoTable(getRowsParams, [], 0);
                return;
            }
            if (!this.isInfiniteScroll()) {
                callGridMethod(getRowsParams.api, 'showLoadingOverlay');
            }
            try {
                switch (this.dataOperationMode) {
                    case DataOperationMode.NONE:
                        const data = this.props.value.getData({
                            cleanMetadata: false,
                            temporaryRecords: this.props.additionalLookupRecords?.(),
                            limit: this.props.fieldProperties.pageSize || 20,
                        });
                        this.insertDataSourceIntoTable(getRowsParams, data, this.getRowCount({ data, getRowsParams }));
                        this.collectionValueChangeSubscription = this.props.value.subscribeForValueChanges(this.onCollectionUpdated);
                        this.collectionValidityChangeSubscription = this.props.value.subscribeForValidityChanges(({ globalValidationState, recordValidationState, recordId }) => {
                            if (this.gridApi) {
                                this.updateColumnValidation(globalValidationState);
                                const rowNode = this.getRowNode({ id: recordId });
                                if (rowNode) {
                                    const previousValidationStateKeys = Object.keys(rowNode.data.__validationState || {});
                                    rowNode.setData({
                                        ...rowNode.data,
                                        __validationState: recordValidationState,
                                    });
                                    // Refresh cells
                                    const columns = difference(Object.keys(recordValidationState), previousValidationStateKeys);
                                    this.gridApi?.refreshCells({
                                        rowNodes: [rowNode],
                                        columns,
                                        force: true,
                                        suppressFlash: true,
                                    });
                                }
                            }
                        });
                        this.selectTableItems({ api: getRowsParams.api, data });
                        break;
                    case DataOperationMode.FETCH_PAGE:
                        this.isFetching = true;
                        const newData = await this.props.value.getPageWithCurrentQueryArguments({
                            tableFieldProperties: this.props.fieldProperties,
                            pageSize,
                            pageNumber,
                            group: undefined,
                            cleanMetadata: false,
                            cursor: this.getCursor(getRowsParams.api),
                        });
                        this.updateColumnValidation(this.props.value.getValidationStateByColumn());
                        this.insertDataSourceIntoTable(getRowsParams, newData, this.getRowCount({ data: newData, getRowsParams }));
                        this.selectTableItems({ api: getRowsParams.api, data: newData });
                        this.isFetching = false;
                        break;
                    case DataOperationMode.REMOVE_RECORD:
                        callGridMethod(getRowsParams.api, 'deselectAll');
                        // Populating values
                        const mergedData = await this.props.value.getRecordWithCurrentQueryArguments({
                            tableFieldProperties: this.props.fieldProperties,
                            pageSize,
                            pageNumber,
                            group: undefined,
                            cursor: this.getCursor(getRowsParams.api),
                            cleanMetadata: false,
                        });
                        this.insertDataSourceIntoTable(getRowsParams, mergedData, this.getRowCount({ data: mergedData, getRowsParams }));
                        this.selectTableItems({ api: getRowsParams.api, data: mergedData });
                        this.updateColumnValidation(this.props.value.getValidationStateByColumn());
                        break;
                    case DataOperationMode.RESET_TABLE:
                        callGridMethod(getRowsParams.api, 'deselectAll');
                        if (this.isNavigationPanel()) {
                            this.props.value.cleanCollectionData();
                            callGridMethod(this.gridApi, 'setGridOption', 'serverSideDatasource', this.serverSideDataSource);
                        }
                        // Map Ag-grid sorting properties
                        const orderBy = getOrderByFromSortModel(getRowsParams.request.sortModel, this.props.fieldProperties.columns || [], this.state.autoGroupColumnDef);
                        // Map Ag-grid filter objects into GraphQL/MongoDB friendly filter data structure
                        const userFilters = this.getFilters(getFilterModel(getRowsParams.api, this.state.groupByColumn?.field));
                        // Map Ag-grid page sizing properties
                        const gridPageSize = (getRowsParams.request.endRow || 0) - (getRowsParams.request.startRow || 0);
                        // Populating values
                        const pageData = await this.props.value.getPage({
                            tableFieldProperties: this.props.fieldProperties,
                            filters: userFilters,
                            orderBy,
                            pageSize: gridPageSize,
                            group: undefined,
                            cursor: this.getCursor(getRowsParams.api),
                            cleanMetadata: false,
                            pageNumber: !this.getCursor(getRowsParams.api) ? pageNumber : undefined,
                            selectedOptionsMenuItem: this.props.selectedOptionsMenuItem,
                        });
                        this.insertDataSourceIntoTable(getRowsParams, pageData, this.getRowCount({ data: pageData, getRowsParams }));
                        this.selectTableItems({ api: getRowsParams.api, data: pageData });
                        callGridMethod(this.gridApi, 'paginationGoToFirstPage');
                        break;
                    case DataOperationMode.FILTER_ERRORS:
                        if (this.props.value.getAllInvalidRecords().length === 0) {
                            this.insertDataSourceIntoTable(getRowsParams, [], 0);
                        }
                        else if (this.props.fieldProperties.isTransient) {
                            this.insertDataSourceIntoTable(getRowsParams, this.props.value.getAllInvalidRecords(), this.getRowCount({ data: this.props.value.getAllInvalidRecords(), getRowsParams }));
                        }
                        else {
                            await this.props.value.fetchInvalidUnloadedRecords();
                            const errorPageData = await this.props.value.getPage({
                                tableFieldProperties: this.props.fieldProperties,
                                filters: this.getFilters(getFilterModel(getRowsParams.api, this.state.groupByColumn?.field)),
                                orderBy: getOrderByFromSortModel(getRowsParams.request.sortModel, this.props.fieldProperties.columns || [], this.state.autoGroupColumnDef),
                                pageSize: (getRowsParams.request.endRow || 0) - (getRowsParams.request.startRow || 0),
                                group: undefined,
                                cursor: this.getCursor(getRowsParams.api),
                                cleanMetadata: false,
                                fetchPageSize: 500,
                            });
                            this.insertDataSourceIntoTable(getRowsParams, errorPageData, this.getRowCount({ data: errorPageData, getRowsParams }));
                        }
                        break;
                    default:
                        break;
                }
                if (this.props.value.getData({
                    cleanMetadata: false,
                    temporaryRecords: this.props.additionalLookupRecords?.(),
                }).length > 0) {
                    this.setState({ tableHasData: true });
                }
                else {
                    this.setState({ tableHasData: false });
                }
                this.dataOperationMode = DataOperationMode.FETCH_PAGE;
            }
            catch (error) {
                xtremConsole.log(error);
                getRowsParams.fail();
            }
            finally {
                await this.setPhantomRow();
                this.adjustTableHeight({ api: getRowsParams.api });
            }
        };
        this.serverSideDataSource = {
            getRows: this.getRows,
        };
        this.isDisabled = () => {
            const x = isFieldDisabled(this.props.screenId, this.props.fieldProperties, this.props.recordContext?._id || null, this.props.recordContext || null);
            return this.props.isParentDisabled || x;
        };
        this.isReadOnly = () => resolveByValue({
            fieldValue: null,
            propertyValue: this.props.isReadOnly,
            skipHexFormat: true,
            rowValue: null,
            screenId: this.props.screenId,
        }) || isFieldReadOnly(this.props.screenId, this.props.fieldProperties, null, undefined);
        this.resizeListener = () => {
            this.autoSizeAllColumns();
        };
        /**
         * Ag-grid lifecycle event listener. It is triggered when the table is prepared with the initial rendering of the
         * controller components and is ready to receive data.
         **/
        this.onGridReady = async (params) => {
            this.gridApi = params.api;
            // Timeout needed in order to avoid ag-grid redraw error
            setTimeout(() => {
                callGridMethod(this.gridApi, 'setGridOption', 'serverSideDatasource', this.serverSideDataSource);
            });
            setTableContext(this.gridApi, c => {
                c.onSelectAll = this.onSelectAll;
                c.onUnselectAll = this.onUnselectAll;
            });
            if (this.props.contextType === ContextType.dialog) {
                (callGridMethod(this.gridApi, 'getRenderedNodes') ?? [])?.[0]?.setSelected?.(true);
            }
            if (this.gridApi) {
                const grouping = getTableViewGrouping(this.props.tableViews);
                const filter = getTableViewFilter(this.props.tableViews);
                if (grouping?.key !== undefined && grouping?.key !== this.state.groupByColumn?.colId) {
                    // Restore group
                    const colDef = callGridMethod(this.gridApi, 'getColumnDefs')?.find(c => c.colId === grouping.key);
                    if (colDef != null) {
                        setTimeout(api => {
                            this.groupByColumn({
                                api,
                                colDef: {
                                    ...colDef,
                                    sort: grouping.sort,
                                },
                                aggFunc: grouping.aggFunc,
                            }, () => {
                                callGridMethod(this.gridApi, 'setFilterModel', filter);
                            });
                        }, 0, this.gridApi);
                    }
                }
                else {
                    callGridMethod(this.gridApi, 'setFilterModel', filter);
                }
            }
            window.addEventListener('resize', this.resizeListener);
        };
        this.setPhantomRow = async () => {
            const phantomRows = await (this.props.fieldProperties.canAddNewLine &&
                !this.props.fieldProperties.isPhantomRowDisabled
                ? this.props.value?.getPhantomRecords()
                : []);
            // Points to the first element to avoid race conditions that could return more than 1 row:
            this.setState(({ phantomRows: currentPhantomRows }) => {
                const newPhantomRows = phantomRows[0] ? [phantomRows[0]] : [];
                return { phantomRows: isEqual(currentPhantomRows, newPhantomRows) ? currentPhantomRows : newPhantomRows };
            });
        };
        /**
         * Gets the IDs of all rows that have been selected.
         *
         * @private
         * @memberof DesktopTableComponent
         */
        this.getSelectedIds = ({ selectedRows }) => {
            return uniq(selectedRows.map(row => row._id));
        };
        this.onSelectAll = (selectedRowCount) => {
            const selectedRows = [];
            callGridMethod(this.gridApi, 'forEachNode', node => {
                if (node.data && node.data.__isGroup) {
                    this.toggleFullGroupSelection(node.data, true);
                }
                selectedRows.push(node.data);
            });
            this.setState({ selectedRowCount, isSelectAllChecked: true, selectedRows }, () => {
                this.props.onTelemetryEvent?.(`tableSelectAllChecked-${this.props.elementId}`, {
                    screenId: this.props.screenId,
                    elementId: this.props.elementId,
                });
                this.props.setFieldProperties?.(this.props.elementId, {
                    ...xtremRedux.getStore().getState().screenDefinitions[this.props.screenId].metadata
                        .uiComponentProperties[this.props.elementId],
                    isSelectAllChecked: true,
                    selectedRecords: this.getSelectedIds({ selectedRows }),
                });
            });
        };
        this.onUnselectAll = () => {
            callGridMethod(this.gridApi, 'forEachNode', node => {
                if (node.data && node.data.__isGroup) {
                    this.toggleFullGroupSelection(node.data, false);
                }
            });
            this.setState({ selectedRowCount: 0, isSelectAllChecked: false, selectedRows: [] }, () => {
                setTimeout(() => {
                    this.props.setFieldProperties?.(this.props.elementId, {
                        ...xtremRedux.getStore().getState().screenDefinitions[this.props.screenId].metadata
                            .uiComponentProperties[this.props.elementId],
                        isSelectAllChecked: false,
                        selectedRecords: [],
                    });
                }, 0);
            });
        };
        this.onSwitchView = (view) => {
            this.props.setFieldProperties?.(this.props.elementId, {
                ...xtremRedux.getStore().getState().screenDefinitions[this.props.screenId].metadata.uiComponentProperties[this.props.elementId],
                tableViewMode: view,
            });
        };
        this.onSwitchCalendarView = (selectedCalendarView) => {
            this.props.onTelemetryEvent?.(`tableCalendarViewSwitched-${this.props.elementId}`, {
                view: selectedCalendarView,
                elementId: this.props.elementId,
                screenId: this.props.screenId,
            });
            this.setState({ selectedCalendarView });
        };
        this.updateSelectedRows = (selectedRows) => {
            this.setState(prevState => ({
                selectedRows: [
                    ...prevState.selectedRows,
                    ...selectedRows.filter(sr => !prevState.selectedRows.find(psr => sr._id === psr._id)),
                ],
            }));
        };
        this.processSelectedGroupState = (selectedRows, isSelected) => {
            const processGroupState = (groupKey, row) => {
                if (!this.state.selectedGroupState || !this.state.selectedGroupState[groupKey]) {
                    return;
                }
                Object.keys(this.state.selectedGroupState[groupKey]).forEach(groupState => {
                    if ((row.__aggFunc && isRowValueInDateGroup(row[groupKey], groupState, row.__aggFunc)) ||
                        (row[groupKey] && row[groupKey] === groupState)) {
                        this.setState(prevState => {
                            const prevSelectedGroupState = prevState.selectedGroupState || {};
                            return {
                                selectedGroupState: updateGroupState(prevSelectedGroupState, groupKey, groupState, isSelected, this.containerRef),
                            };
                        });
                    }
                });
            };
            if (!this.state.selectedGroupState) {
                return;
            }
            const selectedRow = selectedRows[0];
            Object.keys(this.state.selectedGroupState).forEach(groupKey => {
                if (selectedRow[groupKey] !== undefined) {
                    processGroupState(groupKey, selectedRow);
                }
            });
        };
        this.toggleFullGroupSelection = (groupRow, isFullSelection) => {
            const groupKey = groupRow.__groupKey;
            if (isNil(groupKey)) {
                return;
            }
            const keyIndex = get(groupRow, groupKey);
            const newState = isFullSelection ? GroupState.FULL_SELECTED : GroupState.NOT_SELECTED;
            changeGroupIcon(groupRow._id, newState, this.containerRef);
            this.setState(prevState => {
                const prevSelectedGroupState = prevState.selectedGroupState || {};
                const prevGroupState = prevSelectedGroupState[groupKey][keyIndex] || { total: 0 };
                const selectedRowCount = isFullSelection
                    ? prevState.selectedRowCount - (prevGroupState.currentSelected || 0) + prevGroupState.total
                    : prevState.selectedRowCount - (prevGroupState.currentSelected || 0);
                return {
                    selectedRowCount,
                    selectedGroupState: {
                        ...prevSelectedGroupState,
                        [groupKey]: {
                            ...prevSelectedGroupState[groupKey],
                            [keyIndex]: {
                                ...prevGroupState,
                                state: newState,
                                currentSelected: isFullSelection ? prevGroupState.total : 0,
                            },
                        },
                    },
                };
            });
        };
        this.handleRowSelection = (selectedRows, prevSelectedRows) => {
            const numberOfNewSelectedRows = selectedRows.length - prevSelectedRows.length;
            if (numberOfNewSelectedRows === 0) {
                this.newDataSelected = false;
                return;
            }
            const isSelection = numberOfNewSelectedRows > 0;
            const newSelectionSet = xorBy(selectedRows, prevSelectedRows, '_id');
            let selectedChildren = [];
            const isGroupSelection = newSelectionSet.some(row => row.__isGroup);
            newSelectionSet
                .filter(row => row.__isGroup)
                .forEach(row => {
                // If we have a group selected, we need to select all the children.
                selectedChildren = processGroupSelection(this.gridApi, this.props.value, row, isSelection);
                this.toggleFullGroupSelection(row, isSelection);
            });
            if (!isGroupSelection) {
                this.processSelectedGroupState(newSelectionSet, isSelection);
                if (!this.newDataSelected) {
                    this.setState(prevState => ({
                        selectedRowCount: Math.max(prevState.selectedRowCount + numberOfNewSelectedRows, 0),
                    }));
                }
            }
            const allSelectedRows = isSelection
                ? [...selectedRows, ...selectedChildren]
                : differenceBy(selectedRows, selectedChildren, '_id');
            this.setState({
                selectedRows: allSelectedRows,
            });
            const allSelectedSetIds = this.getSelectedIds({ selectedRows: allSelectedRows });
            const newSelectionSetIds = this.getSelectedIds({ selectedRows: [...newSelectionSet, ...selectedChildren] });
            if (this.props.setFieldProperties) {
                this.props.setFieldProperties(this.props.elementId, {
                    ...xtremRedux.getStore().getState().screenDefinitions[this.props.screenId].metadata
                        .uiComponentProperties[this.props.elementId],
                    selectedRecords: allSelectedSetIds,
                });
            }
            this.newDataSelected = false;
            const eventName = isSelection ? 'onRowSelected' : 'onRowUnselected';
            this.notifyNewSelection(newSelectionSetIds, eventName);
        };
        this.notifyNewSelection = (selectedIds, eventName) => {
            selectedIds.forEach(selectedId => {
                const changedItem = this.props.value.getRawRecord({ id: selectedId });
                if (changedItem) {
                    triggerFieldEvent(this.props.screenId, this.props.elementId, eventName, changedItem._id, splitValueToMergedValue(changedItem));
                }
            });
        };
        /**
         * Event listener, triggered when the user clicks the checkbox on the front of the row. Both for selection and
         * deselection.
         * */
        this.onSelectionChange = (event) => {
            if (this.props.setFieldProperties && !event.source.startsWith('api')) {
                const selectedRecords = (xtremRedux.getStore().getState().screenDefinitions[this.props.screenId].metadata
                    .uiComponentProperties[this.props.elementId].selectedRecords || [])
                    .map(id => event.api.getRowNode(id)?.data)
                    .filter(Boolean);
                const previousSelectedIds = this.getSelectedIds({
                    selectedRows: selectedRecords,
                });
                const selectedRows = callGridMethod(event.api, 'getSelectedRows') ?? previousSelectedIds.map(_id => ({ _id }));
                const selectedIds = this.getSelectedIds({ selectedRows });
                // If we still have the same count it means nothing happened.
                if (selectedIds.length === previousSelectedIds.length) {
                    return;
                }
                this.handleRowSelection(selectedRows, selectedRecords);
            }
        };
        this.handleFilterChanged = (params) => {
            const filterModel = getFilterModel(params?.api || this.gridApi, this.state.groupByColumn?.field);
            if (this.props.setTableViewFilter) {
                this.props.setTableViewFilter(0, filterModel);
            }
            this.setShouldResetTable();
            this.clearSelection();
        };
        this.handleColumnRowGroupChanged = (params) => {
            if (this.props.setTableViewGrouping) {
                const columnDefinition = params.column?.getColDef();
                this.props.setTableViewGrouping(0, {
                    key: params.column?.getColId() || '',
                    sort: columnDefinition?.sort || 'asc',
                    aggFunc: columnDefinition?.aggFunc || undefined,
                });
            }
            this.setShouldResetTable();
            this.clearSelection();
        };
        this.setShouldResetTable = () => {
            this.dataOperationMode = DataOperationMode.RESET_TABLE;
        };
        this.onOptionsMenuItemChange = (selectedOptionsMenuItem) => {
            if (this.gridApi) {
                this.setShouldResetTable();
                this.clearSelection();
                this.props.setTableViewOptionsMenuItemAndViewFilter?.(0, selectedOptionsMenuItem, {});
                this.getColumnDefinitions({
                    keepVisibility: true,
                    refreshDataSource: false,
                    selectedOptionsMenuItem,
                    keepGrouping: true,
                });
            }
        };
        this.getRowId = ({ data }) => data._id;
        this.onViewportChanged = params => {
            (callGridMethod(params.api, 'getRenderedNodes') ?? [])
                .filter(row => row.group && row.data)
                .forEach(node => {
                const groupState = this.state.selectedGroupState?.[node.data.__groupKey]?.[node.data.category]?.state;
                if (groupState) {
                    changeGroupIcon(node.data._id, groupState, this.containerRef);
                }
            });
        };
        this.getRowClass = params => {
            if (!params.data || this.props.fieldProperties.isChangeIndicatorDisabled)
                return undefined;
            if (params.data?.__phantom === undefined &&
                (params.data?.__action === RecordActionType.ADDED || params.data?.__action === RecordActionType.MODIFIED)) {
                return 'ag-row-edited';
            }
            return undefined;
        };
        this.gridOptions = {
            onViewportChanged: this.onViewportChanged,
            getRowClass: this.getRowClass,
        };
        this.getColumnDefinitions = ({ refreshDataSource = true, resetSelection = true, keepVisibility = false, resetTableFilters = false, keepGrouping = false, selectedOptionsMenuItem, } = {}) => {
            const columnsData = (this.props.fieldProperties.columns || []).map(columnDefinition => ({
                columnDefinition,
                bind: convertDeepBindToPath(columnDefinition.properties.bind),
                elementId: getNestedFieldElementId(columnDefinition),
            }));
            let columns = getColumns({
                accessBindings: this.props.accessBindings,
                columnsData,
                contextType: this.props.contextType,
                currentTableView: this.props.tableViews?.$current?.level?.[0],
                customizedOrderBy: getTableViewSortOrder(this.props.tableViews),
                elementId: this.props.elementId,
                enumTypes: this.props.enumTypes,
                fieldProperties: () => this.props.fieldProperties,
                isDisabled: this.isDisabled(),
                // eslint-disable-next-line react/no-access-state-in-setstate
                groupBy: this.state.groupByColumn?.colId,
                // eslint-disable-next-line react/no-access-state-in-setstate
                hasFloatingFilters: this.state.hasFloatingFilters,
                isParentDisabled: () => !!this.props.isParentDisabled,
                isReadOnly: this.isReadOnly(),
                level: 0,
                locale: this.props.locale,
                lookupSelectionMode: this.props.lookupSelectionMode,
                nodeTypes: this.props.nodeTypes,
                dataTypes: this.props.dataTypes,
                pageNode: this.props.pageNode,
                screenId: this.props.screenId,
                value: () => this.props.value,
                activeOptionsMenuItem: selectedOptionsMenuItem,
            });
            if (keepVisibility) {
                columns = columns.map(k => {
                    const colState = this.state.columnStates?.find(cs => cs.colId === k.field);
                    if (colState) {
                        return { ...k, hide: !!colState.hide };
                    }
                    const columnHidden = getTableViewColumnHidden(this.props.tableViews);
                    if (columnHidden && k.field) {
                        return { ...k, hide: !!columnHidden[k.field] };
                    }
                    return k;
                });
            }
            this.setState({ columns }, () => {
                if (this.gridApi) {
                    callGridMethod(this.gridApi, 'stopEditing');
                    if (resetSelection) {
                        callGridMethod(this.gridApi, 'deselectAll');
                    }
                    /**
                     * Column definitions have to be reset in order the new properties based on the new collection value
                     * object and event listeners to take effect
                     */
                    callGridMethod(this.gridApi, 'setGridOption', 'columnDefs', this.state.columns);
                    // AG-grid will not update the order of columns unless we explicitly tell it so
                    this.sortColumns();
                    if (refreshDataSource) {
                        /**
                         * Datasource has to be reset so the grid detects the new collection value and rerenders itself.
                         */
                        callGridMethod(this.gridApi, 'setGridOption', 'serverSideDatasource', this.serverSideDataSource);
                        if (resetTableFilters) {
                            callGridMethod(this.gridApi, 'setFilterModel', getTableViewFilter(this.props.tableViews));
                        }
                    }
                    if (keepGrouping && this.state.groupByColumn) {
                        this.groupByColumn({
                            api: this.gridApi,
                            colDef: this.state.groupByColumn,
                            aggFunc: this.state.groupByColumn.aggFunc,
                        });
                    }
                }
            });
        };
        this.hasFilterableColumns = () => this.state.columns.filter(c => c.filter).length > 0;
        this.toggleFloatingFilters = () => {
            this.setState(state => ({
                hasFloatingFilters: !state.hasFloatingFilters,
                autoGroupColumnDef: { ...state.groupByColumn, floatingFilter: !state.hasFloatingFilters },
                groupByColumn: state.groupByColumn
                    ? {
                        ...state.groupByColumn,
                        floatingFilter: !state.hasFloatingFilters,
                    }
                    : undefined,
            }), () => {
                this.props.onTelemetryEvent?.(`tableFloatingFilterRowToggled-${this.props.elementId}`, {
                    hasFloatingFilters: this.state.hasFloatingFilters,
                    elementId: this.props.elementId,
                    screenId: this.props.screenId,
                });
                this.getColumnDefinitions({
                    refreshDataSource: false,
                    resetSelection: false,
                    keepVisibility: true,
                    keepGrouping: true,
                });
            });
        };
        this.exportToExcelWithInfiniteScroll = async (target, options) => {
            const gapi = this.gridApi;
            this.props.onTelemetryEvent?.(`tableDataExported-${target}-${this.props.elementId}`, {
                screenId: this.props.screenId,
                elementId: this.props.elementId,
                target,
            });
            if (gapi) {
                if (!this.props.fieldProperties.isTransient) {
                    this.props.value.cleanCollectionData();
                }
                const pageData = await this.props.value.getPagesIteratively({
                    filters: this.getFilters(getFilterModel(gapi, this.state.groupByColumn?.field)),
                    tableFieldProperties: this.props.fieldProperties,
                    pageSize: 1000,
                    cleanMetadata: false,
                    selectedOptionsMenuItem: this.props.selectedOptionsMenuItem,
                });
                gapi.applyServerSideRowData({
                    successParams: {
                        rowData: pageData,
                        rowCount: pageData.length,
                    },
                });
                if (target === 'excel') {
                    callGridMethod(gapi, 'exportDataAsExcel', options);
                }
                else {
                    callGridMethod(gapi, 'exportDataAsCsv', options);
                }
            }
        };
        this.exportToExcelWithPagination = (target, options) => {
            this.props.onTelemetryEvent?.(`tableDataExported-${target}-${this.props.elementId}`, {
                screenId: this.props.screenId,
                elementId: this.props.elementId,
                target,
            });
            getSafeGridApiContext(currentPage => {
                if (typeof this.props.setGlobalLoading === 'function') {
                    this.props.setGlobalLoading(true);
                }
                const gettingPages = setInterval(() => {
                    getSafeGridApiContext(isLastPageFound => {
                        if (isLastPageFound) {
                            clearInterval(gettingPages);
                            if (target === 'excel') {
                                callGridMethod(this.gridApi, 'exportDataAsExcel', options);
                            }
                            else {
                                callGridMethod(this.gridApi, 'exportDataAsCsv', options);
                            }
                            callGridMethod(this.gridApi, 'paginationGoToPage', currentPage || 0);
                            if (typeof this.props.setGlobalLoading === 'function') {
                                this.props.setGlobalLoading(false);
                            }
                            this.setState({ isExportingExcel: false });
                        }
                        else if (!this.isFetching) {
                            callGridMethod(this.gridApi, 'paginationGoToNextPage');
                            this.setState({ isExportingExcel: true });
                        }
                    }, this.gridApi, 'paginationIsLastPageFound');
                }, 500);
            }, this.gridApi, 'paginationGetCurrentPage');
        };
        this.exportToExcel = (target) => {
            getSafeGridApiContext(columns => {
                const getBindIndexFromColumnId = (column) => {
                    const colId = column.getColDef().field;
                    const tempIndex = Number(column.getColId().replace(`${colId}_`, ''));
                    const index = Number.isNaN(Number(tempIndex)) ? 0 : tempIndex;
                    return { bind: colId, index };
                };
                const options = {
                    columnKeys: getColumnsToExport(columns ?? []),
                    processCellCallback: (params) => {
                        const { bind, index } = getBindIndexFromColumnId(params.column);
                        const filteredColumns = (this.props.fieldProperties.columns || []).filter(c => getNestedFieldElementId(c) === bind ||
                            getNestedFieldElementId(c).startsWith(`${bind}__`));
                        const column = filteredColumns[index];
                        return column ? cellExportFormatter(column, params.value) : '';
                    },
                    fileName: getFieldTitle(this.props.screenId, this.props.fieldProperties, null) || this.props.elementId,
                };
                if (this.isInfiniteScroll()) {
                    this.exportToExcelWithInfiniteScroll(target, options);
                }
                else {
                    this.exportToExcelWithPagination(target, options);
                }
            }, this.gridApi, 'getColumns');
        };
        this.getCacheBlockSize = () => this.props.fieldProperties.pageSize || 20;
        this.isServerSideGroup = (dataItem) => Boolean(dataItem.__isGroup);
        this.excludeInternalColumnsFilter = (column) => {
            const colId = column.getColId();
            return (colId !== undefined &&
                colId !== COLUMN_ID_ROW_SELECTION &&
                colId !== COLUMN_ID_ROW_ACTIONS &&
                colId !== COLUMN_ID_VALIDATIONS);
        };
        this.getAllColumnIds = (gridApi = this.gridApi) => {
            return (callGridMethod(gridApi, 'getColumns') ?? [])
                .filter(this.excludeInternalColumnsFilter)
                .map(column => column.getColId());
        };
        this.getAllDisplayedColumnIds = (gridApi = this.gridApi) => {
            return (callGridMethod(gridApi, 'getAllDisplayedColumns') ?? []).map(column => {
                const colId = column.getColId();
                return colId;
            });
        };
        this.autoSizeAllColumns = (gridApi = this.gridApi) => {
            if (!gridApi) {
                return;
            }
            callGridMethod(gridApi, 'autoSizeColumns', this.getAllColumnIds(gridApi));
            const availableWidth = this.containerRef.current?.clientWidth ?? 0;
            const usedWidth = (gridApi.getAllDisplayedColumns() ?? []).reduce((acc, c) => acc + c.getActualWidth(), 0);
            if (availableWidth - usedWidth > 20) {
                setTimeout(() => {
                    callGridMethod(gridApi, 'sizeColumnsToFit');
                }, 0);
            }
            if (this.state.resizedColumns.length > 0) {
                callGridMethod(this.gridApi, 'setColumnWidths', this.state.resizedColumns);
            }
        };
        this.onFirstDataRendered = (event) => {
            this.adjustTableHeight(event);
            setTimeout(() => {
                this.autoSizeAllColumns(event.api);
            }, 100);
        };
        this.canBeGroupedBy = (column) => {
            const supportedFieldTypes = [
                FieldKey.Reference,
                FieldKey.Label,
                FieldKey.Date,
                FieldKey.RelativeDate,
            ];
            const colDef = column.getColDef();
            if (colDef.cellRendererParams?.fieldProperties?.isTransient) {
                return false;
            }
            return ((supportedFieldTypes.includes(colDef.type) && this.isNavigationPanel()) ||
                column.getId() === AUTO_COLUMN_ID);
        };
        this.ungroupByColumn = ({ colDef, api, }) => {
            let newColumns;
            this.clearSelection();
            this.setState(prevState => {
                newColumns = [
                    ...(prevState.columns ?? []).map(c => {
                        const isPrevGroupByColumn = prevState.groupByColumn && c.field === prevState.groupByColumn.field;
                        return {
                            ...c,
                            ...(isPrevGroupByColumn && { filter: prevState.groupByColumn.filter }),
                            aggFunc: null,
                            rowGroup: false,
                            hide: c.field === colDef.field ? false : c.hide,
                        };
                    }),
                ];
                return { groupByColumn: undefined, columns: newColumns, selectedGroupState: undefined };
            }, () => {
                setTimeout(() => {
                    const filter = getTableViewFilter(this.props.tableViews);
                    // TODO: revisit this fix. It fixes XT-39123.
                    callGridMethod(api, 'setFilterModel', filter);
                    callGridMethod(this.gridApi, 'setGridOption', 'columnDefs', newColumns);
                    if (this.props.setTableViewGrouping) {
                        this.props.setTableViewGrouping(0, undefined);
                    }
                    this.autoSizeAllColumns();
                }, 10);
            });
        };
        this.groupByColumn = ({ colDef, api, aggFunc, }, cb) => {
            let newColumns;
            this.setState(currState => {
                newColumns = [
                    ...(currState.columns ?? []).map(c => {
                        const isGroupByColumn = c.field === colDef.field;
                        const isPrevGroupByColumn = currState.groupByColumn && c.field === currState.groupByColumn.field;
                        let hide = c.hide;
                        let filter = c.filter;
                        if (isGroupByColumn) {
                            hide = true;
                            filter = false;
                            c.aggFunc = aggFunc;
                        }
                        else if (isPrevGroupByColumn) {
                            hide = false;
                            filter = currState.groupByColumn.filter;
                            c.aggFunc = null;
                        }
                        return {
                            ...c,
                            rowGroup: isGroupByColumn,
                            hide,
                            filter,
                        };
                    }),
                ];
                const { cellRenderer, cellEditor, editable, isEditable, resizable, ...colDefProps } = colDef;
                const menuTabs = ['filterMenuTab', 'generalMenuTab'];
                return {
                    groupByColumn: { ...colDef, aggFunc },
                    autoGroupColumnDef: {
                        ...colDefProps,
                        colId: AUTO_COLUMN_ID,
                        rowGroup: true,
                        menuTabs,
                        flex: undefined,
                        width: 250,
                        minWidth: 250,
                        resizable: true,
                        editable: false,
                        ...(this.props.groupTitle && { headerName: this.props.groupTitle }),
                        // by default sort by grouping column in ascending order
                        sort: colDef.sort ?? 'asc',
                        sortable: true,
                        headerName: this.props.groupTitle ? this.props.groupTitle : colDef.headerName,
                        cellRendererParams: {
                            ...colDefProps.cellRendererParams,
                            innerRenderer: cellRenderer,
                        },
                    },
                    columns: newColumns,
                };
            }, () => {
                setTimeout(() => {
                    callGridMethod(api, 'setGridOption', 'columnDefs', newColumns);
                    cb?.();
                    this.autoSizeAllColumns();
                }, 0);
            });
        };
        this.updateColDefsStates = () => {
            return (getSafeGridApiContext((newColumns) => {
                return new Promise(resolve => {
                    this.setState({ columns: newColumns ?? [] }, () => {
                        callGridMethod(this.gridApi, 'setGridOption', 'columnDefs', this.state.columns);
                        resolve();
                    });
                });
            }, this.gridApi, 'getColumnDefs') ?? Promise.resolve());
        };
        this.sortColumns = () => {
            (callGridMethod(this.gridApi, 'getColumnDefs') ?? []).forEach((c) => {
                if (this.gridApi && c.colId && c.xtremSortIndex) {
                    callGridMethod(this.gridApi, 'moveColumn', c.colId, c.xtremSortIndex);
                }
            });
        };
        this.getGroupByMenuItem = ({ isGroupedByThisColumn, colDef, api, }) => {
            if ([FieldKey.Date, FieldKey.RelativeDate].includes(colDef.type)) {
                return {
                    cssClasses: isGroupedByThisColumn ? ['e-table-field-ungroup'] : ['e-table-field-group-by'],
                    name: isGroupedByThisColumn
                        ? localize('@sage/xtrem-ui/ungroup', 'Ungroup')
                        : localize('@sage/xtrem-ui/group-by', 'Group by'),
                    icon: isGroupedByThisColumn
                        ? renderToStaticMarkup(
                        // eslint-disable-next-line react/jsx-indent
                        React.createElement("div", { style: { position: 'relative' } },
                            React.createElement(Icon, { type: "bullet_list" }),
                            React.createElement("div", { style: {
                                    position: 'absolute',
                                    width: '24px',
                                    bottom: '2px',
                                    left: '2px',
                                    borderTop: '1px solid black',
                                    transform: 'rotate(-45deg)',
                                    transformOrigin: '0% 0%',
                                } })))
                        : renderToStaticMarkup(React.createElement(Icon, { type: "bullet_list" })),
                    subMenu: isGroupedByThisColumn
                        ? undefined
                        : [
                            {
                                cssClasses: ['e-table-field-group-by-year'],
                                name: localize('@sage/xtrem-ui/group-by-year', 'Year'),
                                action: () => {
                                    setTimeout(() => {
                                        this.groupByColumn({ api, colDef, aggFunc: 'year' });
                                    }, 10);
                                },
                            },
                            {
                                cssClasses: ['e-table-field-group-by-month'],
                                name: localize('@sage/xtrem-ui/group-by-month', 'Month'),
                                action: () => {
                                    setTimeout(() => {
                                        this.groupByColumn({ api, colDef, aggFunc: 'month' });
                                    }, 10);
                                },
                            },
                            {
                                cssClasses: ['e-table-field-group-by-day'],
                                name: localize('@sage/xtrem-ui/group-by-day', 'Day'),
                                action: () => {
                                    setTimeout(() => {
                                        this.groupByColumn({ api, colDef, aggFunc: 'day' });
                                    }, 10);
                                },
                            },
                        ],
                    action: () => {
                        if (isGroupedByThisColumn) {
                            this.ungroupByColumn({ colDef, api });
                        }
                    },
                };
            }
            return {
                cssClasses: isGroupedByThisColumn ? ['e-table-field-ungroup'] : ['e-table-field-group-by'],
                name: isGroupedByThisColumn
                    ? localize('@sage/xtrem-ui/ungroup', 'Ungroup')
                    : localize('@sage/xtrem-ui/group-by-this-column', 'Group by this column'),
                icon: isGroupedByThisColumn
                    ? renderToStaticMarkup(
                    // eslint-disable-next-line react/jsx-indent
                    React.createElement("div", { style: { position: 'relative' } },
                        React.createElement(Icon, { type: "bullet_list" }),
                        React.createElement("div", { style: {
                                position: 'absolute',
                                width: '24px',
                                bottom: '2px',
                                left: '2px',
                                borderTop: '1px solid black',
                                transform: 'rotate(-45deg)',
                                transformOrigin: '0% 0%',
                            } })))
                    : renderToStaticMarkup(React.createElement(Icon, { type: "bullet_list" })),
                action: () => {
                    if (isGroupedByThisColumn) {
                        this.ungroupByColumn({ colDef, api });
                    }
                    else {
                        setTimeout(() => {
                            this.groupByColumn({ api, colDef });
                        }, 10);
                    }
                },
            };
        };
        this.getMainMenuItems = ({ column, api }) => {
            /**
             * params.defaultItems are
            [
                "pinSubMenu",
                "separator",
                "autoSizeThis",
                "autoSizeAll",
                "separator",
                "separator",
                "resetColumns"
            ]
             */
            const defaultMenuItems = ['pinSubMenu', 'separator', 'autoSizeThis', 'autoSizeAll'];
            const colId = column.getColId();
            const isAutoColumn = colId === AUTO_COLUMN_ID;
            const colDef = isAutoColumn ? this.state.groupByColumn : this.state.columns.find(c => c.field === colId);
            if (!colDef) {
                throw new Error('Unable to find column definition');
            }
            const isGroupedByThisColumn = this.state.groupByColumn?.colId === colId || isAutoColumn;
            return this.canBeGroupedBy(column)
                ? [...defaultMenuItems, 'separator', this.getGroupByMenuItem({ api, isGroupedByThisColumn, colDef })]
                : defaultMenuItems;
        };
        this.onCellFocused = (params) => {
            if (this.props.onFocus) {
                this.props.onFocus(this.props.screenId, this.props.elementId, String(params.rowIndex || ''), typeof params.column === 'object' && params.column ? params.column.getColId() : '');
            }
            if (this.props.lookupSelectionMode === 'single') {
                callGridMethod(this.gridApi, 'forEachNode', node => {
                    if (node.rowIndex === params.rowIndex) {
                        node.setSelected(true);
                    }
                });
            }
        };
        this.onCellKeyDown = (event) => {
            const keyboardEvent = event.event;
            if (this.props.lookupSelectionMode === 'single' && keyboardEvent?.key === 'Enter') {
                this.props.onRowClick?.(event.data._id, keyboardEvent)();
            }
            if (event.data?.__phantom && keyboardEvent.key === 'Enter') {
                if (keyboardEvent.getModifierState('Meta') || keyboardEvent.getModifierState('Control')) {
                    this.props.onTelemetryEvent?.(`tablePhantomRowCommittedByKeyboard-${this.props.elementId}`, {
                        screenId: this.props.screenId,
                        elementId: this.props.elementId,
                    });
                    tryToCommitPhantomRow({
                        api: event.api,
                        screenId: this.props.screenId,
                        elementId: this.props.elementId,
                        value: this.props.value,
                    });
                }
                else if (event.column.getColId() === COLUMN_ID_ROW_ACTIONS) {
                    this.props.value.resetRowToDefaults({
                        id: event.data._id,
                        level: event.data.__level,
                        parentId: event.data.__parentId,
                        isOrganicChange: true,
                        resetDirtiness: true,
                    });
                    callGridMethod(event.api, 'stopEditing');
                    callGridMethod(event.api, 'setFocusedCell', 0, COLUMN_ID_ROW_ACTIONS, 'top');
                }
            }
        };
        this.getAllNotVisibleNotHiddenColumns = (tableProperties) => {
            return (tableProperties.columns || [])
                .filter(c => {
                return c.properties.isHiddenOnMainField;
            })
                .map(c => {
                return convertDeepBindToPathNotNull(c.properties.bind);
            });
        };
        this.hideUnhideColumns = (event) => {
            const gridApi = event?.api ?? this.gridApi;
            if (!gridApi ||
                gridApi.isDestroyed() ||
                !this.props.fieldProperties.canUserHideColumns ||
                (event && ['toolPanelUi', 'api', 'gridOptionsUpdated'].includes(event.source))) {
                return;
            }
            const hiddenColumns = this.state?.columnStates?.filter(c => c.colId && c.hide).map(c => c.colId);
            const allColumns = this.getAllColumnIds(gridApi);
            const visibleColumns = this.getAllDisplayedColumnIds();
            const notVisibleNotHiddenColumns = this.getAllNotVisibleNotHiddenColumns(this.props.fieldProperties);
            const currentlyHiddenColumns = difference(allColumns, visibleColumns);
            const columnsToHide = visibleColumns
                .filter(c => c !== COLUMN_ID_VALIDATIONS)
                .filter(colId => hiddenColumns && hiddenColumns.find(colBind => isColumnBindVariant(colBind, colId)));
            const columnsToUnhide = currentlyHiddenColumns
                .filter(c => c !== COLUMN_ID_VALIDATIONS && c !== this.state.groupByColumn?.field)
                .filter(colId => hiddenColumns && !hiddenColumns.find(colBind => isColumnBindVariant(colBind, colId)))
                .filter(colId => !notVisibleNotHiddenColumns.find(colBind => isColumnBindVariant(colBind, colId)));
            callGridMethod(gridApi, 'setColumnsVisible', columnsToHide, false);
            callGridMethod(gridApi, 'setColumnsVisible', columnsToUnhide, true);
            this.autoSizeAllColumns();
        };
        this.isColumnHidden = (colBind = '') => {
            const hiddenColumns = this.props.fieldProperties.hiddenColumns ?? [];
            return hiddenColumns.includes(colBind);
        };
        this.onFilterByErrors = async () => {
            this.props.onTelemetryEvent?.(`tableFilteredByErrors-${this.props.elementId}`, {
                screenId: this.props.screenId,
                elementId: this.props.elementId,
            });
            this.isFilteringErrors = true;
            this.dataOperationMode = DataOperationMode.FILTER_ERRORS;
            if (this.gridApi) {
                callGridMethod(this.gridApi, 'setGridOption', 'serverSideDatasource', this.serverSideDataSource);
            }
        };
        this.onUnsetFilterByErrors = async () => {
            this.isFilteringErrors = false;
            this.props.value.resetFilter();
            this.dataOperationMode = DataOperationMode.FETCH_PAGE;
            if (this.gridApi) {
                callGridMethod(this.gridApi, 'setFilterModel', null);
                callGridMethod(this.gridApi, 'setGridOption', 'serverSideDatasource', this.serverSideDataSource);
            }
        };
        this.getDefaultHeight = () => {
            return (this.props.fixedHeight ||
                (this.props.numberOfVisibleRows || this.props.fieldProperties.pageSize || 20) * ROW_HEIGHT + 85);
        };
        this.adjustTableHeight = ({ api = this.gridApi } = {}) => {
            if (!this.containerRef.current) {
                return;
            }
            if (this.props.fixedHeight) {
                this.containerRef.current.style.height = `${this.props.fixedHeight}px`;
                return;
            }
            if (!this.props.fieldProperties.canAddNewLine || !api) {
                return;
            }
            // A minimum row height must be set so we have some place to display the empty placeholders
            getSafeGridApiContext(rowCount => {
                const renderedRowCount = Math.max(rowCount, 4);
                const containerHeight = this.containerRef.current?.clientHeight || 0;
                let size = Math.min(renderedRowCount * ROW_HEIGHT + 164, this.getDefaultHeight());
                if (this.props.fieldProperties.canAddNewLine && !this.isDisabled()) {
                    size += ROW_HEIGHT;
                }
                if (containerHeight === size) {
                    return;
                }
                if (this.containerRef.current) {
                    this.containerRef.current.style.height = `${size}px`;
                }
            }, api, 'getDisplayedRowCount');
        };
        /**
         * This method is triggered if the user clicks or tabs outside the table. If this happens we want to stop editing the cell in order
         * to trigger onCellEditingStopped from ag-grid.
         */
        this.onBlurTable = () => {
            /* The timeout it's necessary because the first click into the table triggers this event and we only want to stop editing if we are blur
             * outside the table. */
            setTimeout(() => {
                const { activeElement } = document;
                getSafeGridApiContext(focusedCell => {
                    const focusedColumnId = focusedCell?.column?.getDefinition()?.columnId;
                    const hasLookupDialogOpen = this.props.activeLookupDialog?.elementId === this.props.elementId &&
                        this.props.activeLookupDialog?.nestedField === focusedColumnId;
                    // We check the focus is inside the table element, if not we stop editing the cell.
                    if (activeElement &&
                        !this.containerRef.current?.contains(activeElement) &&
                        !isChildOfElementWithClass(activeElement, 'e-lookup-dialog') &&
                        !isChildOfElementWithClass(activeElement, 'DayPicker-wrapper') &&
                        !isChildOfElementWithAttributeValue(activeElement, 'data-component', 'action-popover') &&
                        !hasLookupDialogOpen) {
                        try {
                            callGridMethod(this.gridApi, 'stopEditing');
                        }
                        catch {
                            // In some units this fails because ag-grid, so fire and forget.
                        }
                    }
                }, this.gridApi, 'getFocusedCell');
            }, 500);
        };
        this.clearSelection = () => {
            if (!this.gridApi || this.gridApi.isDestroyed()) {
                return;
            }
            setTableContext(this.gridApi, c => {
                c.isSelectAllEnabled = false;
                c.totalRowCount = 0;
            });
            this.onUnselectAll();
            (callGridMethod(this.gridApi, 'getSelectedNodes') ?? []).forEach(node => {
                node.setSelected(false, false, 'checkboxSelected');
            });
        };
        this.onColumnPanelHiddenStateChange = async (colId, isHidden) => {
            if (this.gridApi) {
                callGridMethod(this.gridApi, 'setColumnVisible', colId, isHidden);
                await this.updateColumnState();
                await this.updateColDefsStates();
            }
            if (this.state.columnStates && this.props.setTableViewColumnHidden) {
                const visible = this.getAllDisplayedColumnIds();
                const hidden = difference(this.getAllColumnIds(), visible);
                const hiddenColumns = {};
                visible.forEach(column => {
                    hiddenColumns[column] = false;
                });
                hidden.forEach(column => {
                    hiddenColumns[column] = true;
                });
                this.props.setTableViewColumnHidden(0, hiddenColumns);
            }
        };
        this.onColumnPanelOrderChangeChange = async (colIds) => {
            if (this.gridApi) {
                callGridMethod(this.gridApi, 'moveColumns', colIds, 0);
                await this.updateColumnState();
                await this.updateColDefsStates();
            }
            if (this.state.columnStates && this.props.setTableViewColumnOrder) {
                const columnOrder = this.state.columnStates
                    .filter(columnState => columnState.colId !== undefined &&
                    columnState.colId !== COLUMN_ID_ROW_SELECTION &&
                    columnState.colId !== COLUMN_ID_ROW_ACTIONS &&
                    columnState.colId !== COLUMN_ID_VALIDATIONS &&
                    columnState.colId !== COLUMN_ID_AUTO_COLUMN)
                    .map(columnState => columnState.colId);
                this.props.setTableViewColumnOrder(0, columnOrder);
            }
        };
        this.onSortChanged = (event) => {
            if (event.source === 'gridOptionsUpdated') {
                return;
            }
            getSafeGridApiContext(columnState => {
                const groupBySort = columnState.find(c => c.colId === AUTO_COLUMN_ID)?.sort;
                const grouping = getTableViewGrouping(this.props.tableViews);
                if (groupBySort &&
                    this.state.groupByColumn?.field &&
                    groupBySort !== grouping?.key &&
                    this.props.setTableViewGrouping) {
                    this.props.setTableViewGrouping(0, {
                        key: this.state.groupByColumn.field,
                        sort: groupBySort,
                        aggFunc: this.state.groupByColumn.aggFunc || undefined,
                    });
                }
            }, event.api, 'getColumnState');
            const sortOrder = event.api
                .getColumnState()
                .filter(columnState => columnState.colId !== undefined &&
                columnState.colId !== COLUMN_ID_ROW_SELECTION &&
                columnState.colId !== COLUMN_ID_ROW_ACTIONS &&
                columnState.colId !== COLUMN_ID_VALIDATIONS &&
                columnState.colId !== COLUMN_ID_AUTO_COLUMN)
                .map(columnState => {
                return {
                    colId: columnState.colId,
                    sort: columnState.sort ?? null,
                    sortIndex: columnState.sortIndex ?? null,
                };
            });
            if (this.props.setTableViewSortOrder) {
                this.props.setTableViewSortOrder(0, sortOrder);
            }
            this.setShouldResetTable();
        };
        this.onCellClicked = async (event) => {
            if (event.data?.__phantom && event.column.getColId() === COLUMN_ID_ROW_ACTIONS) {
                await this.props.value.resetRowToDefaults({
                    id: event.data._id,
                    level: event.data.__level,
                    parentId: event.data.__parentId,
                    isOrganicChange: true,
                    resetDirtiness: true,
                });
            }
        };
        this.onCellEditingStopped = async (event) => {
            const isPhantom = event.data?.__phantom !== undefined;
            if (!isPhantom) {
                return;
            }
            if (event.value !== event.oldValue) {
                const columnId = event.colDef.field;
                const newValue = event.data[event.colDef.field];
                let mergedValue = this.props.value.getMergedValue({
                    recordId: event.data._id,
                    columnId,
                    value: newValue,
                    level: event.data.__level,
                });
                const dirtyColumns = new Set([...mergedValue.__dirtyColumns, columnId]);
                const validationResult = await this.props.value.runValidationOnRecord({
                    recordData: {
                        ...mergedValue,
                        __dirtyColumns: dirtyColumns,
                    },
                    changedColumnId: event.colDef.field,
                });
                mergedValue = this.props.value.getMergedValue({
                    recordId: event.data._id,
                    columnId,
                    value: newValue,
                    level: event.data.__level,
                });
                const updatedData = {
                    ...mergedValue,
                    __validationState: validationResult,
                };
                this.props.value.setRecordValue({
                    recordData: updatedData,
                    level: event.data.__level,
                    toBeMarkedAsDirty: Array.from(dirtyColumns),
                    isOrganicChange: true,
                    changedColumnId: columnId,
                });
            }
        };
        this.onColumnResized = async (event) => {
            if (event.finished && event.column) {
                const columnKey = event.column.getColId();
                const columnWidth = event.column.getActualWidth();
                const updatedColumn = { key: columnKey, newWidth: columnWidth };
                this.setState(prevState => {
                    const currentColumns = [...prevState.resizedColumns];
                    const columnIndex = currentColumns.findIndex(col => col.key === columnKey);
                    if (columnIndex !== -1) {
                        currentColumns[columnIndex] = updatedColumn;
                    }
                    else {
                        currentColumns.push(updatedColumn);
                    }
                    return { resizedColumns: currentColumns };
                });
            }
        };
        this.onOpenColumnPanel = () => {
            this.setState({ isConfigurationDialogOpen: true });
        };
        this.onCloseColumnPanel = () => {
            this.setState({ isConfigurationDialogOpen: false });
        };
        this.noRowsOverlayComponentParams = {
            ...this.props,
            isEditable: !this.isReadOnly() && !this.isDisabled(),
        };
        this.onRowClicked = e => {
            return onRowClick(this.props.onRowClick)(e);
        };
        this.rowClassRules = getRowClassRules({
            canSelect: this.props.fieldProperties.canSelect || false,
            isChangeIndicatorDisabled: this.props.fieldProperties.isChangeIndicatorDisabled,
            openedRecordId: this.props.openedRecordId,
        });
        this.isInfiniteScroll = () => {
            return Boolean(this.props.isUsingInfiniteScroll || (this.props.fixedHeight && this.props.fixedHeight > 0));
        };
        setAgGridLicence();
        this.state = {
            autoGroupColumnDef: {
                colId: AUTO_COLUMN_ID,
                rowGroup: true,
                menuTabs: ['filterMenuTab', 'generalMenuTab'],
                flex: undefined,
                width: 250,
                minWidth: 250,
                resizable: true,
                editable: false,
                ...(this.props.groupTitle && { headerName: this.props.groupTitle }),
                sortable: true,
            },
            isConfigurationDialogOpen: false,
            columns: [],
            // Enable floating filters for lookups
            hasFloatingFilters: this.props.lookupSelectionMode !== undefined,
            isExportingExcel: false,
            isSelectAllChecked: false,
            phantomRows: [],
            selectedRows: [],
            selectedRowCount: 0,
            selectedCalendarView: 'dayGridMonth',
            tableHasData: props.value &&
                props.value.getData &&
                props.value.getData({ temporaryRecords: this.props.additionalLookupRecords?.() }).length > 0,
            resizedColumns: [],
        };
    }
    componentDidMount() {
        this.getColumnDefinitions({
            keepVisibility: true,
            selectedOptionsMenuItem: this.props.selectedOptionsMenuItem,
        });
        this.actionSubscription = subscribeToActions(action => {
            if (action.type === ActionType.RedrawComponent &&
                this.gridApi &&
                this.props.screenId === action.value.screenId &&
                this.props.elementId === action.value.elementId) {
                const columns = action.value.columnBind
                    ? (callGridMethod(this.gridApi, 'getColumnDefs') ?? [])
                        ?.filter((c) => c.field?.indexOf(action.value.columnBind) === 0)
                        .map((c) => c.colId)
                    : undefined;
                if (columns && columns.length > 0) {
                    this.gridApi.refreshCells({ force: true, columns });
                }
                else {
                    this.getColumnDefinitions({ keepVisibility: true });
                }
            }
        });
    }
    componentDidUpdate(prevProps) {
        if (!this.isNavigationPanel() && this.props.selectedRecordId !== prevProps.selectedRecordId) {
            this.resetOptionsMenu();
        }
        if (!this.gridApi) {
            return;
        }
        if (!this.props.isInFocus && prevProps.isInFocus) {
            callGridMethod(this.gridApi, 'stopEditing', true);
        }
        // When the value changes (e.g the user switches between records), we should unset filters that were applied to the table
        if (!this.isNavigationPanel() && prevProps.value !== this.props.value) {
            callGridMethod(this.gridApi, 'setFilterModel', {});
        }
        else if (this.isNavigationPanel() &&
            this.props.value &&
            !isEqual(prevProps.tableViews, this.props.tableViews)) {
            callGridMethod(this.gridApi, 'setFilterModel', getTableViewFilter(this.props.tableViews));
        }
        if (!this.state.columnStates) {
            this.initColumnState();
        }
        if (!isEqual(prevProps.selectedOptionsMenuItem, this.props.selectedOptionsMenuItem)) {
            callGridMethod(this.gridApi, 'setGridOption', 'serverSideDatasource', this.serverSideDataSource);
        }
        if (prevProps.fieldProperties.isPhantomRowDisabled !== this.props.fieldProperties.isPhantomRowDisabled) {
            this.setPhantomRow();
            setTimeout(() => callGridMethod(this.gridApi, 'refreshServerSide', { purge: true }), 0);
        }
        if (prevProps.value !== this.props.value ||
            prevProps.fieldProperties.isDisabled !== this.props.fieldProperties.isDisabled ||
            prevProps.fieldProperties.tableViewMode !== this.props.fieldProperties.tableViewMode ||
            prevProps.isParentDisabled !== this.props.isParentDisabled ||
            prevProps.isReadOnly !== this.props.isReadOnly) {
            this.dataOperationMode =
                prevProps.fieldProperties.tableViewMode !== this.props.fieldProperties.tableViewMode
                    ? DataOperationMode.RESET_TABLE
                    : DataOperationMode.NONE;
            const resetTableFilters = this.isNavigationPanel();
            this.getColumnDefinitions({ keepVisibility: true, keepGrouping: true, resetTableFilters });
            // Revalidate field with current data
            this.props.value?.validateField();
            const ctx = getTableContext(this.gridApi);
            if (ctx) {
                // Reset headers
                setTableContext(this.gridApi, c => {
                    c.headerClasses = {};
                });
                if (!this.isNavigationPanel()) {
                    this.gridApi.refreshHeader();
                }
            }
        }
        if (!this.isNavigationPanel() && this.props.openedRecordId !== prevProps.openedRecordId) {
            const rowNodes = [];
            if (typeof prevProps.openedRecordId === 'string') {
                const node = this.getRowNode({ id: prevProps.openedRecordId, gridApi: this.gridApi });
                if (node) {
                    rowNodes.push(node);
                }
            }
            if (typeof this.props.openedRecordId === 'string') {
                const node = this.getRowNode({ id: this.props.openedRecordId, gridApi: this.gridApi });
                if (node) {
                    rowNodes.push(node);
                }
            }
            if (rowNodes.length) {
                setTimeout(() => callGridMethod(this.gridApi, 'redrawRows', { rowNodes }));
            }
        }
        if (!isEqual(sortBy(prevProps.fieldProperties.hiddenColumns), sortBy(this.props.fieldProperties.hiddenColumns))) {
            this.hideUnhideColumns();
            this.autoSizeAllColumns();
        }
        if (!isEqual(sortBy(prevProps.fieldProperties.selectedRecords), sortBy(this.props.fieldProperties.selectedRecords))) {
            callGridMethod(this.gridApi, 'deselectAll');
            this.selectTableItems({ api: this.gridApi });
        }
    }
    componentWillUnmount() {
        if (this.actionSubscription) {
            this.actionSubscription();
        }
        if (this.collectionValueChangeSubscription) {
            this.collectionValueChangeSubscription();
        }
        if (this.collectionValidityChangeSubscription) {
            this.collectionValidityChangeSubscription();
        }
        window.removeEventListener('resize', this.resizeListener);
    }
    getCursor(api) {
        return callGridMethod(api, 'getDisplayedRowAtIndex', (callGridMethod(api, 'getLastDisplayedRow') ?? 1) - 1)
            ?.data?.__cursor;
    }
    getFilters(filterModel) {
        return Object.keys(filterModel)
            .map(mapAgGridFilterToXtremFilters(filterModel))
            .concat(this.isFilteringErrors
            ? [
                {
                    id: '_id',
                    value: [
                        {
                            filterValue: this.props.value.getAllInvalidRecords().map(r => r._id),
                            filterType: {
                                text: '_id in list',
                                value: SET,
                            },
                        },
                    ],
                },
            ]
            : []);
    }
    async initColumnState() {
        getSafeGridApiContext(columnStates => {
            return new Promise(resolve => {
                const columnHidden = getTableViewColumnHidden(this.props.tableViews);
                if (columnHidden) {
                    columnStates.forEach((column) => {
                        if (Boolean(column.hide) !== Boolean(columnHidden[column.colId])) {
                            column.hide = Boolean(columnHidden[column.colId]);
                        }
                    });
                }
                this.setState({ columnStates }, resolve);
            });
        }, this.gridApi, 'getColumnState');
    }
    async updateColumnState() {
        getSafeGridApiContext(columnStates => {
            return new Promise(resolve => {
                this.setState({ columnStates }, resolve);
            });
        }, this.gridApi, 'getColumnState');
    }
    renderCalendarView() {
        const eventCard = getCardDefinitionFromColumns({
            columns: this.props.fieldProperties.columns,
            mobileCard: this.props.fieldProperties.mobileCard,
            screenId: this.props.screenId,
            isGreaterThanSmall: true,
            hiddenColumns: this.props.fieldProperties.hiddenColumns,
            sortColumns: this.props.fieldProperties.sortColumns,
        });
        if (!this.props.fieldProperties.startDateField) {
            // This should never be the case because the calendar view is only available if the startDateField property is defined
            return null;
        }
        return (React.createElement("div", { className: "e-navigation-panel-calendar-wrapper" },
            React.createElement(AsyncCalendarBodyComponent, { cardColor: this.props.fieldProperties.cardColor, elementId: this.props.elementId, endDateField: this.props.fieldProperties.endDateField, eventCard: eventCard, selectedView: this.state.selectedCalendarView, fieldProperties: this.props.fieldProperties, fixedHeight: this.props.fixedHeight ? this.props.fixedHeight - 40 : undefined, isGreaterThanSmall: true, onEventClick: (_id, event, isModifierKeyPushed) => {
                    this.props.onRowClick?.(_id, event, isModifierKeyPushed)();
                }, screenId: this.props.screenId, selectedOptionsMenuItem: this.props.selectedOptionsMenuItem, startDateField: this.props.fieldProperties.startDateField, value: this.props.value, isEventMovable: this.props.fieldProperties.isEventMovable })));
    }
    render() {
        const domLayout = this.isInfiniteScroll() ? 'normal' : 'autoHeight';
        const hasBulkActions = (this.props.bulkActions ?? []).length > 0;
        const shouldRenderBulkActionBar = hasBulkActions && this.state.selectedRowCount > 0 && this.gridApi;
        const hasAggregatedGroupColumns = !!this.props.fieldProperties?.columns?.find(c => !!c.properties.groupAggregationMethod);
        const columnPanelColumnStates = getColumnStatesForColumnPanel(this.props.screenId, (callGridMethod(this.gridApi, 'getColumnDefs') ?? []), this.state.columnStates);
        const hasAddItemsButton = Boolean(this.props.fieldProperties.canAddNewLine && !this.props.fieldProperties.isPhantomRowDisabled);
        return (React.createElement(React.Fragment, null,
            process.env.NODE_ENV === 'test' && this.gridApi && React.createElement("p", { "data-testid": "api", style: { display: 'none' } }),
            shouldRenderBulkActionBar && this.gridApi && (React.createElement(DesktopTableBulkActionBar, { screenId: this.props.screenId, onClearSelection: this.clearSelection, isSelectAllChecked: this.state.isSelectAllChecked, bulkActions: this.props.bulkActions, gridApi: this.gridApi, selectedRowCount: this.state.selectedRowCount, selectedGroupState: this.state.selectedGroupState, groupByColumnField: this.state.groupByColumn?.field, onTelemetryEvent: this.props.onTelemetryEvent })),
            (!this.props.lookupSelectionMode || this.props.elementId === navigationPanelId) &&
                !shouldRenderBulkActionBar && (React.createElement(DesktopTableHeaderComponent, { columnPanelDisabled: !!this.state.groupByColumn || !this.props.fieldProperties.canUserHideColumns, elementId: this.props.elementId, fieldProperties: this.props.fieldProperties, hasAddItemsButton: hasAddItemsButton, hasCalendarView: !!this.props.fieldProperties.startDateField, hasData: this.state.tableHasData, hasFloatingFilters: this.state.hasFloatingFilters, hasSidebar: Boolean(this.props.fieldProperties.sidebar), isDisabled: this.isDisabled(), isExportDisabled: !!this.state.groupByColumn, isReadOnly: this.isReadOnly(), onExport: this.exportToExcel, onFilterByErrors: this.onFilterByErrors, onFocusPhantomRow: this.focusPhantomRowAndStartEditing, onOpenColumnPanel: this.onOpenColumnPanel, onOptionsMenuItemChange: this.onOptionsMenuItemChange, onSwitchCalendarView: this.onSwitchCalendarView, onSwitchView: this.onSwitchView, onTelemetryEvent: this.props.onTelemetryEvent, onToggleFilters: this.hasFilterableColumns() ? this.toggleFloatingFilters : undefined, onUnsetFilterByErrors: this.onUnsetFilterByErrors, screenId: this.props.screenId, selectedCalendarView: this.state.selectedCalendarView, selectedOptionsMenuItem: this.props.selectedOptionsMenuItem, tableViewMode: this.props.fieldProperties.tableViewMode, validationErrors: this.props.validationErrors })),
            React.createElement("div", { "data-testid": `e-table-field-${this.props.elementId}`, style: {
                    height: '100%',
                    overflow: 'hidden',
                }, className: "e-table-field-desktop-wrapper" },
                this.props.fieldProperties.tableViewMode === 'calendar' && this.renderCalendarView(),
                (this.props.fieldProperties.tableViewMode === 'table' ||
                    !this.props.fieldProperties.tableViewMode) && (React.createElement("div", { onBlur: this.onBlurTable, ref: this.containerRef, style: {
                        height: this.isInfiniteScroll() ? `${this.getDefaultHeight()}px` : '100%',
                        maxHeight: '100%',
                        width: '100%',
                        visibility: this.state.isExportingExcel ? 'hidden' : 'visible',
                    }, className: "ag-theme-balham" },
                    React.createElement(AgGridReact, { gridId: this.gridId, autoSizeStrategy: this.autoSizeStrategy, autoGroupColumnDef: this.state.autoGroupColumnDef, cacheBlockSize: this.getCacheBlockSize(), columnDefs: this.state.columns, components: frameworkComponents, debug: isDevMode() && !isNoConsole(), defaultColDef: this.defaultColDef, domLayout: domLayout, floatingFiltersHeight: ROW_HEIGHT, getMainMenuItems: this.getMainMenuItems, getRowId: this.getRowId, gridOptions: this.gridOptions, groupDisplayType: "singleColumn", headerHeight: ROW_HEIGHT, isServerSideGroup: this.isServerSideGroup, loadingCellRenderer: TableLoadingCellRenderer, loadingCellRendererParams: this.loadingCellRendererParams, loadingOverlayComponent: "Loader", loadingOverlayComponentParams: this.loadingOverlayComponentParams, localeText: localeText(), maxConcurrentDatasourceRequests: 1, modules: activatedAgGridModules, noRowsOverlayComponent: "DesktopTableEmptyComponent", noRowsOverlayComponentParams: this.noRowsOverlayComponentParams, onCellClicked: this.onCellClicked, onCellEditingStopped: this.onCellEditingStopped, onCellFocused: this.onCellFocused, onCellKeyDown: this.onCellKeyDown, onColumnEverythingChanged: this.hideUnhideColumns, onColumnResized: this.onColumnResized, onColumnRowGroupChanged: this.handleColumnRowGroupChanged, onFilterChanged: this.handleFilterChanged, onFirstDataRendered: this.onFirstDataRendered, onGridReady: this.onGridReady, onGridSizeChanged: this.adjustTableHeight, onRowClicked: this.onRowClicked, onSelectionChanged: this.onSelectionChange, onSortChanged: this.onSortChanged, pagination: !this.isInfiniteScroll(), paginationPageSize: this.props.fieldProperties.pageSize || 20, paginationPageSizeSelector: false, pinnedTopRowData: this.state.phantomRows ?? [], rowClassRules: this.rowClassRules, rowHeight: ROW_HEIGHT, rowModelType: "serverSide", rowSelection: this.props.fieldProperties.canSelect
                            ? this.props.lookupSelectionMode || 'multiple'
                            : undefined, serverSideSortAllLevels: true, singleClickEdit: true, suppressAggFuncInHeader: true, suppressCellFocus: this.isDisabled(), suppressColumnVirtualisation: process.env.NODE_ENV === 'test', suppressContextMenu: true, suppressDragLeaveHidesColumns: true, suppressExcelExport: false, suppressPaginationPanel: this.isDisabled(), suppressRowClickSelection: true, undoRedoCellEditing: true, valueCacheNeverExpires: true, groupTotalRow: hasAggregatedGroupColumns ? 'bottom' : undefined })))),
            React.createElement(TableConfigurationDialog, { isDialogOpen: this.state.isConfigurationDialogOpen, onDialogClose: this.onCloseColumnPanel, columnPanelColumnStates: columnPanelColumnStates, onChangeColumnHiddenState: this.onColumnPanelHiddenStateChange, onOrderChange: this.onColumnPanelOrderChangeChange })));
    }
}
const mapStateToProps = (state, props) => {
    const screenDefinition = getPageDefinitionFromState(props.screenId, state);
    const screenElement = getScreenElement(screenDefinition);
    const selectedRecordId = screenDefinition.queryParameters?._id;
    const openedRecordId = state?.sidebar?.screenId === props.screenId && state?.sidebar?.elementId === props.elementId
        ? state.sidebar.recordId
        : undefined;
    const pageNode = getPagePropertiesFromPageDefinition(screenDefinition).node;
    const tableViews = screenDefinition.tableViews[props.elementId];
    const selectedOptionsMenuItem = getActiveOptionsMenu(props.screenId, tableViews, props.fieldProperties.optionsMenu);
    return {
        ...props,
        isInFocus: !!state.focusPosition &&
            state.focusPosition.elementId === props.elementId &&
            state.focusPosition.screenId === props.screenId &&
            !state.focusPosition.nestedField,
        graphApi: screenElement.$.graph,
        selectedOptionsMenuItem,
        selectedRecordId,
        openedRecordId,
        activeLookupDialog: state.activeLookupDialog,
        tableViews,
        pageNode,
        accessBindings: screenDefinition.accessBindings,
        dataTypes: state.dataTypes,
        onTelemetryEvent: state.applicationContext?.onTelemetryEvent,
    };
};
const mapDispatchToProps = (dispatch, props) => ({
    setTableViewColumnHidden: (level, columnHidden) => {
        dispatch(xtremRedux.actions.setTableViewColumnHidden(props.screenId, props.elementId, level, columnHidden));
    },
    setTableViewColumnOrder: (level, columnOrder) => {
        dispatch(xtremRedux.actions.setTableViewColumnOrder(props.screenId, props.elementId, level, columnOrder));
    },
    setTableViewFilter: (level, filter) => {
        dispatch(xtremRedux.actions.setTableViewFilter(props.screenId, props.elementId, level, filter));
    },
    setTableViewGrouping: (level, grouping) => {
        dispatch(xtremRedux.actions.setTableViewGrouping(props.screenId, props.elementId, level, grouping));
    },
    setTableViewOptionsMenuItem: (level, optionsMenuItem) => {
        dispatch(xtremRedux.actions.setTableViewOptionMenuItem(props.screenId, props.elementId, level, optionsMenuItem));
    },
    setTableViewOptionsMenuItemAndViewFilter: (level, optionsMenuItem, filter) => {
        dispatch(xtremRedux.actions.setTableViewOptionsMenuItemAndViewFilter(props.screenId, props.elementId, level, optionsMenuItem, filter));
    },
    setTableViewSortOrder: (level, sortOrder) => {
        dispatch(xtremRedux.actions.setTableViewSortOrder(props.screenId, props.elementId, level, sortOrder));
    },
});
export const ConnectedDesktopTableComponent = connect(mapStateToProps, mapDispatchToProps)(DesktopTableComponent);
export default ConnectedDesktopTableComponent;
//# sourceMappingURL=desktop-table-component.js.map