import { Datetime, DateValue, formatDateToCurrentLocale, datePropertyValueToDateString } from '@sage/xtrem-date-time';
import { intervalToDuration } from 'date-fns';
import { getStore } from '../../../redux';
import { isDateOrDateTime } from '../../../utils/formatters';
function addWeeks(duration) {
    const extras = duration.days >= 7 ? { weeks: Math.trunc(duration.days / 7), days: duration.days % 7 } : { weeks: 0 };
    return { ...duration, ...extras };
}
const getFormattedValue = (date) => {
    const valueType = isDateOrDateTime(date);
    if (valueType === 'date') {
        return DateValue.parse(datePropertyValueToDateString(date));
    }
    return Datetime.parse(datePropertyValueToDateString(date, true));
};
export function toRelative_(type, lang, end, start = new Date(Date.now())) {
    const startDate = getFormattedValue(start);
    const endDate = getFormattedValue(end);
    /* eslint-enable */
    const direction = Math.sign(endDate.toJsDate().getTime() - startDate.toJsDate().getTime());
    if (direction < 1) {
        const format = (value, unit) => {
            // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Intl/RelativeTimeFormat/RelativeTimeFormat#parameters
            const intlRelativeTimeFormat = new Intl.RelativeTimeFormat(lang, {
                numeric: 'auto',
                localeMatcher: 'best fit',
                style: 'long',
            });
            return intlRelativeTimeFormat.format(value, unit);
        };
        const duration = addWeeks(intervalToDuration({ start: startDate.toJsDate().getTime(), end: endDate.toJsDate().getTime() }));
        const { years, months, weeks, days, hours, minutes, seconds } = duration;
        if (years > 0) {
            return format(direction * years, 'year');
        }
        if (months > 0) {
            return format(direction * months, 'month');
        }
        if (weeks > 0) {
            return format(direction * weeks, 'week');
        }
        if (type === 'date') {
            const startDateValue = DateValue.parse(startDate.toString().substring(0, 10));
            const endDateValue = DateValue.parse(endDate.toString().substring(0, 10));
            const day = startDateValue.daysDiff(endDateValue);
            return format(direction * day, 'day');
        }
        if (days > 0) {
            return format(direction * days, 'day');
        }
        if (hours > 0) {
            return format(direction * hours, 'hour');
        }
        if (minutes > 0) {
            return format(direction * minutes, 'minute');
        }
        if (seconds >= 0) {
            if (seconds < 20) {
                return format(0, 'second');
            }
            return format(direction * seconds, 'second');
        }
    }
    return false;
}
export function toRelative(type, end) {
    const state = getStore().getState();
    return toRelative_(type, state.applicationContext?.locale || 'en-US', end);
}
export const getRelativeDateDisplayValue = (value, scope) => {
    let correctedScope = scope || null;
    if (!correctedScope) {
        correctedScope = isDateOrDateTime(value);
    }
    return toRelative(correctedScope ?? 'datetime', value) || formatDateToCurrentLocale(value);
};
//# sourceMappingURL=relative-date-utils.js.map