import { get, set } from 'lodash';
import { convertDeepBindToPathNotNull } from '../../../utils/nested-field-utils';
import { splitValueToMergedValue } from '../../../utils/transformers';
import { FieldKey } from '../../types';
import { isFieldDisabled, isFieldReadOnly } from '../carbon-helpers';
export function getReferenceImage({ value, fieldProperties: { imageField }, }) {
    if (!value || !imageField) {
        return undefined;
    }
    return get(value, String(imageField), undefined);
}
export function getReferenceSearchText({ value, fieldProperties: { valueField }, }) {
    if (!value || !valueField || !get(value, getReferencePath(valueField), '')) {
        return '';
    }
    return String(get(value, getReferencePath(valueField), ''));
}
export function getReferenceHelperText({ value, fieldProperties: { helperTextField, helperText }, }) {
    if (!value || !helperTextField) {
        return helperText;
    }
    return get(value, getReferencePath(helperTextField), helperText);
}
export function getReferenceTunnelLinkId({ fieldProperties, value, }) {
    if (!value) {
        return undefined;
    }
    if (fieldProperties.tunnelPageIdField) {
        const fieldPath = convertDeepBindToPathNotNull(fieldProperties.tunnelPageIdField);
        const _id = get(value, fieldPath);
        if (_id) {
            return _id;
        }
        return undefined;
    }
    return value._id;
}
export function isReferenceDisabled({ screenId, fieldProperties, value, rowValue, }) {
    return isFieldDisabled(screenId, fieldProperties, value, rowValue ? splitValueToMergedValue(rowValue) : rowValue);
}
export function isReferenceReadOnly({ screenId, fieldProperties, value, rowValue, contextType, }) {
    return isFieldReadOnly(screenId, fieldProperties, value, rowValue ? splitValueToMergedValue(rowValue) : rowValue, contextType);
}
export function hasHelperTextField({ screenId, fieldProperties, value, rowValue, contextType, }) {
    return Boolean(!isReferenceReadOnly({
        screenId,
        fieldProperties,
        value,
        rowValue: rowValue ? splitValueToMergedValue(rowValue) : rowValue,
        contextType,
    }) &&
        !isReferenceDisabled({ screenId, fieldProperties, value, rowValue }) &&
        !fieldProperties.isHelperTextHidden);
}
export function hasLookupIcon({ screenId, fieldProperties, value, rowValue, contextType, }) {
    return Boolean(!isReferenceReadOnly({
        screenId,
        fieldProperties,
        value: value,
        rowValue: rowValue ? splitValueToMergedValue(rowValue) : rowValue,
        contextType,
    }) &&
        !isReferenceDisabled({ screenId, fieldProperties, value: value, rowValue }) &&
        fieldProperties.columns &&
        fieldProperties.columns.length > 0);
}
export function getReferenceSelectedRecord({ value, fieldProperties, }) {
    const searchText = getReferenceSearchText({ value, fieldProperties });
    const cleanValue = value ? splitValueToMergedValue(value) : value;
    return searchText
        ? {
            value: searchText,
            id: value._id ?? value.id,
            image: getReferenceImage({ value: cleanValue, fieldProperties }),
            helperText: getReferenceHelperText({ value: cleanValue, fieldProperties }),
            displayedAs: searchText,
            __collectionItem: cleanValue,
        }
        : undefined;
}
export function nodeToSelectItem({ collectionItem, fieldProperties: { valueField, helperTextField, imageField }, }) {
    const value = get(collectionItem, convertDeepBindToPathNotNull(valueField));
    const helperText = helperTextField
        ? get(collectionItem, convertDeepBindToPathNotNull(helperTextField))
        : null;
    const displayedAs = value;
    return {
        id: collectionItem._id ?? collectionItem.id,
        displayedAs,
        helperText,
        image: imageField
            ? get(collectionItem, convertDeepBindToPathNotNull(imageField))
            : undefined,
        value,
        __collectionItem: collectionItem,
    };
}
export function nodesToSelectItems({ nodes, fieldProperties, }) {
    if (!nodes) {
        return [];
    }
    return nodes.map(collectionItem => nodeToSelectItem({ collectionItem, fieldProperties }));
}
export const getReferenceValueField = (fieldProperties) => {
    return getReferencePath(fieldProperties.valueField);
};
export const getReferenceValueFieldPath = (fieldProperties) => {
    const valueFieldPath = getReferencePath(fieldProperties.valueField);
    if (valueFieldPath) {
        return `${convertDeepBindToPathNotNull(fieldProperties.bind)}.${valueFieldPath}`;
    }
    return convertDeepBindToPathNotNull(fieldProperties.bind);
};
export const getReferenceOrderBy = (fieldProperties) => {
    if (fieldProperties.orderBy) {
        const orderBy = Object.prototype.hasOwnProperty.call(fieldProperties.orderBy, '_id')
            ? fieldProperties.orderBy
            : { ...fieldProperties.orderBy, _id: 1 };
        return orderBy;
    }
    const firstColumn = fieldProperties.columns?.filter(c => c.type !== FieldKey.Image)?.[0];
    if (firstColumn) {
        const bind = firstColumn.type === FieldKey.Reference
            ? `${convertDeepBindToPathNotNull(firstColumn.properties.bind)}.${getReferencePath(firstColumn.properties.valueField)}`
            : convertDeepBindToPathNotNull(firstColumn.properties.bind);
        const orderBy = set({}, convertDeepBindToPathNotNull(bind), 1);
        if (bind !== '_id') {
            orderBy._id = 1;
        }
        return orderBy;
    }
    if (fieldProperties.valueField) {
        const orderBy = set({}, convertDeepBindToPathNotNull(fieldProperties.valueField), 1);
        if (fieldProperties.valueField !== '_id') {
            orderBy._id = 1;
        }
        return orderBy;
    }
    return { _id: 1 };
};
export function getReferencePath(value) {
    if (!value) {
        return undefined;
    }
    if (typeof value === 'string' || typeof value === 'number') {
        return String(value);
    }
    let path = '';
    const getPath = (obj) => {
        if (typeof obj !== 'object') {
            return;
        }
        Object.keys(obj).forEach(key => {
            path += `${key}.`;
            getPath(obj[key]);
        });
    };
    getPath(value);
    return path.slice(0, -1);
}
export const isStringReferenceValueField = (valueField) => typeof valueField === 'string';
//# sourceMappingURL=reference-utils.js.map