import { get, isEqual } from 'lodash';
import React from 'react';
import { convertDeepBindToPathNotNull } from '../../../utils/nested-field-utils';
import { AUTO_COLUMN_ID } from '../../../utils/table-component-utils';
import { getReferenceValueFieldPath } from './reference-utils';
import { getImageUrlFromValue } from '../image/image-utils';
import Link from 'carbon-react/esm/components/link';
import { splitValueToMergedValue } from '../../../utils/transformers';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import * as xtremRedux from '../../../redux';
import { connect } from 'react-redux';
import { getElementAccessStatusWithoutId } from '../../../utils/access-utils';
import { TUNNEL_LINK_CLASS } from '../../../utils/constants';
export class ReferenceCellRenderer extends React.Component {
    constructor() {
        super(...arguments);
        this.linkRef = React.createRef();
        this.onFocus = () => {
            if (this.shouldRenderTunnelPageLink()) {
                this.linkRef.current?.focus();
            }
        };
        /**
         * Searches for the reference field image from the row using the the dot path, first using the bind path of the reference column, then the path of the `imageField`;
         * @returns
         */
        this.getImageFromProps = () => {
            if (!this.props.fieldProperties.imageField) {
                return undefined;
            }
            return (get(this.props.data, `${this.props.colDef.field}.${convertDeepBindToPathNotNull(this.props.fieldProperties.imageField)}`) || null);
        };
        this.shouldRenderTunnelPageLink = () => (this.props.isTableReadOnly || this.isReadOnly()) && !!this.props.hasAccessToTunnelPage;
        this.isReadOnly = () => resolveByValue({
            screenId: this.props.screenId,
            propertyValue: this.props.fieldProperties.isReadOnly,
            rowValue: splitValueToMergedValue(this.props.data),
            fieldValue: this.props.value,
            skipHexFormat: true,
        });
        this.onTunnelLinkClick = async (ev) => {
            if (ev.ctrlKey || ev.metaKey) {
                return;
            }
            ev.preventDefault();
            const referenceValue = get(this.props.data, convertDeepBindToPathNotNull(this.props.fieldProperties.bind)) || null;
            const newValue = await this.props.openTunnel(this.props.fieldProperties, referenceValue);
            // Only update the value if the tunnel results some changes. If null returned, no changes were done.
            if (newValue && this.props.setValue) {
                this.props.setValue(newValue);
            }
        };
    }
    componentDidMount() {
        if (this.props.eGridCell && this.shouldRenderTunnelPageLink()) {
            this.props.eGridCell.addEventListener('focus', this.onFocus);
        }
    }
    shouldComponentUpdate(nextProps) {
        const isMandatory = this.props.fieldProperties.isMandatory ?? false;
        const validationErrors = this.props
            .validationErrors ?? false;
        if (!isEqual(this.props.value, nextProps.value)) {
            return true;
        }
        // Trigger updates when validation is updated:
        if (isMandatory && !this.props.value) {
            return true;
        }
        if (!isEqual(validationErrors, nextProps
            .validationErrors)) {
            return true;
        }
        return false;
    }
    componentWillUnmount() {
        if (this.props.eGridCell && this.shouldRenderTunnelPageLink()) {
            this.props.eGridCell.removeEventListener('focus', this.onFocus);
        }
    }
    getTunnelPageLink() {
        const referencedObjectId = get(this.props.data, `${convertDeepBindToPathNotNull(this.props.fieldProperties.bind)}._id`);
        return `${this.props.fieldProperties.tunnelPage}/${btoa(JSON.stringify({ _id: referencedObjectId }))}`;
    }
    render() {
        if (!this.props.data ||
            (!this.props.isTree && !this.props.data.__isGroup && this.props.colDef.colId === AUTO_COLUMN_ID)) {
            return null;
        }
        // If group append children count
        const displayValue = this.props.valueFormatted ?? '';
        const image = this.getImageFromProps();
        if (this.props.node?.footer) {
            return (React.createElement(this.props.fieldProperties.wrapper, { ...this.props },
                React.createElement("strong", null, get(this.props.data, `${convertDeepBindToPathNotNull(this.props.fieldProperties.bind)}._id`))));
        }
        const shouldHideValue = !!this.props.data.__isGroup &&
            this.props.colDef.colId !== AUTO_COLUMN_ID &&
            this.props.data.__groupKey === getReferenceValueFieldPath(this.props.fieldProperties);
        return (React.createElement(this.props.fieldProperties.wrapper, { ...this.props },
            React.createElement("div", { style: { display: 'flex', alignItems: 'center' } },
                image && (React.createElement("img", { src: getImageUrlFromValue(image.value), style: { width: '24px', height: '24px', paddingRight: '10px' }, alt: displayValue })),
                this.shouldRenderTunnelPageLink() && !shouldHideValue && (React.createElement("div", { "data-testid": `${this.props.tableElementId}-${this.props.node.rowIndex}` },
                    React.createElement(Link, { onClick: this.onTunnelLinkClick, variant: "neutral", href: this.getTunnelPageLink(), ref: this.linkRef, className: TUNNEL_LINK_CLASS }, displayValue))),
                !this.shouldRenderTunnelPageLink() && (React.createElement("span", { "data-testid": `${this.props.tableElementId}-${this.props.node.rowIndex}-${this.props.api.getColumns().indexOf(this.props.column) + 1}`, style: { textOverflow: 'ellipsis', overflow: 'hidden' } }, shouldHideValue ? '' : displayValue)))));
    }
}
const mapStateToProps = (state, props) => {
    const screenDefinition = state.screenDefinitions[props.screenId];
    const hasAccessToTunnelPage = !!props.fieldProperties.tunnelPage &&
        getElementAccessStatusWithoutId(screenDefinition.accessBindings, {
            node: String(props.fieldProperties.node),
            bind: '$read',
        }) === 'authorized';
    return {
        ...props,
        hasAccessToTunnelPage,
        openTunnel: xtremRedux.actions.actionStub,
    };
};
const extendedMapDispatchToProps = (dispatch, props) => {
    return {
        openTunnel: (fieldProperties, value) => dispatch(xtremRedux.actions.openTunnel({
            elementId: props.elementId,
            parentElementId: props.tableElementId,
            screenId: props.screenId,
            recordContext: props.data,
            contextNode: props.contextNode,
            fieldProperties,
            value,
        })),
    };
};
export const ConnectedReferenceCellRenderer = connect(mapStateToProps, extendedMapDispatchToProps)(ReferenceCellRenderer);
export default ConnectedReferenceCellRenderer;
//# sourceMappingURL=reference-cell-renderer.js.map