import React from 'react';
import { localize } from '../../../service/i18n-service';
import { defaultCellEditorStyles, getInitialCellEditorState } from '../../../utils/ag-grid/ag-grid-cell-editor-utils';
import { parseLocalizedNumberStringToNumber } from '../../../utils/formatters';
import { splitValueToMergedValue } from '../../../utils/transformers';
import { onNumericInputKeyDown } from './numeric-cell-utils';
export default class NumericEditor extends React.Component {
    constructor(props) {
        super(props);
        this.state = getInitialCellEditorState({
            eventKey: props.eventKey,
            initialValue: props.initialValue,
            isNumeric: true,
        });
        this.input = React.createRef();
        this.onKeyDown = this.onKeyDown.bind(this);
        this.handleChange = this.handleChange.bind(this);
        // We need to call onValueChange to reflect the getInitialCellEditorState value to the parent, for example if the user starts editing inserting a number to not lose it.
        this.props.onValueChange(this.state.value);
    }
    componentDidMount() {
        const input = this.input.current;
        if (!input) {
            return;
        }
        input.focus();
        if (this.state.highlightOnFocus) {
            input.select();
            this.setState({
                highlightOnFocus: false,
            });
        }
        else {
            // when we started editing, we want the caret at the end, not the start.
            // this comes into play in two scenarios: a) when user hits F2 and b)
            // when user hits a printable character, then on IE (and only IE) the caret
            // was placed after the first character, thus 'apply' would end up as 'pplea'
            const length = input.value ? input.value.length : 0;
            if (length > 0) {
                // The selection cannot be set on number input types, so we temporarily switch the type to text.
                input.setSelectionRange(length, length);
            }
        }
        input.addEventListener('keydown', this.onKeyDown);
    }
    componentWillUnmount() {
        this.input.current?.removeEventListener('keydown', this.onKeyDown);
    }
    handleChange(event) {
        this.setState({ value: event.target.value });
        if (event.target.value === '') {
            this.props.onValueChange(null);
        }
        else {
            this.props.onValueChange(parseLocalizedNumberStringToNumber(event.target.value, localize('@sage/xtrem-ui/number-format-separator', '.')));
        }
    }
    onKeyDown(event) {
        const scale = this.props.colDef?.cellEditorParams?.fieldProperties?.scale;
        onNumericInputKeyDown({
            event,
            screenId: this.props.screenId,
            scale,
            fieldValue: this.props.value,
            rowValue: splitValueToMergedValue(this.props.data),
        });
    }
    render() {
        return (React.createElement("div", { className: "ag-cell-edit-wrapper" },
            React.createElement("input", { className: "e-numeric-cell-editor", "data-testid": `${this.props.tableElementId}-${this.props.node.rowIndex}-${this.props.api.getColumns().indexOf(this.props.column) + 1}`, ref: this.input, type: "text", value: this.state.value, onChange: this.handleChange, style: defaultCellEditorStyles })));
    }
}
//# sourceMappingURL=numeric-cell-editor.js.map