import * as React from 'react';
import { connect } from 'react-redux';
import { EditableFieldBaseComponent, mapDispatchToProps, mapStateToProps } from '../field-base-component';
import FileInput from 'carbon-react/esm/components/file-input';
import { localize } from '../../../service/i18n-service';
import { getDataTestIdAttribute, getComponentClass } from '../../../utils/dom';
import { confirmationDialog, errorDialog } from '../../../service/dialog-service';
import { ContextType } from '../../../types';
import { AsyncTableComponent } from '../table/async-table-component';
import { getPageDefinitionFromState } from '../../../utils/state-utils';
import { getScreenElement } from '../../../service/screen-base-definition';
import { depositFileToCollectionValue, getMultiFileDepositFieldColumns } from './multi-file-deposit-utils';
import { STATUS_CREATED, STATUS_UPLOADING, getMimeTypeFromExtension, isDisallowedMimeType, showNotAllowedTypeToast, } from '../../../utils/file-deposit-utils';
import { RecordActionType } from '../../../service/collection-data-types';
export class MultiFileDepositComponent extends EditableFieldBaseComponent {
    constructor(props) {
        super(props);
        this.onChange = (fileList) => {
            if (!fileList.length) {
                return;
            }
            const targetNode = this.props.graphApi.node(this.props.fieldProperties.attachmentNode);
            Array.from(fileList).forEach(file => {
                if (!this.props.value) {
                    return;
                }
                const mimeType = file.type || getMimeTypeFromExtension(file.name);
                if (this.props.fieldProperties.fileTypes &&
                    isDisallowedMimeType(this.props.fieldProperties.fileTypes, mimeType, file.name)) {
                    this.props.onTelemetryEvent?.(`multiFileDepositFileMimeTypeValidationFailed-${this.props.elementId}`, {
                        elementId: this.props.elementId,
                        screenId: this.props.screenId,
                        mimeType,
                    });
                    showNotAllowedTypeToast(mimeType);
                    return;
                }
                this.props.onTelemetryEvent?.(`multiFileDepositFileAdded-${this.props.elementId}`, {
                    mimeType,
                    method: 'fileBrowserSelection',
                    elementId: this.props.elementId,
                    screenId: this.props.screenId,
                });
                // We intentionally ignore the promise here, as we don't want to sequentially upload files
                depositFileToCollectionValue({
                    value: this.props.value,
                    file,
                    targetNodeMutations: targetNode.mutations,
                }).catch(err => {
                    errorDialog(this.props.screenId, localize('@sage/xtrem-ui/file-upload-failed', 'Failed to upload file.'), err);
                });
            });
        };
        this.cancelFileUploading = (recordId) => {
            const tokenSource = this.props.value?.getRawRecord({ id: recordId }).cancelTokenSource;
            if (tokenSource) {
                tokenSource.cancel(`Upload cancelled for recordId: ${recordId}`);
            }
        };
        this.toggleDropzone = () => {
            this.setState(prevState => ({
                isFileInputVisible: !prevState.isFileInputVisible,
            }));
        };
        this.renderUploadedFilesTable = () => {
            if (!this.props.value) {
                return null;
            }
            const inlineActions = [
                {
                    icon: 'cross_circle',
                    title: localize('@sage/xtrem-ui/multi-file-deposit-cancel', 'Cancel'),
                    isDestructive: true,
                    isDisplayed: (_id, rowItem) => {
                        return (rowItem?.attachment.status === STATUS_CREATED || rowItem?.attachment.status === STATUS_UPLOADING);
                    },
                    onClick: (_id, rowItem) => this.displayCancelUploadDialog(rowItem._id),
                },
                {
                    icon: 'bin',
                    title: localize('@sage/xtrem-ui/multi-file-deposit-remove', 'Remove'),
                    isDestructive: true,
                    isDisabled: (_id, rowItem) => {
                        return (rowItem?.attachment.status === STATUS_CREATED || rowItem?.attachment.status === STATUS_UPLOADING);
                    },
                    onClick: (_id, rowItem) => this.displayRemoveFileDialog(rowItem._id),
                },
            ];
            const additionalFieldActions = [
                {
                    id: 'toggle',
                    icon: 'upload',
                    title: this.state.isFileInputVisible
                        ? localize('@sage/xtrem-ui/file-component-hide-upload-area', 'Hide upload area')
                        : localize('@sage/xtrem-ui/file-component-show-upload-area', 'Show upload area'),
                    onClick: this.toggleDropzone,
                },
            ];
            return (React.createElement(AsyncTableComponent, { accessBindings: this.props.accessBindings, browser: this.props.browser, contextType: ContextType.table, dataTypes: this.props.dataTypes, elementId: this.props.elementId, enumTypes: this.props.enumTypes, fieldProperties: {
                    ...this.props.fieldProperties,
                    isChangeIndicatorDisabled: true,
                    columns: getMultiFileDepositFieldColumns(),
                    inlineActions,
                    additionalFieldActions,
                    pageSize: 8,
                }, isUsingInfiniteScroll: true, locale: this.props.locale, nodeTypes: this.props.nodeTypes, screenId: this.props.screenId, setFieldProperties: this.props.setFieldProperties, tableViews: {}, validationErrors: [], value: this.props.value }));
        };
        const value = this.props.value;
        this.state = {
            hasValue: !!value && value.getData().length > 0,
            isFileInputVisible: !value || value.getData().length === 0,
        };
    }
    componentDidMount() {
        this.changeSubscription = this.props.value?.subscribeForValueChanges(type => {
            if (type === RecordActionType.ADDED && !this.state.hasValue) {
                this.setState({ hasValue: true, isFileInputVisible: false });
            }
        });
    }
    componentWillUnmount() {
        this.changeSubscription?.();
    }
    async displayCancelUploadDialog(recordId) {
        const confirmation = await confirmationDialog(this.props.screenId, 'warn', localize('@sage/xtrem-ui/multi-file-upload-cancel-title', 'Cancel upload'), localize('@sage/xtrem-ui/multi-file-upload-cancel', 'Cancel file upload in progress.'), {
            acceptButton: {
                text: localize('@sage/xtrem-ui/continue', 'Continue'),
            },
            cancelButton: {
                text: localize('@sage/xtrem-ui/cancel', 'Cancel'),
            },
            size: 'small',
        });
        if (confirmation) {
            this.props.onTelemetryEvent?.(`multiFileDepositFileUploadCancelled-${this.props.elementId}`, {
                elementId: this.props.elementId,
                screenId: this.props.screenId,
            });
            this.cancelFileUploading(recordId);
        }
    }
    async displayRemoveFileDialog(recordId) {
        const confirmation = await confirmationDialog(this.props.screenId, 'warn', localize('@sage/xtrem-ui/multi-file-upload-remove-title', 'Remove file'), localize('@sage/xtrem-ui/multi-file-upload-remove', 'Remove uploaded file.'), {
            size: 'small',
        });
        if (confirmation) {
            this.props.onTelemetryEvent?.(`multiFileDepositFileRemoved-${this.props.elementId}`, {
                elementId: this.props.elementId,
                screenId: this.props.screenId,
            });
            this.props.value?.removeRecord({ recordId });
        }
    }
    render() {
        const title = this.getTitle();
        const fieldProperties = this.props.fieldProperties;
        const dragAndDropText = !this.state.hasValue
            ? localize('@sage/xtrem-ui/file-component-drag-drop-empty', 'or \n This list has no attached files yet. \n Drag and drop your file here.')
            : localize('@sage/xtrem-ui/file-component-drag-drop', 'or \n Drag and drop your file here');
        return (React.createElement("div", { "data-testid": getDataTestIdAttribute('multi-file-deposit', title, this.props.elementId), className: getComponentClass(this.props, 'e-multi-file-deposit-field') },
            this.state.isFileInputVisible && (React.createElement(FileInput, { "data-testid": "e-multi-file-deposit-field-upload-area", maxWidth: "100%", isVertical: true, onChange: this.onChange, accept: fieldProperties.fileTypes, buttonText: localize('@sage/xtrem-ui/file-component-browse-file', 'Browse to File'), dragAndDropText: dragAndDropText, minHeight: this.state.hasValue ? '200px' : '400px' })),
            this.state.hasValue && this.renderUploadedFilesTable()));
    }
}
export const ConnectedMultiFileDepositComponent = connect((state, props) => {
    const screenDefinition = getPageDefinitionFromState(props.screenId, state);
    const screenElement = getScreenElement(screenDefinition);
    const basicProps = mapStateToProps()(state, props);
    return {
        ...props,
        ...basicProps,
        accessBindings: screenDefinition.accessBindings || {},
        enumTypes: state.enumTypes,
        nodeTypes: state.nodeTypes,
        dataTypes: state.dataTypes,
        graphApi: screenElement.$.graph,
        browser: state.browser,
        onTelemetryEvent: state.applicationContext?.onTelemetryEvent,
    };
}, mapDispatchToProps())(MultiFileDepositComponent);
export default ConnectedMultiFileDepositComponent;
//# sourceMappingURL=multi-file-deposit-component.js.map