import * as React from 'react';
import { connect } from 'react-redux';
import { localize } from '../../../service/i18n-service';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { Portrait } from '../../ui/portrait-component';
import { StyledIconButton } from '../../ui/styled-icon-button';
import { FieldLabel, HelperText } from '../carbon-utility-components';
import { EditableFieldBaseComponent, mapDispatchToProps, mapReadonlyStateToProps } from '../field-base-component';
import { handleChange } from '../../../utils/abstract-fields-utils';
import { getImageUrlFromValue } from './image-utils';
import { ContextType } from '../../../types';
import ButtonMinor from 'carbon-react/esm/components/button-minor';
export class ImageComponent extends EditableFieldBaseComponent {
    constructor(props) {
        super(props);
        this.getPlaceholderValue = () => resolveByValue({
            propertyValue: this.props.fieldProperties.placeholderValue,
            skipHexFormat: true,
            screenId: this.props.screenId,
            // Intentionally undefined, the validation rule should not depend on the value that is validated by that very rule
            rowValue: undefined,
            // Pass in the row value as the field value because the field value is passed in as the first argument to the functional code
            fieldValue: this.props.handlersArguments?.rowValue,
        });
        this.getPlaceholderMode = () => resolveByValue({
            propertyValue: this.props.fieldProperties.placeholderMode,
            skipHexFormat: true,
            screenId: this.props.screenId,
            // Intentionally undefined, the validation rule should not depend on the value that is validated by that very rule
            rowValue: undefined,
            // Pass in the row value as the field value because the field value is passed in as the first argument to the functional code
            fieldValue: this.props.handlersArguments?.rowValue,
        });
        this.onDragOver = (event) => {
            event.stopPropagation();
            event.preventDefault();
        };
        this.onDrop = (event) => {
            event.preventDefault();
            if (event.dataTransfer.items) {
                if (event.dataTransfer.items && event.dataTransfer.items.length > 0 && event.dataTransfer.items[0].kind) {
                    const file = event.dataTransfer.items[0].getAsFile();
                    if (file) {
                        this.readFile(file);
                    }
                }
                else if (event.dataTransfer.files.length) {
                    this.readFile(event.dataTransfer.files[0]);
                }
            }
        };
        this.onCreateElementClick = () => this.fileRef.current?.click();
        this.onFileChanged = (event) => {
            if (event.target.files && event.target.files.length > 0) {
                this.readFile(event.target.files[0]);
            }
        };
        this.onRemoveImage = () => handleChange(this.props.elementId, null, this.props.setFieldValue, this.props.validate, this.triggerChangeListener);
        this.onFullScreenOpen = () => this.setState({ isZooming: true });
        this.onFullScreenClose = () => this.setState({ isZooming: false });
        this.renderRemoveIcon = () => (React.createElement("div", { className: "e-image-field-remove" },
            React.createElement(StyledIconButton, { "data-testid": "e-image-field-remove", onClick: this.onRemoveImage, iconType: "cross", iconTooltipMessage: localize('@sage/xtrem-ui/image-field-remove-image', 'Remove'), "aria-label": localize('@sage/xtrem-ui/image-field-remove-image', 'Remove'), buttonType: "tertiary" })));
        this.renderReadOnlyPlaceholder = () => (React.createElement("div", { className: "e-image-field-placeholder", "data-testid": "e-image-field-placeholder" },
            React.createElement(Portrait, { size: "M", placeholderMode: this.getPlaceholderMode(), placeholderValue: this.getPlaceholderValue(), icon: "image" })));
        this.renderValueBody = (width) => (React.createElement("div", { onDoubleClick: !this.isDisabled() ? this.onFullScreenOpen : undefined },
            !this.props.isNested && (React.createElement("img", { src: getImageUrlFromValue(this.getValue()?.value), alt: this.getTitle() || this.getPlaceholderValue() || '', width: width, height: this.props.fieldProperties.height ? this.props.fieldProperties.height : 'auto' })),
            this.props.isNested && (React.createElement(Portrait, { image: { value: `${this.getValue()?.value}` }, placeholderValue: this.getTitle() || this.getPlaceholderValue() || '', shape: this.props.fieldProperties.shape, size: "M" }))));
        this.renderFullScreenView = () => (React.createElement("div", { className: "e-image-field-full-screen", onDoubleClick: !this.isDisabled() ? this.onFullScreenClose : undefined },
            React.createElement("img", { src: getImageUrlFromValue(this.getValue()?.value), alt: this.getTitle() || this.getPlaceholderValue() || '' })));
        this.fileRef = React.createRef();
        this.state = { isZooming: false };
    }
    readFile(file) {
        if (file && file.type.indexOf('image/') === 0) {
            const reader = new FileReader();
            reader.onload = () => {
                if (reader.result) {
                    const components = reader.result.toString().split(',');
                    if (components.length > 1) {
                        handleChange(this.props.elementId, { value: components[1] }, this.props.setFieldValue, this.props.validate, this.triggerChangeListener);
                    }
                }
            };
            reader.readAsDataURL(file);
        }
    }
    renderUploadArea(width) {
        const uploadBoxHeight = this.props.fieldProperties.height ? this.props.fieldProperties.height : '48px';
        return (React.createElement("div", { style: { width, height: uploadBoxHeight, lineHeight: uploadBoxHeight }, onDrop: this.onDrop, onDragOver: this.onDragOver },
            React.createElement(ButtonMinor, { buttonType: "secondary", iconType: "plus", iconPosition: "after", fullWidth: true, onClick: !this.isDisabled() ? this.onCreateElementClick : undefined, disabled: this.isDisabled() },
                React.createElement("label", { htmlFor: "inputFile", onClick: (e) => e.preventDefault() }, localize('@sage/xtrem-ui/image-component-add-image', 'Add an image')),
                React.createElement("input", { id: "inputFile", type: "file", className: "e-hidden", ref: this.fileRef, accept: "image/*", onChange: this.onFileChanged }))));
    }
    render() {
        let width;
        if (this.props.fieldProperties.width) {
            width = this.props.fieldProperties.width;
        }
        else if (this.props.fieldProperties.isFullWidth) {
            width = '100%';
        }
        else {
            width = 'auto';
        }
        const value = this.getValue();
        const hasValue = value && value.value;
        const isReadOnly = this.isReadOnly();
        const isDisabled = this.isDisabled();
        const title = this.getTitle();
        const hasTitle = !this.props.fieldProperties.isTitleHidden && title !== '' && title !== undefined;
        const isTable = this.props.contextType === ContextType.table || this.props.contextType === ContextType.tableSummary;
        return (React.createElement("div", { ...this.getBaseAttributesDivWrapper('image', 'e-image-field', this.props.contextType, this.props.handlersArguments?.rowValue, this.props.isNested), onClick: !isDisabled ? this.getClickHandler() : undefined },
            hasTitle && React.createElement(FieldLabel, { label: title }),
            React.createElement("div", { className: "e-image-field-content-wrapper" },
                hasValue && this.renderValueBody(width),
                hasValue && this.state.isZooming && this.renderFullScreenView(),
                hasValue && !isReadOnly && !isDisabled && !isTable && this.renderRemoveIcon(),
                !hasValue && isReadOnly && this.renderReadOnlyPlaceholder(),
                !hasValue && !isReadOnly && this.renderUploadArea(width)),
            React.createElement(HelperText, { helperText: this.props.fieldProperties.helperText })));
    }
}
export const ConnectedImageComponent = connect(mapReadonlyStateToProps(), mapDispatchToProps())(ImageComponent);
export default ConnectedImageComponent;
//# sourceMappingURL=image-component.js.map