import { FieldKey } from '@sage/xtrem-shared';
import { isEqual, set, uniq } from 'lodash';
import * as React from 'react';
import { connect } from 'react-redux';
import * as xtremRedux from '../../../redux';
import { getScreenElement } from '../../../service/screen-base-definition';
import { handleChange } from '../../../utils/abstract-fields-utils';
import { convertDeepBindToPathNotNull } from '../../../utils/nested-field-utils';
import { calculateContainerWidth } from '../../../utils/responsive-utils';
import { Pod } from '../../ui/pod/pod-component';
import { EditableFieldBaseComponent, mapDispatchToProps, mapStateToProps } from '../field-base-component';
import { schemaTypeNameFromNodeName } from '../../../utils/transformers';
import Loader from 'carbon-react/esm/components/loader';
import { createDynamicPodReferenceField } from '../../../utils/data-type-utils';
export class DynamicPodComponent extends EditableFieldBaseComponent {
    constructor(props) {
        super(props);
        this.onChange = async (bind, value) => {
            const newValue = { ...this.props.value };
            set(newValue, bind, value);
            handleChange(this.props.elementId, newValue, this.props.setFieldValue, this.props.validate, this.triggerChangeListener);
        };
        this.onRemove = () => handleChange(this.props.elementId, null, this.props.setFieldValue, this.props.validate, this.triggerChangeListener);
        this.onNewPod = async () => {
            const columns = this.props.fieldProperties.columns;
            let value = columns?.reduce((prev, curr) => {
                return set(prev, convertDeepBindToPathNotNull(curr.properties.bind), null);
            }, {});
            if (this.props.fieldProperties.onAddButtonClick) {
                const state = xtremRedux.getStore().getState();
                const screenDefinition = state.screenDefinitions[this.props.screenId];
                const newValue = await this.props.fieldProperties.onAddButtonClick.apply(getScreenElement(screenDefinition));
                value = { ...value, ...newValue };
            }
            handleChange(this.props.elementId, value, this.props.setFieldValue, this.props.validate, this.triggerChangeListener);
        };
        this.state = { fetchingReferenceTypes: true, fetchingOptionTypes: true };
    }
    getMissingOptionTypes() {
        return (this.props.fieldProperties.columns || [])
            .filter(c => !!c.properties.optionType)
            .map(c => c.properties.optionType);
    }
    getMissingReferenceTypes() {
        return uniq((this.props.fieldProperties.columns || [])
            .filter(c => c.type === FieldKey.Reference &&
            c.properties.node &&
            !this.props.nodeTypes?.[schemaTypeNameFromNodeName(String(c.properties.node))])
            .map(c => String(schemaTypeNameFromNodeName(c.properties.node))));
    }
    checkDataModel() {
        const missingOptionTypes = this.getMissingOptionTypes();
        const locale = this.props.locale || 'en-US';
        if (missingOptionTypes.length > 0) {
            this.props.loadEnumTypes(missingOptionTypes, locale);
        }
        else {
            this.setState({ fetchingOptionTypes: false });
        }
        const missingReferenceTypes = this.getMissingReferenceTypes();
        if (missingReferenceTypes.length > 0) {
            this.props.loadDataTypes(missingReferenceTypes, locale).then(() => {
                this.setState({ fetchingReferenceTypes: false });
            });
        }
        else {
            this.setState({ fetchingReferenceTypes: false });
        }
    }
    componentDidMount() {
        this.checkDataModel();
    }
    componentDidUpdate(prevProps) {
        if (!isEqual(prevProps.fieldProperties.columns, this.props.fieldProperties.columns)) {
            this.checkDataModel();
        }
    }
    addReferenceDataTypeInfo() {
        return (this.props.fieldProperties.columns || []).map(c => {
            if (c.properties._controlObjectType === FieldKey.Reference &&
                this.props.nodeTypes &&
                this.props.dataTypes) {
                return createDynamicPodReferenceField({
                    column: c,
                    nodeTypes: this.props.nodeTypes,
                    dataTypes: this.props.dataTypes,
                });
            }
            return c;
        });
    }
    render() {
        const { fieldProperties, browser, availableColumns, value, elementId, screenId } = this.props;
        const podWidth = browser && calculateContainerWidth(browser.is, availableColumns || 12, 'small');
        if (this.state.fetchingOptionTypes || this.state.fetchingReferenceTypes) {
            return React.createElement(Loader, null);
        }
        const podProps = {
            value,
            fieldProperties: { ...fieldProperties, columns: this.addReferenceDataTypeInfo() },
            browser,
            elementId,
            screenId,
            availableColumns: podWidth,
            isDisabled: this.isDisabled(),
            isReadOnly: this.isReadOnly(),
            onBlockClick: this.getClickHandler(),
            onChange: this.onChange,
            baseAttributesDivWrapper: this.getBaseAttributesDivWrapper('dyamic-pod', 'e-dyamic-pod-field', this.props.contextType, this.props.handlersArguments?.rowValue, this.props.isNested),
            onNewPod: this.onNewPod,
            contextType: this.props.contextType,
            onRemove: this.onRemove,
            validationErrors: this.props.validationErrors,
        };
        return React.createElement(Pod, { ...podProps });
    }
}
const extendedMapStateToProps = (state, props) => {
    const componentProperties = mapStateToProps()(state, props);
    const screenDefinition = state.screenDefinitions[props.screenId];
    if (componentProperties.fieldProperties?.onAddButtonClick) {
        componentProperties.fieldProperties.onAddButtonClick =
            componentProperties.fieldProperties.onAddButtonClick.bind(getScreenElement(screenDefinition));
    }
    return {
        ...props,
        ...componentProperties,
        enumTypes: state.enumTypes || {},
        dataTypes: state.dataTypes || {},
        loadEnumTypes: xtremRedux.actions.actionStub,
        loadDataTypes: xtremRedux.actions.actionStub,
    };
};
const extendedMapDispatchToProps = (dispatch, props) => {
    const defaultMapDispatchToProps = mapDispatchToProps()(dispatch, props);
    return {
        ...defaultMapDispatchToProps,
        loadEnumTypes: (optionTypes, locale) => dispatch(xtremRedux.actions.loadEnumType(optionTypes, locale)),
        loadDataTypes: (referenceTypes, locale) => dispatch(xtremRedux.actions.fetchNodeTypes(referenceTypes, locale)),
    };
};
export const ConnectedDynamicPodComponent = connect(extendedMapStateToProps, extendedMapDispatchToProps)(DynamicPodComponent);
export default ConnectedDynamicPodComponent;
//# sourceMappingURL=dynamic-pod-component.js.map