import React from 'react';
import { CalendarGrid, Popover, Calendar, DialogTrigger, I18nProvider } from 'react-aria-components';
import Textbox from 'carbon-react/esm/components/textbox';
import { datetimeToCalendarDate, formatDatetime, makeDatetime } from './datetime-range-utils';
import { useTime } from '../../ui/time/use-time';
import { TimeComponent as TimeComp } from '../../ui/time/time-component';
import { localize } from '../../../service/i18n-service';
import { Time } from '@sage/xtrem-date-time';
import { MonthYearHeaderComponent } from './month-year-header-component';
import { CalendarCellComponent } from './calendar-cell-component';
export function DatetimeInputComponent({ 'aria-label': ariaLabel, screenId, elementId, date, fieldId, inputRef, isDisabled, isPopoverOpen, isReadOnly, isTimeZoneHidden, locale, maxDate, minDate, onDateChange, onPopperOpenChange, onTimeChange, rangeStartDate, time, timeZone = 'UTC', title, type, validationError, }) {
    const [isManuallyClosed, setIsManuallyClosed] = React.useState(false);
    const popoverRef = React.useRef(null);
    const handleDateChange = React.useCallback((newDate) => {
        onDateChange(newDate);
    }, [onDateChange]);
    const handleTimeChange = React.useCallback((newTime) => {
        onTimeChange(newTime);
    }, [onTimeChange]);
    const isDateUnavailable = (startDate, minDatetime, maxDatetime) => (calendarDate) => {
        if (minDatetime && calendarDate.compare(datetimeToCalendarDate(minDatetime, 'UTC')) < 0)
            return true;
        if (maxDatetime && calendarDate.compare(datetimeToCalendarDate(maxDatetime, 'UTC')) > 0)
            return true;
        if (startDate && calendarDate.compare(startDate) < 0)
            return true;
        return false;
    };
    const rangeStartCalendarDate = rangeStartDate;
    const minDateValue = minDate ? datetimeToCalendarDate(minDate, 'UTC') : undefined;
    const maxDateValue = maxDate ? datetimeToCalendarDate(maxDate, 'UTC') : undefined;
    const { hasAmPmToggle, hoursRef, maxHours, minHours, minutesRef, onHoursBlur, onHoursChange, onKeyDown, onMinutesBlur, onBlurField, onMinutesChange, state, toggleChange, } = useTime({ elementId: fieldId, locale, onChange: handleTimeChange, value: time?.toString() });
    const handlePopoverOpenChange = React.useCallback((isOpen) => {
        if (!isOpen) {
            inputRef?.current?.blur();
            onBlurField();
            setIsManuallyClosed(true);
            setTimeout(() => setIsManuallyClosed(false), 0);
        }
        onPopperOpenChange(isOpen, type);
    }, [inputRef, onBlurField, onPopperOpenChange, type]);
    const handleFocus = React.useCallback(() => {
        if (!isManuallyClosed) {
            handlePopoverOpenChange(true);
        }
    }, [handlePopoverOpenChange, isManuallyClosed]);
    const onToggleChange = React.useCallback((_, toggle) => {
        toggleChange(toggle);
    }, [toggleChange]);
    const shouldCloseOnInteractOutside = React.useCallback(() => {
        return true;
    }, []);
    React.useEffect(() => {
        if (isPopoverOpen) {
            setTimeout(() => {
                popoverRef.current?.querySelector('.react-aria-CalendarCell[tabindex="0"]')?.focus();
            }, 50);
        }
    }, [isPopoverOpen]);
    const onInputKeyDown = React.useCallback((ev) => {
        if (ev.key === 'Tab') {
            handlePopoverOpenChange(false);
        }
        if (ev.key === 'Backspace' || ev.key === 'Delete') {
            handleDateChange(null);
            handleTimeChange(null);
        }
        if (ev.key.startsWith('Key') || ev.key.startsWith('Digit') || ev.key.startsWith('Numpad')) {
            // Prevent the user from typing in the input field
            ev.preventDefault();
            ev.stopPropagation();
        }
    }, [handleDateChange, handlePopoverOpenChange, handleTimeChange]);
    const inputValue = date && time ? formatDatetime(makeDatetime(date, Time.parse(time), timeZone), locale) : '';
    return (React.createElement(DialogTrigger, { onOpenChange: handlePopoverOpenChange, isOpen: isPopoverOpen },
        React.createElement("div", { className: "e-combined-input-wrapper" },
            React.createElement(Textbox, { "data-testid": `e-datetime-input-${type}`, label: title, "aria-label": ariaLabel, size: "medium", inputIcon: "calendar", value: inputValue, onFocus: handleFocus, onClick: handleFocus, ref: inputRef, onKeyDown: onInputKeyDown, disabled: isDisabled, readOnly: isReadOnly, error: validationError })),
        React.createElement(Popover, { triggerRef: inputRef, ref: popoverRef, placement: "bottom", crossOffset: 25, shouldCloseOnInteractOutside: shouldCloseOnInteractOutside },
            React.createElement(I18nProvider, { locale: locale },
                React.createElement("div", { className: "e-popover-dialog" },
                    React.createElement("div", { className: "e-calendar-container", "data-testid": `e-datetime-input-${type}-date-picker` },
                        React.createElement(Calendar, { maxValue: maxDateValue, minValue: minDateValue, isDisabled: isDisabled, onChange: handleDateChange, isReadOnly: isReadOnly, value: date, isDateUnavailable: isDateUnavailable(rangeStartCalendarDate, minDate, maxDate) },
                            React.createElement(MonthYearHeaderComponent, null),
                            React.createElement(CalendarGrid, { className: "e-calendar-grid-styled", weekdayStyle: "short" }, day => (React.createElement(CalendarCellComponent, { day: day, rangeStartCalendarDate: rangeStartCalendarDate, selectedDate: date || undefined, timeZone: timeZone }))))),
                    React.createElement("div", { className: "e-time-component-container e-time-field" },
                        React.createElement("div", { className: "e-time-component-title" }, localize('@sage/xtrem-ui/date-time-range-time', 'Time')),
                        React.createElement("div", { className: "e-time-component-wrapper" },
                            React.createElement(TimeComp, { screenId: screenId, elementId: elementId, dataTestId: `e-datetime-input-${type}-time-input`, fieldId: fieldId, isReadOnly: isReadOnly, hoursRef: hoursRef, minutesRef: minutesRef, isDisabled: isDisabled, maxHours: maxHours, minHours: minHours, onHoursBlur: onHoursBlur, onHoursChange: onHoursChange, onKeyDown: onKeyDown, onMinutesBlur: onMinutesBlur, onMinutesChange: onMinutesChange, state: state, hasAmPmToggle: hasAmPmToggle, localize: localize, onToggleChange: onToggleChange })),
                        !isTimeZoneHidden && (React.createElement("div", { className: "e-time-component-timezone-container" },
                            React.createElement("div", { className: "e-time-component-timezone-input" },
                                React.createElement(Textbox, { label: localize('@sage/xtrem-ui/date-time-range-time-zone', 'Time zone:'), labelInline: true, id: "time-component-timezone", "data-testid": `e-datetime-input-${type}-time-input-timezone`, value: timeZone, readOnly: true }))))))))));
}
//# sourceMappingURL=datetime-input-component.js.map