import { formatDateToCurrentLocale, isValidIsoDate, datePropertyValueToDateString } from '@sage/xtrem-date-time';
import CarbonDateComponent from 'carbon-react/esm/components/date';
import * as React from 'react';
import { connect } from 'react-redux';
import * as xtremRedux from '../../../redux';
import { handleChange } from '../../../utils/abstract-fields-utils';
import { isUnmodifiedAlphabeticalChar } from '../../../utils/keyboard-event-utils';
import { getCommonCarbonComponentProperties, isFieldDisabled } from '../carbon-helpers';
import { CarbonWrapper } from '../carbon-wrapper';
import { ErrorableFieldBaseComponent, mapDispatchToProps, mapStateToProps } from '../field-base-component';
import { debounce, throttle } from 'lodash';
import { triggerFieldEvent } from '../../../utils/events';
import I18nProvider from 'carbon-react/esm/components/i18n-provider';
import { carbonLocale } from '../../../utils/carbon-locale';
export class DateComponent extends ErrorableFieldBaseComponent {
    constructor(props) {
        super(props);
        this.dateInputRef = React.createRef();
        this.onInputValueChanged = debounce(async (searchText) => {
            await triggerFieldEvent(this.props.screenId, this.props.elementId, 'onInputValueChange', searchText);
        }, 150);
        this.dispatchChange = throttle((rawValue) => {
            handleChange(this.props.elementId, rawValue || null, this.props.setFieldValue, this.validate(), this.triggerChangeListener);
        }, 40);
        this.onBlur = (event) => {
            const carbonProps = getCommonCarbonComponentProperties(this.props);
            carbonProps.onBlur();
            this.setState({ internalError: undefined });
            if (this.isReadOnly() || this.isDisabled()) {
                return;
            }
            if (datePropertyValueToDateString(event.target.value.rawValue) !==
                datePropertyValueToDateString(this.props.value)) {
                this.dispatchChange(event.target.value.rawValue);
            }
        };
        this.onChange = async (event) => {
            if (event.target?.value) {
                this.setState({ ...event.target.value });
                await this.onInputValueChanged(event.target.value.rawValue || event.target.value.formattedValue);
            }
        };
        this.onKeyDown = (event) => {
            if (isUnmodifiedAlphabeticalChar(event)) {
                event.preventDefault();
            }
        };
        this.getFormattedDate = (rawDateValue) => {
            return rawDateValue && isValidIsoDate(rawDateValue)
                ? formatDateToCurrentLocale(rawDateValue, this.props.locale, 'FullDate')
                : '';
        };
        this.validate = () => {
            if (this.state.internalError) {
                this.props.addInternalError?.(this.props.elementId, {
                    screenId: this.props.screenId,
                    elementId: this.props.elementId,
                    validationRule: 'dateValue',
                    message: this.state.internalError,
                });
            }
            else {
                this.props.removeInternalError?.(this.props.elementId);
            }
            return this.props.validate;
        };
        this.state = { formattedValue: this.getFormattedDate(this.props.value), rawValue: this.props.value };
    }
    UNSAFE_componentWillReceiveProps(nextProps) {
        if (this.props.isInFocus && !nextProps.isInFocus && this.state.rawValue !== this.props.value) {
            this.dispatchChange(this.state.rawValue);
        }
        const nextRawValue = nextProps.value;
        if (!this.state.rawValue || this.state.rawValue !== nextRawValue) {
            this.setState({ formattedValue: this.getFormattedDate(nextRawValue), rawValue: nextRawValue });
        }
    }
    render() {
        const maxDate = datePropertyValueToDateString(this.getResolvedProperty('maxDate')) || undefined;
        const minDate = datePropertyValueToDateString(this.getResolvedProperty('minDate')) || undefined;
        return (React.createElement(CarbonWrapper, { ...this.props, className: "e-date-field", componentName: "date", componentRef: this.componentRef, handlersArguments: this.props.handlersArguments, noReadOnlySupport: true, value: this.state.formattedValue },
            React.createElement(I18nProvider, { locale: carbonLocale(this.props.locale || 'en-US') },
                React.createElement(CarbonDateComponent, { ...getCommonCarbonComponentProperties(this.props), disablePortal: true, ref: this.dateInputRef, allowEmptyValue: true, onClick: this.getClickHandler(), onChange: this.onChange, onKeyDown: this.onKeyDown, value: this.state.formattedValue || '', maxDate: maxDate, minDate: minDate, onBlur: this.onBlur, error: !this.props.isParentDisabled &&
                        !this.props.isParentHidden &&
                        !isFieldDisabled(this.props.screenId, this.props.fieldProperties, this.props.value, this.props.handlersArguments?.rowValue)
                        ? this.props.validationErrors?.[0]?.message || this.state.internalError
                        : undefined }))));
    }
}
const extendedMapDispatchToProps = (dispatch, props) => {
    const defaultMapDispatchToProps = mapDispatchToProps()(dispatch, props);
    return {
        ...defaultMapDispatchToProps,
        addInternalError: (elementId, errorMessage) => {
            dispatch(xtremRedux.actions.addInternalError(props.screenId, elementId, errorMessage));
        },
        removeInternalError: (elementId) => {
            dispatch(xtremRedux.actions.removeInternalError(props.screenId, elementId));
        },
    };
};
export const ConnectedDateComponent = connect(mapStateToProps, extendedMapDispatchToProps)(DateComponent);
export default ConnectedDateComponent;
//# sourceMappingURL=date-component.js.map