import { formatDateToCurrentLocale } from '@sage/xtrem-date-time';
import Date from 'carbon-react/esm/components/date';
import I18nProvider from 'carbon-react/esm/components/i18n-provider';
import uniqueId from 'lodash/uniqueId';
import React from 'react';
import { getStore } from '../../../redux';
import { getInitialCellEditorState, isFinishedEditingPressed, isKeyPressedDateComponent, isLeftOrRight, setDefaultAgGridInputStyles, } from '../../../utils/ag-grid/ag-grid-cell-editor-utils';
import { carbonLocale } from '../../../utils/carbon-locale';
import { isDeleteOrBackspace } from '../../../utils/keyboard-event-utils';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { splitValueToMergedValue } from '../../../utils/transformers';
export default class DateCellEditor extends React.Component {
    constructor(props) {
        super(props);
        this.getFormattedDate = () => this.state.value ? formatDateToCurrentLocale(this.state.rawValue, this.state.locale, 'FullDate') : '';
        this.getCarbonProps = () => {
            return {
                autoComplete: 'off',
                name: this.props.column?.getColDef().field,
                className: '',
                size: 'small',
                id: uniqueId(),
            };
        };
        this.state = this.createInitialState(props);
        this.onKeyDown = this.onKeyDown.bind(this);
        this.handleChange = this.handleChange.bind(this);
    }
    componentDidMount() {
        const input = this.props.eGridCell.querySelector('input');
        if (!input) {
            return;
        }
        // try to match the styles of ag-grid
        setDefaultAgGridInputStyles(input);
        const inputDiv = input.closest('div');
        if (inputDiv) {
            inputDiv.style.width = '100%';
            inputDiv.style.height = '100%';
            inputDiv.style.border = 'none';
            inputDiv.style.display = 'flex';
            inputDiv.style.alignItems = 'center';
        }
        input.focus();
        if (this.state.highlightOnFocus) {
            input.select();
            this.setState({
                highlightOnFocus: false,
            });
        }
        else {
            // when we started editing, we want the caret at the end, not the start.
            // this comes into play in two scenarios: a) when user hits F2 and b)
            // when user hits a printable character, then on IE (and only IE) the caret
            // was placed after the first character, thus 'apply' would end up as 'pplea'
            const length = input.value ? input.value.length : 0;
            if (length > 0) {
                input.setSelectionRange(length, length);
            }
        }
    }
    handleChange(event) {
        if (event.target?.value) {
            this.setState({ value: event.target.value.formattedValue, rawValue: event.target.value.rawValue }, () => {
                if (this.state.rawValue != null) {
                    let parsedDate = '';
                    try {
                        parsedDate = this.getFormattedDate();
                    }
                    catch (err) {
                        // The date is not valid yet. For instance the user wnat to enter 15/10/2020 but he has only entered 15/10/202 (he is about to enter the last '0')
                        // Nothing to do, just leave the parsedDate as empty
                    }
                    this.props.onValueChange(parsedDate !== '' ? this.state.rawValue : null);
                }
                else {
                    this.props.onValueChange(null);
                }
            });
        }
    }
    onKeyDown(event) {
        if (isLeftOrRight(event) || isDeleteOrBackspace(event)) {
            event.stopPropagation();
            return;
        }
        if (!isFinishedEditingPressed(event) && !isKeyPressedDateComponent(event) && event.preventDefault) {
            event.preventDefault();
        }
    }
    createInitialState(props) {
        const initialState = getInitialCellEditorState({ eventKey: props.eventKey, initialValue: props.initialValue });
        return {
            ...initialState,
            rawValue: initialState.value,
            locale: (getStore().getState().applicationContext?.locale || 'en-US'),
        };
    }
    render() {
        const width = this.props.eGridCell?.style?.width || '200px';
        const { fieldProperties, tableElementId, screenId, data } = this.props;
        const rowValue = splitValueToMergedValue(data);
        const maxDate = resolveByValue({
            screenId,
            propertyValue: fieldProperties.maxDate,
            rowValue,
            skipHexFormat: true,
            fieldValue: null,
        });
        const minDate = resolveByValue({
            screenId,
            propertyValue: fieldProperties.minDate,
            rowValue,
            skipHexFormat: true,
            fieldValue: null,
        });
        const allColumns = this.props.api.getColumns() || [];
        return (React.createElement("div", { className: "ag-cell-edit-wrapper e-nested-cell-field-date", "data-testid": `${tableElementId}-${this.props.node.rowIndex}-${allColumns.indexOf(this.props.column) + 1}`, style: { width } },
            React.createElement(I18nProvider, { locale: carbonLocale(this.state.locale || 'en-US') },
                React.createElement(Date, { allowEmptyValue: true, onChange: this.handleChange, value: this.state.value || '', maxDate: maxDate, minDate: minDate, ...this.getCarbonProps(), onKeyDown: this.onKeyDown }))));
    }
}
//# sourceMappingURL=date-cell-editor.js.map