import { Checkbox } from 'carbon-react/esm/components/checkbox';
import * as React from 'react';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { splitValueToMergedValue } from '../../../utils/transformers';
import { usePrevious } from '@sage/xtrem-ui-components';
import { SPACEBAR } from '../../../utils/keyboard-event-utils';
export const CheckboxCellRenderer = React.memo(({ colDef, column, data, eGridCell, eventKey, fieldProperties, initialValue, isEditing, isParentFieldDisabled, isTableReadOnly, node, onValueChange, screenId, stopEditing, value, ...rest }) => {
    const checkboxRef = React.useRef(null);
    const [checked, setChecked] = React.useState((isEditing ? initialValue : value) || false);
    const previousChecked = usePrevious(checked);
    // react to external value changes
    React.useEffect(() => {
        setChecked(value);
    }, [value]);
    const onFocus = React.useCallback(() => {
        checkboxRef.current?.focus();
    }, []);
    React.useEffect(() => {
        eGridCell.addEventListener('focus', onFocus);
        return () => {
            eGridCell.removeEventListener('focus', onFocus);
        };
    }, [eGridCell, onFocus]);
    // update grid value upon state changes
    React.useEffect(() => {
        if (previousChecked === undefined || checked === previousChecked) {
            return;
        }
        node.setDataValue(column.getColId(), checked);
        if (isEditing) {
            onValueChange(checked);
            stopEditing(true);
        }
        else {
            node.setDataValue(column.getColId(), checked);
        }
        // onValueChange is not memoized by ag-grid
        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, [checked, column, node, isEditing, previousChecked]);
    const onChange = React.useCallback(newValue => {
        setChecked(newValue);
    }, []);
    React.useEffect(() => {
        if (isEditing) {
            checkboxRef.current?.focus();
        }
    }, [isEditing]);
    const isDisabled = React.useMemo(() => isParentFieldDisabled ||
        resolveByValue({
            screenId,
            propertyValue: fieldProperties.isDisabled,
            rowValue: splitValueToMergedValue(data),
            fieldValue: value,
            skipHexFormat: true,
        }), [data, fieldProperties.isDisabled, isParentFieldDisabled, screenId, value]);
    const isFieldReadOnly = React.useMemo(() => resolveByValue({
        screenId,
        propertyValue: fieldProperties.isReadOnly,
        rowValue: splitValueToMergedValue(data),
        fieldValue: value,
        skipHexFormat: true,
    }), [data, fieldProperties.isReadOnly, screenId, value]);
    const isColumnDisabled = React.useMemo(() => !colDef.isEditable(data), [colDef, data]);
    const onCheckboxChange = React.useCallback(e => onChange(e.target.checked), [onChange]);
    const onKeyDownCapture = React.useCallback(e => {
        if (e.code === SPACEBAR) {
            e.stopPropagation();
            onChange(!e.target.checked);
        }
    }, [onChange]);
    const onClickCapture = React.useCallback(e => {
        if (!isEditing) {
            // prevent ag-grid from entering edit mode when checkbox is clicked
            e.stopPropagation();
        }
        onChange((e?.target).checked);
    }, [isEditing, onChange]);
    const renderCheckbox = React.useCallback(() => {
        return (React.createElement("div", { style: {
                margin: '3px',
            } },
            React.createElement("div", { style: {
                    display: 'flex',
                    flex: 1,
                    justifyContent: 'center',
                } },
                React.createElement(Checkbox, { ref: checkboxRef, "aria-label": colDef.headerName, value: "checked", checked: checked, disabled: Boolean(isDisabled || isFieldReadOnly || isTableReadOnly || isColumnDisabled), onChange: onCheckboxChange, onKeyDownCapture: onKeyDownCapture, onClickCapture: onClickCapture }))));
    }, [
        colDef.headerName,
        checked,
        isDisabled,
        isFieldReadOnly,
        isTableReadOnly,
        isColumnDisabled,
        onCheckboxChange,
        onKeyDownCapture,
        onClickCapture,
    ]);
    if (isEditing) {
        // no wrapper in edit mode
        return renderCheckbox();
    }
    return (React.createElement(fieldProperties.wrapper, { colDef: colDef, column: column, data: data, eGridCell: eGridCell, eventKey: eventKey, fieldProperties: fieldProperties, initialValue: initialValue, isEditing: isEditing, isParentFieldDisabled: isParentFieldDisabled, isTableReadOnly: isTableReadOnly, node: node, onValueChange: onValueChange, screenId: screenId, stopEditing: stopEditing, value: value, ...rest, textAlign: "center" }, renderCheckbox()));
});
CheckboxCellRenderer.displayName = 'CheckboxCellRenderer';
//# sourceMappingURL=checkbox-cell-renderer.js.map