import * as wrapperService from '../../service/wrapper-service';
import { ContextType } from '../../types';
import { triggerFieldEvent, triggerNestedFieldEvent } from '../../utils/events';
import { resolveByValue } from '../../utils/resolve-value-utils';
import { splitValueToMergedValue } from '../../utils/transformers';
import { getFieldElement } from './carbon-utility-components';
import { handleChange } from './field-base-utils';
import { getReferenceValueField } from './reference/reference-utils';
export const getLabelTitle = (screenId, fieldProperties, rowValue) => {
    if (fieldProperties.isTitleHidden) {
        return undefined;
    }
    const resolvedTitle = getFieldTitle(screenId, fieldProperties, rowValue);
    const isMandatory = resolveByValue({
        propertyValue: fieldProperties.isMandatory,
        screenId,
        skipHexFormat: true,
        fieldValue: undefined,
        rowValue: undefined,
    });
    const isNotZero = resolveByValue({
        propertyValue: fieldProperties.isNotZero,
        screenId,
        skipHexFormat: true,
        fieldValue: undefined,
        rowValue: undefined,
    });
    if (resolvedTitle && (isMandatory || isNotZero)) {
        return `${resolvedTitle} *`;
    }
    if (resolvedTitle) {
        return `${resolvedTitle}`;
    }
    if (isMandatory || isNotZero) {
        return '*';
    }
    return undefined;
};
export const generateFieldId = ({ contextType, elementId, fieldProperties, isNested, parentElementId, screenId, }) => {
    const nestedValueField = isNested && Object.prototype.hasOwnProperty.call(fieldProperties, 'valueField')
        ? getReferenceValueField(fieldProperties)?.split('.').join('-')
        : undefined;
    return [screenId, parentElementId, elementId, nestedValueField, contextType].filter(Boolean).join('-');
};
export const isFieldReadOnly = (screenId, fieldProperties, fieldValue, rowValue, contextType) => contextType === ContextType.navigationPanel ||
    contextType === ContextType.header ||
    resolveByValue({
        fieldValue,
        propertyValue: fieldProperties.isReadOnly,
        skipHexFormat: true,
        rowValue: rowValue ? splitValueToMergedValue(rowValue) : rowValue,
        screenId,
    });
export const getFieldIndicatorStatus = ({ validationErrors, screenId, value: fieldValue, fieldProperties, handlersArguments, isParentDisabled, isParentHidden, }) => {
    const rowValue = handlersArguments?.rowValue ? splitValueToMergedValue(handlersArguments?.rowValue) : undefined;
    const error = !isParentDisabled &&
        !isParentHidden &&
        !isFieldDisabled(screenId, fieldProperties, fieldValue, handlersArguments?.rowValue)
        ? validationErrors?.[0]?.message || undefined
        : undefined;
    const warning = (!error &&
        resolveByValue({
            screenId,
            fieldValue,
            propertyValue: fieldProperties.warningMessage,
            rowValue,
            skipHexFormat: true,
        })) ||
        undefined;
    const info = (!error &&
        !warning &&
        resolveByValue({
            screenId,
            fieldValue,
            propertyValue: fieldProperties.infoMessage,
            rowValue,
            skipHexFormat: true,
        })) ||
        undefined;
    return { info, error, warning };
};
export const isFieldDisabled = (screenId, fieldProperties, fieldValue, rowValue) => resolveByValue({
    fieldValue,
    propertyValue: fieldProperties.isDisabled,
    skipHexFormat: true,
    rowValue: rowValue ? splitValueToMergedValue(rowValue) : rowValue,
    screenId,
});
export const getFieldTitle = (screenId, fieldProperties, rowValue) => resolveByValue({
    screenId,
    propertyValue: fieldProperties.title,
    skipHexFormat: true,
    rowValue: splitValueToMergedValue(rowValue),
});
export const isFieldHidden = (screenId, fieldProperties, rowValue) => resolveByValue({
    screenId,
    propertyValue: fieldProperties.isHidden,
    skipHexFormat: true,
    rowValue: splitValueToMergedValue(rowValue),
});
export const isFieldTitleHidden = (screenId, fieldProperties, rowValue) => resolveByValue({
    screenId,
    propertyValue: fieldProperties.isTitleHidden,
    skipHexFormat: true,
    rowValue: splitValueToMergedValue(rowValue),
});
export const getFieldHelperText = (screenId, fieldProperties, rowValue) => resolveByValue({
    screenId,
    propertyValue: fieldProperties.helperText,
    skipHexFormat: true,
    rowValue: splitValueToMergedValue(rowValue),
});
export const isFieldHelperTextHidden = (screenId, fieldProperties, rowValue) => resolveByValue({
    screenId,
    propertyValue: fieldProperties.isHelperTextHidden,
    skipHexFormat: true,
    rowValue: splitValueToMergedValue(rowValue),
});
export const getPageSubtitle = (screenId, fieldProperties, rowValue) => resolveByValue({
    screenId,
    propertyValue: fieldProperties.subtitle,
    skipHexFormat: true,
    rowValue: splitValueToMergedValue(rowValue),
}) || null;
export const onBlurHandler = (props) => {
    if (isFieldDisabled(props.screenId, props.fieldProperties, props.value, props.handlersArguments?.rowValue)) {
        return;
    }
    if (isFieldReadOnly(props.screenId, props.fieldProperties, props.value, props.handlersArguments?.rowValue)) {
        return;
    }
    wrapperService.onBlur({
        elementId: props.parentElementId || props.elementId,
        screenId: props.screenId,
        nestedElementId: props.parentElementId ? props.elementId : undefined,
        recordId: props.handlersArguments?.rowValue._id,
    });
};
export const onFocusHandler = (props) => {
    if (isFieldDisabled(props.screenId, props.fieldProperties, props.value, props.handlersArguments?.rowValue)) {
        return;
    }
    if (isFieldReadOnly(props.screenId, props.fieldProperties, props.value, props.handlersArguments?.rowValue)) {
        return;
    }
    const callback = async (rawValue) => {
        await props.setFieldValue(props.elementId, rawValue);
        const triggerChangeListener = async () => {
            const nestedFieldsAdditionalProperties = props;
            if (nestedFieldsAdditionalProperties.isNested) {
                await triggerNestedFieldEvent(props.screenId, nestedFieldsAdditionalProperties.parentElementId || props.elementId, props.fieldProperties, 'onChange', nestedFieldsAdditionalProperties.handlersArguments?.rowValue._id, nestedFieldsAdditionalProperties.handlersArguments?.rowValue);
            }
            else {
                await triggerFieldEvent(props.screenId, props.elementId, 'onChange');
            }
        };
        handleChange(props.elementId, rawValue, props.setFieldValue, props.validate, triggerChangeListener);
    };
    wrapperService.onFocus({
        elementId: props.parentElementId || props.elementId,
        screenId: props.screenId,
        nestedElementId: props.parentElementId ? props.elementId : undefined,
        recordId: props.handlersArguments?.rowValue._id,
        title: getFieldTitle(props.screenId, props.fieldProperties, null),
    }, callback);
};
export const getCommonCarbonComponentProperties = (props, children) => {
    const rowValue = props.handlersArguments?.rowValue
        ? splitValueToMergedValue(props.handlersArguments?.rowValue)
        : undefined;
    const { contextType, elementId, fieldProperties, isNested = false, parentElementId, screenId, isParentHidden, isParentDisabled, } = props;
    const { error, warning, info } = getFieldIndicatorStatus(props);
    const carbonProps = {
        autoComplete: 'off',
        name: props.elementId,
        disabled: isParentDisabled ||
            isParentHidden ||
            isFieldDisabled(props.screenId, props.fieldProperties, props.value, rowValue),
        readOnly: isFieldReadOnly(props.screenId, props.fieldProperties, props.value, rowValue, props.contextType),
        fieldHelp: props.fieldProperties.helperText,
        label: props.contextType === ContextType.navigationPanel || props.contextType === ContextType.header
            ? undefined
            : getLabelTitle(props.screenId, props.fieldProperties, rowValue),
        size: props.fieldProperties.size,
        placeholder: props.fieldProperties.placeholder,
        children: children ||
            getFieldElement('e-field-postfix', resolveByValue({
                screenId: props.screenId,
                fieldValue: props.value,
                propertyValue: props.fieldProperties.postfix,
                rowValue,
                skipHexFormat: true,
            })),
        leftChildren: getFieldElement('e-field-prefix', resolveByValue({
            screenId: props.screenId,
            fieldValue: props.value,
            propertyValue: props.fieldProperties.prefix,
            rowValue,
            skipHexFormat: true,
        })),
        tabindex: props.contextType === ContextType.navigationPanel ? -1 : undefined,
        onFocus: () => {
            onFocusHandler(props);
            props.onFocus();
        },
        onBlur: () => onBlurHandler(props),
        error,
        info,
        warning,
        validationOnLabel: true,
        id: generateFieldId({ screenId, elementId, contextType, fieldProperties, parentElementId, isNested }),
    };
    if (carbonProps.disabled) {
        // Carbon throws an error if we display any tooltip help on disabled fields
        delete carbonProps.error;
        delete carbonProps.info;
        delete carbonProps.warning;
    }
    if (!props.fieldProperties.helperText) {
        carbonProps['aria-describedby'] = undefined;
    }
    carbonProps['data-label'] = getDataLabel(props);
    return carbonProps;
};
/**
 * 'data-label' is used for alignment purposes only
 *
 * @param {(EditableFieldComponentProperties<any, any> & FieldComponentExternalProperties & NestedFieldsAdditionalProperties)} props
 * @returns
 */
function getDataLabel(props) {
    if (props.fieldProperties.isTitleHidden) {
        return undefined;
    }
    if (props.contextType === ContextType.navigationPanel || props.contextType === ContextType.header) {
        return getFieldTitle(props.screenId, props.fieldProperties, props.handlersArguments?.rowValue) ?? '';
    }
    return getLabelTitle(props.screenId, props.fieldProperties, props.handlersArguments?.rowValue);
}
//# sourceMappingURL=carbon-helpers.js.map