import * as React from 'react';
import { connect } from 'react-redux';
import * as xtremRedux from '../../../redux';
import { RenderingRouter } from '../../../render/rendering-router';
import { subscribe, unsubscribe } from '../../../service/shortcut-service';
import { getSectionValidationMessage } from '../../../service/validation-service';
import { ContextType } from '../../../types';
import { scrollWithAnimationTo } from '../../../utils/dom';
import { getHeaderSection, getVisibleSections } from '../../../utils/layout-utils';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { getNavigationPanelState, getPageDefinitionFromState, getPagePropertiesFromState, } from '../../../utils/state-utils';
import { getFieldTitle } from '../../field/carbon-helpers';
import { XtremTabs } from '../../ui/tabs/xtrem-tabs';
import ToggleNavigationPanelButton from '../navigation-panel/toggle-navigation-panel-button';
import { ConnectedPageHeaderCardComponent } from './page-header-card';
import IconButton from 'carbon-react/esm/components/icon-button';
import Icon from 'carbon-react/esm/components/icon';
import { getRouter } from '../../../service/router';
import { localize } from '../../../service/i18n-service';
import * as tokens from '@sage/design-tokens/js/base/common';
import { getIdFieldValue } from '../../../utils/id-field-utils';
import { PageTitle } from '../../ui/page-title';
import FieldWrapper from '../../../render/field-wrapper';
import { getScreenElement } from '../../../service/screen-base-definition';
import HeaderNavigationArrows from './header-navigation-arrows';
import { ConnectedHeaderQuickActions } from './header-quick-actions';
import { ConnectedHeaderDropdownActions } from './header-dropdown-actions';
import { getHeaderTitles } from '../../../utils/page-utils';
import { Page360Switch } from '../page-360/page-360-switch';
import { HEADER_IMAGE, HEADER_TITLE, NEW_PAGE } from '../../../utils/constants';
import { PrintRecordButton } from './print-record-button';
import { Portrait } from '../../ui/portrait-component';
import { getImageUrlFromValue } from '../../field/image/image-utils';
import { navigationPanelId } from '../navigation-panel/navigation-panel-types';
import { WizardSteps } from './wizard-steps';
export class XtremHeader extends React.Component {
    constructor(props) {
        super(props);
        this.tabDictionary = {};
        this.hotkeySubscriptions = [];
        this.parentRef = React.createRef();
        this.getActiveSectionIndex = () => this.props.sections.findIndex(s => this.props.activeSection === s.$containerId);
        this.setActiveSectionByIndex = (index) => {
            if (index < 0 || index >= this.props.sections.length) {
                return;
            }
            this.setActiveSection(this.props.sections[index].$containerId);
        };
        this.setActiveSection = (containerId) => {
            this.props.setActiveSection(containerId);
            if (this.props.pageBodyRef.current && this.props.pageMode !== 'tabs' && this.props.pageMode !== 'wizard') {
                scrollWithAnimationTo(this.props.pageBodyRef.current.querySelector(`#${containerId}`), this.props.pageBodyRef.current, 80);
            }
        };
        this.hasTabs = () => this.getVisibleTabs().length > 1;
        this.getClasses = () => {
            const classes = ['e-header'];
            if (this.props.contextType) {
                classes.push(`e-header-context-${this.props.contextType}`);
            }
            if (!this.hasTabs()) {
                classes.push('e-no-tabs');
            }
            return classes.join(' ');
        };
        this.getVisibleTabs = () => {
            if (this.props.areNavigationTabsHidden) {
                return [];
            }
            return this.props.sections
                .filter(item => getFieldTitle(this.props.screenId, this.props.sectionsProperties[item.$containerId], null))
                .filter(item => !resolveByValue({
                screenId: this.props.screenId,
                skipHexFormat: true,
                propertyValue: this.props.sectionsProperties[item.$containerId].isHidden,
                rowValue: null,
            }))
                .map(c => c.$containerId);
        };
        this.onRecordClose = () => {
            this.props.navigationPanelValue?.restoreFilterSnapshot();
            getRouter(this.props.screenId).closeRecord();
        };
        this.renderTabs = () => {
            if (!this.hasTabs()) {
                return null;
            }
            const sections = this.getVisibleTabs().map(id => ({
                id,
                title: getFieldTitle(this.props.screenId, this.props.sectionsProperties[id], null),
                validationMessage: this.props.sectionValidationErrors[id],
            }));
            if (this.props.pageMode === 'wizard') {
                return (React.createElement("div", { className: "e-header-wizard-steps-container" },
                    React.createElement(WizardSteps, { screenId: this.props.screenId, sections: sections, selectedSection: this.props.activeSection })));
            }
            return (React.createElement("div", { className: "e-header-nav", "data-testid": "e-header-nav-tabs" },
                React.createElement(XtremTabs, { selectedTabId: this.props.activeSection || this.getVisibleTabs()[0], onTabChange: this.setActiveSection, tabs: sections })));
        };
        this.onHeaderSectionToggle = () => {
            this.setState(state => ({ isHeaderSectionClosed: !state.isHeaderSectionClosed }), () => {
                if (this.props.pageBodyRef.current) {
                    /**
                     * Even if the parent element gets resized by hiding the top section,
                     * the resize event is only triggered automatically if the window is resized
                     *  */
                    const event = new UIEvent('resize', { view: window, bubbles: true, cancelable: false, detail: 0 });
                    this.props.pageBodyRef.current.dispatchEvent(event);
                }
            });
        };
        this.renderCloseIcon = () => {
            if (this.props.hasNavigationPanel) {
                return (React.createElement("div", { className: "e-header-close-icon-col" },
                    React.createElement(IconButton, { "aria-label": localize('@sage/xtrem-ui/close-record', 'Close record'), "data-testid": "e-page-close-button", "data-pendoid": "closeRecord", onAction: this.onRecordClose },
                        React.createElement(Icon, { type: "close", color: tokens.colorsUtilityMajor400, tooltipMessage: localize('@sage/xtrem-ui/close-record', 'Close record') }))));
            }
            return null;
        };
        this.renderHeaderToggleButton = () => {
            if (this.props.headerSection) {
                const label = this.state.isHeaderSectionClosed
                    ? localize('@sage/xtrem-ui/open-header-section', 'Open header')
                    : localize('@sage/xtrem-ui/close-header-section', 'Close header');
                return (React.createElement("div", { className: "e-header-toggle-icon-col" },
                    React.createElement(IconButton, { "aria-label": label, "data-testid": "e-toggle-header-section", onAction: this.onHeaderSectionToggle, mr: "24px", "data-pendoid": this.state.isHeaderSectionClosed ? 'headerOpen' : 'headerClose' },
                        React.createElement(Icon, { type: this.state.isHeaderSectionClosed ? 'chevron_up' : 'chevron_down', color: tokens.colorsUtilityMajor400, tooltipMessage: label }))));
            }
            return null;
        };
        this.renderTitleImage = (title) => (React.createElement("div", { className: "e-header-image-container" },
            this.props.headerImage && (React.createElement("div", { className: "e-header-image", style: { backgroundImage: `url('${getImageUrlFromValue(this.props.headerImage)}')` }, "aria-label": title })),
            !this.props.headerImage && this.props.headerTitle && (React.createElement(Portrait, { size: "M", placeholderMode: "Initials", placeholderValue: this.props.headerTitle, icon: "image" })),
            !this.props.headerImage && !this.props.headerTitle && (React.createElement("img", { className: "e-header-image", src: "/images/no-available-image.svg", alt: localize('@sage/xtrem-ui/no-available-image', 'No available image') }))));
        this.state = {
            isHeaderSectionClosed: false,
        };
    }
    componentDidMount() {
        this.setIndicatorPosition();
        this.hotkeySubscriptions.push(subscribe(['escape', 'arrowup'], () => {
            const index = this.getActiveSectionIndex();
            this.setActiveSectionByIndex(index - 1);
        }));
        this.hotkeySubscriptions.push(subscribe(['escape', 'arrowdown'], () => {
            const index = this.getActiveSectionIndex();
            this.setActiveSectionByIndex(index + 1);
        }));
        // We subscribe to each button in order to prevent issues from overlapping screens
        for (let i = 1; i < 10; i += 1) {
            this.hotkeySubscriptions.push(subscribe(['escape', i.toString()], () => {
                this.setActiveSectionByIndex(i - 1);
            }));
        }
    }
    componentDidUpdate(prevProps) {
        this.setIndicatorPosition();
        if (this.props.isNavigationPanelOpened !== prevProps.isNavigationPanelOpened) {
            this.parentRef.current?.focus();
        }
        const visibleSections = this.getVisibleTabs();
        if (this.props.activeSection &&
            this.props.pageMode !== 'wizard' &&
            !this.props.areNavigationTabsHidden &&
            this.getVisibleTabs().indexOf(this.props.activeSection) === -1) {
            this.props.setActiveSection(visibleSections[0]);
        }
    }
    componentWillUnmount() {
        this.hotkeySubscriptions.forEach(unsubscribe);
    }
    setIndicatorPosition() {
        if (this.activeIndicator &&
            this.tabContainer &&
            this.props.activeSection &&
            this.tabDictionary[this.props.activeSection]) {
            const activeTabRect = this.tabDictionary[this.props.activeSection].getBoundingClientRect();
            const containerX = this.tabContainer.getBoundingClientRect().x || 0;
            this.activeIndicator.style.left = `${(activeTabRect.x || 0) - containerX}px`;
            this.activeIndicator.style.width = `${activeTabRect.width - 16}px`;
        }
    }
    render() {
        const { isNavigationPanelOpened, isNavigationPanelHeaderHidden } = this.props;
        const { subtitle, title } = getHeaderTitles({
            title: this.props.title,
            subtitle: this.props.subtitle,
            idFieldValue: this.props.idFieldValue,
            isRecordCreationPage: this.props.isRecordCreationPage,
            objectTypePlural: this.props.objectTypePlural,
            objectTypeSingular: this.props.objectTypeSingular,
        });
        const router = getRouter(this.props.screenId);
        const hierarchyParentPage = resolveByValue({
            screenId: this.props.screenId,
            skipHexFormat: true,
            propertyValue: this.props.hierarchyParentPage,
            rowValue: null,
        });
        return (React.createElement("header", { className: this.getClasses(), "data-testid": "e-header" },
            React.createElement("div", { className: "e-header-title-row" },
                this.props.hasNavigationPanel &&
                    !this.props.canToggleNavigationPanel &&
                    !isNavigationPanelOpened &&
                    !isNavigationPanelHeaderHidden && (React.createElement(ToggleNavigationPanelButton, { setNavigationPanelIsOpened: this.props.setNavigationPanelIsOpened, isNavigationPanelOpened: this.props.isNavigationPanelOpened, parentRef: this.parentRef })),
                this.props.contextType === ContextType.page && !this.props.isTitleHidden && (React.createElement("div", { className: "e-header-title-col" },
                    React.createElement("div", { className: "e-header-title-container" },
                        hierarchyParentPage && (React.createElement("div", { className: "e-header-navigation-arrow-hierarchy-parent-page-container", "data-testid": "e-header-navigation-arrow-hierarchy-parent-page-container" },
                            React.createElement(IconButton, { "aria-label": localize('@sage/xtrem-ui/hierarchy-parent-page', 'Hierarchy parent page'), "data-testid": "e-header-navigation-arrow-hierarchy-parent-page", onAction: () => router.goTo(hierarchyParentPage) },
                                React.createElement(Icon, { type: "arrow_left", color: tokens.colorsYin090, tooltipMessage: localize('@sage/xtrem-ui/hierarchy-parent-page', 'Hierarchy parent page') })))),
                        this.props.shouldDisplayNavigationArrows && (React.createElement(HeaderNavigationArrows, { screenId: this.props.screenId })),
                        this.props.isHeaderImageNeeded && this.renderTitleImage(title),
                        React.createElement("div", null,
                            React.createElement(PageTitle, { title: title }),
                            !!subtitle && React.createElement("p", { className: "e-header-subtitle" }, subtitle)),
                        this.props.headerLabelField && (React.createElement("div", { className: "e-header-title-label" }, this.props.headerLabelField && (React.createElement(FieldWrapper, { item: { $bind: this.props.headerLabelField.id }, screenId: this.props.screenId, contextType: ContextType.pageHeader }))))))),
                this.props.hasDropdownOrQuickActions && (React.createElement("span", { className: "e-header-line-icon-group" },
                    false && ( // XT-78503 HIDE PRINT BUTTON
                    React.createElement(PrintRecordButton, { screenId: this.props.screenId })),
                    React.createElement(ConnectedHeaderQuickActions, { screenId: this.props.screenId }),
                    React.createElement(ConnectedHeaderDropdownActions, { screenId: this.props.screenId }))),
                this.props.has360View && !this.props.isNewPage && React.createElement(Page360Switch, { screenId: this.props.screenId }),
                (this.props.hasNavigationPanel || this.props.headerSection) && (React.createElement("span", { className: "e-header-line-icon-group e-header-line-icon-group-close" },
                    this.renderHeaderToggleButton(),
                    this.renderCloseIcon())),
                this.props.headerLineBlock && (React.createElement("div", { className: "e-header-line-block" },
                    React.createElement(RenderingRouter, { screenId: this.props.screenId, item: this.props.headerLineBlock, contextType: ContextType.header, availableColumns: 4 })))),
            React.createElement(ConnectedPageHeaderCardComponent, { screenId: this.props.screenId }),
            this.props.headerSection && !this.props.is360ViewOn && !this.state.isHeaderSectionClosed && (React.createElement("div", { className: "e-header-section", "data-testid": "e-header-section" },
                React.createElement(RenderingRouter, { screenId: this.props.screenId, item: this.props.headerSection, availableColumns: this.props.availableColumns }))),
            !this.props.areNavigationTabsHidden && !this.props.is360ViewOn && this.renderTabs()));
    }
}
const mapStateToProps = (state, props) => {
    const pageDefinition = getPageDefinitionFromState(props.screenId, state);
    const sectionIds = pageDefinition.metadata.layout.$items
        .filter(section => !!section.$containerId)
        .map(section => section.$containerId);
    const sectionsProperties = Object.keys(pageDefinition.metadata.uiComponentProperties)
        .filter(property => sectionIds.indexOf(property) > -1)
        .reduce((properties, nextKey) => ({
        ...properties,
        [nextKey]: pageDefinition.metadata.uiComponentProperties[nextKey],
    }), {});
    const navigationPanelState = getNavigationPanelState(props.screenId, state);
    const pageProperties = getPagePropertiesFromState(props.screenId, state);
    const navPanelDefinition = pageDefinition.metadata.uiComponentProperties[navigationPanelId];
    const isHeaderImageNeeded = !!navPanelDefinition?.mobileCard?.image;
    const sectionValidationErrors = sectionIds.reduce((prevValue, sectionId) => {
        return { ...prevValue, [sectionId]: getSectionValidationMessage(props.screenId, sectionId, state) };
    }, {});
    const locale = state.applicationContext?.locale || 'en-US';
    const idFieldValue = getIdFieldValue(pageDefinition, locale);
    const headerLabelField = pageProperties.headerLabel?.apply(getScreenElement(pageDefinition));
    const headerSection = getHeaderSection(pageDefinition, state.browser.lessThan.m);
    const headerQuickActions = resolveByValue({
        skipHexFormat: true,
        screenId: props.screenId,
        propertyValue: pageProperties.headerQuickActions,
        rowValue: null,
        fieldValue: null,
    });
    const has360View = resolveByValue({
        skipHexFormat: true,
        screenId: props.screenId,
        propertyValue: pageProperties.has360View,
        rowValue: null,
        fieldValue: null,
    }) || false;
    const headerDropdownActions = resolveByValue({
        skipHexFormat: true,
        screenId: props.screenId,
        propertyValue: pageProperties.headerDropDownActions,
        rowValue: null,
        fieldValue: null,
    });
    const headerImage = state.screenDefinitions[props.screenId]?.values[HEADER_IMAGE];
    const headerTitle = state.screenDefinitions[props.screenId]?.values[HEADER_TITLE];
    const getLastValue = (obj) => {
        if (typeof obj !== 'object' || obj === null)
            return obj;
        const values = Object.values(obj);
        if (values.length === 0)
            return '';
        return getLastValue(values[values.length - 1]);
    };
    const finalHeaderTitleValue = typeof headerTitle === 'object' && headerTitle !== null ? getLastValue(headerTitle) : headerTitle || '';
    const finalHeaderImageValue = typeof headerImage === 'object' && headerImage !== null
        ? getLastValue(headerImage)
        : headerImage?.value || undefined;
    return {
        ...props,
        canToggleNavigationPanel: state.browser.greaterThan.l,
        has360View,
        hasDropdownOrQuickActions: headerQuickActions?.length > 0 || headerDropdownActions?.length > 0,
        headerLabelField,
        headerSection: headerSection?.layout || null,
        headerImage: finalHeaderImageValue,
        isHeaderImageNeeded,
        headerTitle: finalHeaderTitleValue,
        hierarchyParentPage: pageProperties.hierarchyParentPage,
        idFieldValue,
        is360ViewOn: !!pageDefinition.is360ViewOn,
        isNavigationPanelHeaderHidden: !!navigationPanelState?.isHeaderHidden,
        isNavigationPanelOpened: !!navigationPanelState?.isOpened,
        isNewPage: pageDefinition.queryParameters?._id === NEW_PAGE,
        isTitleHidden: pageProperties.isTitleHidden,
        navigationPanelValue: pageDefinition.navigationPanel?.value,
        objectTypePlural: pageProperties.objectTypePlural,
        objectTypeSingular: pageProperties.objectTypeSingular,
        pageMode: pageProperties?.mode || 'default',
        sections: getVisibleSections(pageDefinition, state.activeDialogs, state.browser.lessThan.m),
        sectionsProperties,
        sectionValidationErrors,
        setNavigationPanelIsOpened: xtremRedux.actions.actionStub,
        shouldDisplayNavigationArrows: props.hasNavigationPanel && !state.browser.is.xs && !navigationPanelState?.isOpened,
        subtitle: pageProperties.subtitle,
        title: (pageProperties && getFieldTitle(props.screenId, pageProperties, null)) || '',
    };
};
const mapDispatchToProps = (dispatch, props) => {
    return {
        setNavigationPanelIsOpened: (isOpened) => dispatch(xtremRedux.actions.setNavigationPanelIsOpened(isOpened, props.screenId)),
    };
};
export const ConnectedXtremHeader = connect(mapStateToProps, mapDispatchToProps)(XtremHeader);
//# sourceMappingURL=xtrem-header.js.map