import React from 'react';
import { connect } from 'react-redux';
import { ContextType } from '../../../types';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { getPageDefinitionFromState, getPagePropertiesFromPageDefinition } from '../../../utils/state-utils';
import { CardComponent } from '../../ui/card/card-component';
const transformToNestedFieldDefinition = (pageDefinition, field) => {
    if (field) {
        const elementId = field.id;
        const properties = pageDefinition.metadata.uiComponentProperties[elementId];
        const defaultUiProperties = pageDefinition.metadata.defaultUiComponentProperties[elementId];
        return {
            properties: { ...properties, bind: properties.bind || elementId },
            defaultUiProperties: { ...defaultUiProperties, bind: properties.bind || elementId },
            type: defaultUiProperties._controlObjectType,
        };
    }
    return undefined;
};
export function PageHeaderCardComponent({ screenId, cardDefinition, value, }) {
    if (!cardDefinition || !value) {
        return null;
    }
    return (React.createElement(CardComponent, { screenId: screenId, cardDefinition: cardDefinition, contextType: ContextType.pageHeader, parentElementId: "$page", value: value }));
}
const mapStateToProps = (state, props) => {
    const { screenId } = props;
    if (!state.browser.is.xs) {
        return { screenId, cardDefinition: null, value: null };
    }
    const pageDefinition = getPageDefinitionFromState(screenId, state);
    const pageProperties = getPagePropertiesFromPageDefinition(pageDefinition);
    const headerCard = resolveByValue({
        screenId,
        propertyValue: pageProperties.headerCard,
        rowValue: null,
        fieldValue: null,
        skipHexFormat: true,
    });
    if (!headerCard) {
        return { screenId, cardDefinition: null, value: null };
    }
    const title = transformToNestedFieldDefinition(pageDefinition, headerCard.title);
    if (!title) {
        throw new Error(`Title is not defined for page header card of ${screenId}`);
    }
    const cardDefinition = {
        title,
        titleRight: transformToNestedFieldDefinition(pageDefinition, headerCard.titleRight),
        line2: transformToNestedFieldDefinition(pageDefinition, headerCard.line2),
        line2Right: transformToNestedFieldDefinition(pageDefinition, headerCard.line2Right),
        line3: transformToNestedFieldDefinition(pageDefinition, headerCard.line3),
        line3Right: transformToNestedFieldDefinition(pageDefinition, headerCard.line3Right),
        image: transformToNestedFieldDefinition(pageDefinition, headerCard.image),
    };
    const usedKeys = Object.keys(cardDefinition)
        .map(k => cardDefinition[k]?.properties.bind)
        .filter(v => !!v);
    return {
        ...props,
        cardDefinition,
        value: Object.keys(pageDefinition.values).reduce((prevValue, key) => {
            if (usedKeys.includes(key)) {
                return { ...prevValue, [key]: pageDefinition.values[key] };
            }
            return prevValue;
        }, {}),
    };
};
export const ConnectedPageHeaderCardComponent = connect(mapStateToProps)(PageHeaderCardComponent);
//# sourceMappingURL=page-header-card.js.map