/**
 * @packageDocumentation
 * @module root
 * */
import { dispatchContainerValidation } from '../../../service/dispatch-service';
import { getFocussedField } from '../../../service/field-state-service';
import { getPageMetadata } from '../../../service/page-metadata';
import { getUiComponentProperties, setUiComponentProperties } from '../../../service/transactions-service';
import { setDefaultProperties } from '../../../utils/abstract-fields-utils';
import { AbstractDecorator } from '../../abstract-decorator';
import { DetailPanelControlObject, detailPanelId, PageControlObject } from '../../control-objects';
import { pageAction } from '../../decorators';
import { withoutNestedTechnical } from '../../nested-fields';
import { ContainerKey } from '../../types';
import { DetailPanelLayout, PageLayout } from '../layouts';
import * as standardActions from './standard-page-actions';
import { addCustomFieldsToPageBody } from '../../../service/customization-service';
import { isEmpty } from 'lodash';
import * as xtremRedux from '../../../redux';
import { localize } from '../../../service/i18n-service';
import { BULK_ACTION_ASYNC_EXPORT } from '../../../utils/constants';
import { applyDefaultValuesOnNestedField, getCardDefinitionFromReferenceDataType, } from '../../../utils/data-type-utils';
import { schemaTypeNameFromNodeName } from '../../../utils/transformers';
export class PageDecorator extends AbstractDecorator {
    constructor() {
        super(...arguments);
        this._layout = PageLayout;
        this._controlObjectConstructor = PageControlObject;
        this.buildActions = () => {
            if (this._metadataProps.pageMetadata) {
                Object.keys(this._metadataProps.pageMetadata.pageActionThunks).forEach(k => {
                    this._metadataProps.pageMetadata.pageActions[k] = this._metadataProps.pageMetadata.pageActionThunks[k](this.nodeTypes, this.dataTypes);
                });
            }
            return this;
        };
        this.setUiComponentProperties = () => {
            const applicableProperties = {
                areNavigationTabsHidden: this._metadataProps.properties.areNavigationTabsHidden,
                createAction: this._metadataProps.properties.createAction,
                has360View: this._metadataProps.properties.has360View,
                headerCard: this._metadataProps.properties.headerCard,
                headerDropDownActions: this._metadataProps.properties.headerDropDownActions,
                headerLabel: this._metadataProps.properties.headerLabel,
                headerQuickActions: this._metadataProps.properties.headerQuickActions,
                headerSection: this._metadataProps.properties.headerSection,
                hierarchyParentPage: this._metadataProps.properties.hierarchyParentPage,
                idField: this._metadataProps.properties.idField,
                isDisabled: this._metadataProps.properties.isDisabled,
                isHidden: this._metadataProps.properties.isHidden,
                isTitleHidden: this._metadataProps.properties.isTitleHidden,
                isTransient: this._metadataProps.properties.isTransient,
                mode: this._metadataProps.properties.mode,
                module: this._metadataProps.properties.module,
                node: this._metadataProps.properties.node,
                objectTypePlural: this._metadataProps.properties.objectTypePlural,
                objectTypeSingular: this._metadataProps.properties.objectTypeSingular,
                on360ViewSwitched: this._metadataProps.properties.on360ViewSwitched,
                onClose: this._metadataProps.properties.onClose,
                onDirtyStateUpdated: this._metadataProps.properties.onDirtyStateUpdated,
                onError: this._metadataProps.properties.onError,
                onLoad: this._metadataProps.properties.onLoad,
                priority: this._metadataProps.properties.priority,
                skipDirtyCheck: this._metadataProps.properties.skipDirtyCheck,
                subtitle: this._metadataProps.properties.subtitle,
                title: this._metadataProps.properties.title,
                validation: this._metadataProps.properties.validation,
            };
            const properties = setDefaultProperties(applicableProperties, this._controlObjectConstructor.defaultUiProperties, ContainerKey.Page);
            if (this._metadataProps.properties.isTransient) {
                Object.keys(this._metadataProps.pageMetadata.uiComponentProperties).forEach(key => {
                    this._metadataProps.pageMetadata.uiComponentProperties[key].isTransient = true;
                });
            }
            this._metadataProps.pageMetadata.uiComponentProperties[this.elementId] = Object.assign(this._metadataProps.pageMetadata.uiComponentProperties[this.elementId] || {}, properties);
            this._metadataProps.pageMetadata.defaultUiComponentProperties[this.elementId] = {
                ...this._metadataProps.pageMetadata.defaultUiComponentProperties[this.elementId],
                ...properties,
            };
            this._metadataProps.pageMetadata.defaultUiComponentProperties[this.elementId].navigationPanel = this.getNavigationPanelProps();
            this._metadataProps.pageMetadata.uiComponentProperties[this.elementId].navigationPanel = this.getNavigationPanelProps();
            if (this._metadataProps.properties.navigationPanel) {
                const canFilter = this._metadataProps.properties.navigationPanel.canFilter;
                this._metadataProps.properties.navigationPanel.canFilter = canFilter === undefined ? true : canFilter;
                const nestedFields = this._metadataProps.properties.navigationPanel.listItem;
                if (nestedFields) {
                    withoutNestedTechnical(Object.values(nestedFields).filter(Boolean)).forEach(nestedField => {
                        nestedField.properties = setDefaultProperties(nestedField.properties, nestedField.defaultUiProperties, nestedField.type);
                        if (this._metadataProps.properties.node) {
                            applyDefaultValuesOnNestedField(this.nodeTypes, this.dataTypes, nestedField, String(this._metadataProps.properties.node));
                        }
                    });
                }
            }
            this._metadataProps.pageMetadata.businessActionsThunk = this._metadataProps.properties.businessActions;
            this._metadataProps.pageMetadata.defaultEntryThunk = this._metadataProps.properties.defaultEntry;
            if (this._metadataProps.properties.detailPanel) {
                this._metadataProps.pageMetadata.detailPanelThunk = (ctx) => {
                    const detailPanelProperties = this._metadataProps.properties.detailPanel.apply(ctx);
                    const layoutBuilder = new DetailPanelLayout(ctx, this.elementId, this.nodeTypes, this.dataTypes, {
                        pageMetadata: this._metadataProps.pageMetadata,
                        detailPanelHeader: detailPanelProperties.header,
                        detailPanelSections: detailPanelProperties.sections,
                    }).build();
                    const { detailPanelHeaderLayout, detailPanelSectionsLayout } = layoutBuilder.layout;
                    const containerControlObject = new DetailPanelControlObject({
                        screenId: this.elementId,
                        elementId: detailPanelId,
                        getUiComponentProperties: getUiComponentProperties,
                        headerSection: detailPanelHeaderLayout,
                        sections: detailPanelSectionsLayout,
                        setUiComponentProperties,
                        layout: {
                            detailPanelHeaderLayout: layoutBuilder.detailPanelHeaderLayout,
                            detailPanelSectionsLayout: layoutBuilder.detailPanelSectionsLayout,
                        },
                        footerActions: detailPanelProperties.footerActions,
                    });
                    this._metadataProps.pageMetadata.controlObjects[detailPanelId] = containerControlObject;
                    const uiComponentProperties = {
                        isHidden: detailPanelProperties.isHidden,
                        title: detailPanelProperties.title,
                        isTransient: detailPanelProperties.isTransient,
                        isDisabled: detailPanelProperties.isDisabled,
                        isExtendable: detailPanelProperties.isExtendable,
                        activeSection: detailPanelProperties.activeSection,
                        isCloseButtonHidden: detailPanelProperties.isCloseButtonHidden,
                    };
                    this._metadataProps.pageMetadata.uiComponentProperties[detailPanelId] = uiComponentProperties;
                    this._metadataProps.pageMetadata.defaultUiComponentProperties[detailPanelId] = {
                        ...uiComponentProperties,
                    };
                    return containerControlObject;
                };
            }
            // Collect hotkeys defined for various page elements
            Object.keys(this._metadataProps.pageMetadata.uiComponentProperties).forEach(elementId => {
                const element = this._metadataProps.pageMetadata.uiComponentProperties[elementId];
                if (element.shortcut) {
                    this._metadataProps.pageMetadata.elementsWithShortcut.push({
                        elementId,
                        shortcut: element.shortcut,
                    });
                }
            });
            return this;
        };
        this.getNavigationPanelProps = () => {
            const pageProperties = this._metadataProps.properties;
            if (pageProperties.isTransient || !pageProperties.node) {
                return undefined;
            }
            const contextNode = String(pageProperties.node);
            let navigationPanel = pageProperties.navigationPanel || null;
            // The application developer may define a navigation panel with a `null` value to override the automatic generation of the navigation panel.
            const hasOwnNavPanelDefinition = !!navigationPanel || Object.keys(pageProperties).includes('navigationPanel');
            const nodeType = pageProperties.node
                ? this.nodeTypes[schemaTypeNameFromNodeName(pageProperties.node)]
                : undefined;
            const dataType = nodeType?.defaultDataType ? this.dataTypes[nodeType.defaultDataType] : undefined;
            if (!navigationPanel && !hasOwnNavPanelDefinition && dataType?.value) {
                navigationPanel = {
                    listItem: getCardDefinitionFromReferenceDataType({
                        contextNode,
                        dataType,
                        dataTypes: this.dataTypes,
                        nodeTypes: this.nodeTypes,
                    }),
                };
            }
            if (!navigationPanel) {
                return undefined;
            }
            const state = xtremRedux.getStore().getState();
            let bulkActions = navigationPanel.bulkActions || [];
            if (state.exportConfigurationPage &&
                !pageProperties.isTransient &&
                pageProperties.node &&
                !isEmpty(this._metadataProps.pageMetadata?.exportTemplatesByNode?.[String(pageProperties.node)])) {
                /**
                 * If the page is associated with a node that has at least 1 export template, then we add an additional bulk action.
                 *  */
                bulkActions = [
                    ...bulkActions,
                    {
                        mutation: BULK_ACTION_ASYNC_EXPORT,
                        title: localize('@sage/xtrem-ui/bulk-action-async-export', 'Export'),
                        buttonType: 'tertiary',
                        icon: 'csv',
                        configurationPage: state.exportConfigurationPage,
                    },
                ];
            }
            const navigationPanelProperties = {
                extensionListItem: [],
                onSelectAfter: [],
                optionMenus: [],
                onOptionsMenuValueChangeAfter: [],
                onEmptyStateLinkClickAfter: [],
                isHeaderHidden: false,
                ...navigationPanel,
                bulkActions,
            };
            if (!navigationPanelProperties.menuType) {
                navigationPanelProperties.menuType = 'dropdown';
            }
            return navigationPanelProperties;
        };
    }
}
/**
 * Initializes the decorated member as a [Page]{@link PageControlObject} container with the provided properties
 *
 * @param properties The properties that the [Page]{@link PageControlObject} container will be initialized with
 */
export function page(properties) {
    return (ctor) => {
        const pageMetadata = getPageMetadata(ctor);
        pageMetadata.rootNode = properties.node ? String(properties.node) : undefined;
        pageMetadata.isTransient = properties.isTransient || false;
        pageMetadata.pageThunk = (nodeTypes, dataTypes) => {
            pageAction(standardActions.remove())(ctor.prototype, '$standardDeleteAction');
            pageAction(standardActions.save())(ctor.prototype, '$standardSaveAction');
            pageAction(standardActions.create())(ctor.prototype, '$standardNewAction');
            pageAction(standardActions.cancel())(ctor.prototype, '$standardCancelAction');
            pageAction(standardActions.duplicate())(ctor.prototype, '$standardDuplicateAction');
            pageAction(standardActions.executeDuplication())(ctor.prototype, '$standardExecuteDuplicationAction');
            pageAction(standardActions.openAttachmentsDialog())(ctor.prototype, '$standardOpenAttachmentsDialog');
            pageAction(standardActions.openCustomizationPageWizard())(ctor.prototype, '$standardOpenCustomizationPageWizardAction');
            pageAction(standardActions.openRecordHistory())(ctor.prototype, '$standardOpenRecordHistoryAction');
            pageAction(standardActions.confirmDialog())(ctor.prototype, '$standardDialogConfirmationAction');
            if (properties.node && nodeTypes) {
                addCustomFieldsToPageBody(nodeTypes, pageMetadata, ctor);
            }
            return new PageDecorator(ctor, pageMetadata.screenId, {
                pageMetadata,
                properties,
            }, ContainerKey.Page, nodeTypes, dataTypes, {
                getFocussedField,
                dispatchPageValidation: () => dispatchContainerValidation(pageMetadata.screenId, pageMetadata.screenId),
            }).build().controlObject;
        };
    };
}
//# sourceMappingURL=page-decorator.js.map