import * as React from 'react';
import { collectSections } from './dialog-utils';
import { useSelector, useDispatch } from 'react-redux';
import * as xtremRedux from '../../../redux';
import { XtremTabs } from '../../ui/tabs/xtrem-tabs';
import { resolveByValue } from '../../../utils/resolve-value-utils';
import { getPageDefinition } from '../../../utils/state-utils';
import { getSectionValidationMessage } from '../../../service/validation-service';
import { WizardSteps } from '../page/wizard-steps';
export function DialogHeader({ dialog, setSelectedSection, selectedSection, }) {
    const dispatch = useDispatch();
    const screenDefinitions = useSelector(state => state.screenDefinitions);
    const pageSelectedSection = useSelector(state => dialog.screenId && dialog.type === 'page'
        ? state.screenDefinitions[dialog.screenId]?.activeSection || null
        : null);
    const sectionValidationErrors = useSelector(state => {
        if (dialog.screenId && dialog.type === 'page') {
            const pageDefinition = getPageDefinition(dialog.screenId, state);
            if (!pageDefinition) {
                return {};
            }
            const sectionIds = pageDefinition.metadata.layout.$items
                .filter(section => !!section.$containerId)
                .map(section => section.$containerId);
            return sectionIds.reduce((result, sectionId) => {
                if (dialog.screenId) {
                    return {
                        ...result,
                        [sectionId]: getSectionValidationMessage(dialog.screenId, sectionId, state),
                    };
                }
                return result;
            }, {});
        }
        return {};
    });
    const onTabChange = React.useCallback((activeSection) => {
        if (dialog.type === 'page' && dialog.screenId) {
            dispatch(xtremRedux.actions.setActiveSection(dialog.screenId, activeSection));
        }
        else {
            setSelectedSection(activeSection);
        }
    }, [dispatch, dialog, setSelectedSection]);
    // For custom dialogs, the selected section is tracked in the local state of the dialog component
    // For page dialogs, the selected section is tracked in the application state.
    const sections = collectSections(dialog, screenDefinitions);
    const selectedTabId = pageSelectedSection || selectedSection || sections[0]?.id || null;
    const useTabs = sections.length > 1 &&
        // The tabs should only be displayed if the content is not page type OR if its a page which is set to tabs mode
        (dialog.type !== 'page' || dialog.content.mode === 'tabs');
    const useWizardSteps = sections.length > 1 &&
        // The tabs should only be displayed if the content is a page that is set to wizard mode
        dialog.type === 'page' &&
        dialog.content.mode === 'wizard';
    if (useTabs) {
        return (React.createElement(XtremTabs, { inSidebar: true, onTabChange: onTabChange, selectedTabId: selectedTabId || '', tabs: sections
                .map(section => ({
                id: section.id,
                title: resolveByValue({
                    propertyValue: section.title,
                    screenId: dialog.screenId || '',
                    fieldValue: null,
                    rowValue: null,
                    skipHexFormat: true,
                }),
                validationMessage: sectionValidationErrors[section.id],
            }))
                .filter(t => !!t.title) }));
    }
    if (useWizardSteps) {
        return React.createElement(WizardSteps, { screenId: dialog.screenId || '', sections: sections, selectedSection: selectedTabId });
    }
    return null;
}
//# sourceMappingURL=dialog-header.js.map