"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.unwrapError = exports.InteropError = exports.TimeoutError = exports.LogicError = exports.SecurityError = exports.SystemError = exports.AuthorizationError = exports.DataInputError = exports.BusinessRuleError = exports.ApplicationError = exports.BaseError = exports.ApiError = exports.ValidationSeverity = void 0;
const security_1 = require("./security");
var ValidationSeverity;
(function (ValidationSeverity) {
    ValidationSeverity[ValidationSeverity["test"] = 0] = "test";
    ValidationSeverity[ValidationSeverity["info"] = 1] = "info";
    ValidationSeverity[ValidationSeverity["warn"] = 2] = "warn";
    ValidationSeverity[ValidationSeverity["error"] = 3] = "error";
    ValidationSeverity[ValidationSeverity["exception"] = 4] = "exception";
})(ValidationSeverity || (exports.ValidationSeverity = ValidationSeverity = {}));
class ApiError extends Error {
    constructor(message, diagnoses) {
        super(message);
        this.diagnoses = diagnoses;
    }
}
exports.ApiError = ApiError;
class BaseError extends Error {
    constructor(message, innerError) {
        super(message);
        this.innerError = innerError;
        // The following statement is necessary for any class that inherits the Error class.
        // https://github.com/Microsoft/TypeScript-wiki/blob/master/Breaking-Changes.md#extending-built-ins-like-error-array-and-map-may-no-longer-work
        Object.setPrototypeOf(this, BaseError.prototype);
    }
}
exports.BaseError = BaseError;
class ApplicationError extends BaseError {
    constructor(message, innerError) {
        super(message.replace(/&quot;/g, '"').replace(/&#x27;/g, "'"), innerError);
        // The following statement is necessary for any class that inherits the Error class.
        // https://github.com/Microsoft/TypeScript-wiki/blob/master/Breaking-Changes.md#extending-built-ins-like-error-array-and-map-may-no-longer-work
        Object.setPrototypeOf(this, ApplicationError.prototype);
    }
}
exports.ApplicationError = ApplicationError;
/**
 * Thrown when a business rule is broken.
 * The message is localized, and reaches the end user.
 */
class BusinessRuleError extends ApplicationError {
    constructor(message, innerError, diagnoses) {
        super(message, innerError);
        // The following statement is necessary for any class that inherits the Error class.
        // https://github.com/Microsoft/TypeScript-wiki/blob/master/Breaking-Changes.md#extending-built-ins-like-error-array-and-map-may-no-longer-work
        Object.setPrototypeOf(this, BusinessRuleError.prototype);
        if (diagnoses) {
            diagnoses.forEach(d => {
                d.severity = d.severity || 3;
            });
        }
        this.extensions = {
            code: 'business-rule-error',
            diagnoses: diagnoses || [
                {
                    severity: 3,
                    message,
                    path: [],
                },
            ],
        };
    }
}
exports.BusinessRuleError = BusinessRuleError;
/**
 * Thrown when the data provided by the user is faulty.
 * The message is localized, and reaches the end user.
 */
class DataInputError extends ApplicationError {
    constructor(message, innerError) {
        super(message, innerError);
        // The following statement is necessary for any class that inherits the Error class.
        // https://github.com/Microsoft/TypeScript-wiki/blob/master/Breaking-Changes.md#extending-built-ins-like-error-array-and-map-may-no-longer-work
        Object.setPrototypeOf(this, DataInputError.prototype);
        this.extensions = {
            code: 'data-input-error',
            diagnoses: [
                {
                    severity: 3,
                    message,
                    path: [],
                },
            ],
        };
    }
}
exports.DataInputError = DataInputError;
/**
 * Thrown if the query tries to reach unauthorized data.
 * The message is localized, and reaches the end user.
 */
class AuthorizationError extends ApplicationError {
    constructor(message, innerError) {
        super(message, innerError);
        // The following statement is necessary for any class that inherits the Error class.
        // https://github.com/Microsoft/TypeScript-wiki/blob/master/Breaking-Changes.md#extending-built-ins-like-error-array-and-map-may-no-longer-work
        Object.setPrototypeOf(this, AuthorizationError.prototype);
        this.extensions = {
            code: 'authorization-error',
            diagnoses: [
                {
                    severity: 3,
                    message,
                    path: [],
                },
            ],
        };
    }
}
exports.AuthorizationError = AuthorizationError;
/**
 * Not a logical error nor an application error.
 * For instance: file not found or failed connection, or unknown.
 * Message not localized, not returned to user.
 */
class SystemError extends BaseError {
    constructor(message, innerError) {
        super(message, innerError);
        // The following statement is necessary for any class that inherits the Error class.
        // https://github.com/Microsoft/TypeScript-wiki/blob/master/Breaking-Changes.md#extending-built-ins-like-error-array-and-map-may-no-longer-work
        Object.setPrototypeOf(this, SystemError.prototype);
        this.extensions = {
            code: 'system-error',
            diagnoses: [
                {
                    severity: 3,
                    message,
                    path: [],
                },
            ],
        };
    }
}
exports.SystemError = SystemError;
/**
 * Security errors are reserved for errors that are related to security threats.
 * Message not localized, not returned to user.
 */
class SecurityError extends BaseError {
    constructor(message, innerError) {
        super(`[SECURITY] ${message}`, innerError);
        // The following statement is necessary for any class that inherits the Error class.
        // https://github.com/Microsoft/TypeScript-wiki/blob/master/Breaking-Changes.md#extending-built-ins-like-error-array-and-map-may-no-longer-work
        Object.setPrototypeOf(this, SecurityError.prototype);
        this.extensions = {
            code: 'security-error',
            diagnoses: [
                {
                    severity: 3,
                    message,
                    path: [],
                },
            ],
        };
    }
    securityMessage(context) {
        const details = (0, security_1.getRequestContextDetailsMessage)(context.request);
        return (this.stack ?? this.message).replace(this.message, `${this.message} - ${details}`);
    }
}
exports.SecurityError = SecurityError;
/**
 * Logical error, independent from external environment and data.
 * For instance: failed assertion.
 * Message not localized, not returned to user.
 */
class LogicError extends BaseError {
    constructor(message, innerError) {
        super(message, innerError);
        // The following statement is necessary for any class that inherits the Error class.
        // https://github.com/Microsoft/TypeScript-wiki/blob/master/Breaking-Changes.md#extending-built-ins-like-error-array-and-map-may-no-longer-work
        Object.setPrototypeOf(this, LogicError.prototype);
        this.extensions = {
            code: 'logic-error',
            diagnoses: [
                {
                    severity: 3,
                    message,
                    path: [],
                },
            ],
        };
    }
}
exports.LogicError = LogicError;
/**
 * Timeout error, thrown if the time limit for an operation has been exceeded
 * For instance: GraphQl time limit.
 * The message is localized, and reaches the end user.
 */
class TimeoutError extends ApplicationError {
    constructor(message, innerError) {
        super(message, innerError);
        // The following statement is necessary for any class that inherits the Error class.
        // https://github.com/Microsoft/TypeScript-wiki/blob/master/Breaking-Changes.md#extending-built-ins-like-error-array-and-map-may-no-longer-work
        Object.setPrototypeOf(this, TimeoutError.prototype);
        this.extensions = {
            code: 'timeout-error',
            diagnoses: [
                {
                    severity: 3,
                    message,
                    path: [],
                },
            ],
        };
    }
}
exports.TimeoutError = TimeoutError;
class InteropError extends BaseError {
    constructor(message, innerError) {
        super(message, innerError);
        // The following statement is necessary for any class that inherits the Error class.
        // https://github.com/Microsoft/TypeScript-wiki/blob/master/Breaking-Changes.md#extending-built-ins-like-error-array-and-map-may-no-longer-work
        Object.setPrototypeOf(this, InteropError.prototype);
        this.extensions = {
            code: 'interop-error',
            diagnoses: [
                {
                    severity: 3,
                    message,
                    path: [],
                },
            ],
        };
    }
}
exports.InteropError = InteropError;
function unwrapError(err) {
    const innerError = err.innerError;
    return innerError ? unwrapError(innerError) : err;
}
exports.unwrapError = unwrapError;
//# sourceMappingURL=errors.js.map