export declare enum ValidationSeverity {
    test = 0,
    info = 1,
    warn = 2,
    error = 3,
    exception = 4
}
export declare class ApiError extends Error {
    readonly diagnoses: any[];
    constructor(message: string, diagnoses: any[]);
}
export interface Diagnosis {
    severity?: ValidationSeverity;
    message?: string;
    path?: string[];
}
export interface HttpError {
    status: number;
    statusText?: string;
}
interface Extensions {
    code: string;
    diagnoses: any[];
    http?: HttpError;
    rawError?: any;
}
export declare abstract class BaseError extends Error {
    readonly innerError?: Error | undefined;
    /** This attribute will be used to propagate this diagnoses to the GraphQL response */
    extensions: Extensions;
    constructor(message: string, innerError?: Error | undefined);
}
export declare abstract class ApplicationError extends BaseError {
    constructor(message: string, innerError?: Error);
}
/**
 * Thrown when a business rule is broken.
 * The message is localized, and reaches the end user.
 */
export declare class BusinessRuleError extends ApplicationError {
    constructor(message: string, innerError?: Error, diagnoses?: Diagnosis[]);
}
/**
 * Thrown when the data provided by the user is faulty.
 * The message is localized, and reaches the end user.
 */
export declare class DataInputError extends ApplicationError {
    constructor(message: string, innerError?: Error);
}
/**
 * Thrown if the query tries to reach unauthorized data.
 * The message is localized, and reaches the end user.
 */
export declare class AuthorizationError extends ApplicationError {
    constructor(message: string, innerError?: Error);
}
/**
 * Not a logical error nor an application error.
 * For instance: file not found or failed connection, or unknown.
 * Message not localized, not returned to user.
 */
export declare class SystemError extends BaseError {
    constructor(message: string, innerError?: Error);
}
/**
 * Security errors are reserved for errors that are related to security threats.
 * Message not localized, not returned to user.
 */
export declare class SecurityError extends BaseError {
    constructor(message: string, innerError?: Error);
    securityMessage(context: any): string;
}
/**
 * Logical error, independent from external environment and data.
 * For instance: failed assertion.
 * Message not localized, not returned to user.
 */
export declare class LogicError extends BaseError {
    constructor(message: string, innerError?: Error);
}
/**
 * Timeout error, thrown if the time limit for an operation has been exceeded
 * For instance: GraphQl time limit.
 * The message is localized, and reaches the end user.
 */
export declare class TimeoutError extends ApplicationError {
    constructor(message: string, innerError?: Error);
}
export declare class InteropError extends BaseError {
    constructor(message: string, innerError?: Error);
}
export declare function unwrapError(err: Error): Error & {
    code?: string;
};
export {};
//# sourceMappingURL=errors.d.ts.map