"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const xtrem_shared_1 = require("@sage/xtrem-shared");
const chai_1 = require("chai");
const fs = require("fs");
const path = require("path");
const sinon = require("sinon");
const mt = require("../lib/transformers/message-transformer");
const test_helper_1 = require("./test-helper");
const FIXTURE_PAGE = path.resolve('./test/fixtures/test-app/lib/pages/action-buttons.ts');
const FIXTURE_PAGE_DECORATED = path.resolve('./test/fixtures/test-app/lib/pages/action-buttons-decorated.ts');
const FIXTURE_NODE_INDEX = path.resolve('./test/fixtures/test-app/lib/nodes/_index.ts');
const FIXTURE_NODE = path.resolve('./test/fixtures/test-app/lib/nodes/address.ts');
const FIXTURE_PAGE_INDEX = path.resolve('./test/fixtures/test-app/lib/pages/index.ts');
function checkLanguageFiles(stub, dictionaryContent) {
    const callNumber = stub.callCount;
    for (let i = 0; i < callNumber; i += 1) {
        const callArguments = stub.getCall(i).args;
        const filePath = callArguments[0].replace(/\\/g, '/'); // CC Necessary for Windows
        const fileName = filePath.split('/').pop().replace('.json', '');
        if (fileName !== 'base') {
            (0, chai_1.expect)(xtrem_shared_1.supportedLocales).contains(fileName);
        }
        (0, chai_1.expect)(filePath).match(new RegExp(`test-app/lib/i18n/${fileName}.json$`));
        const dictionary = JSON.parse(callArguments[1]);
        if (dictionaryContent) {
            (0, chai_1.expect)(dictionary).to.include(dictionaryContent[fileName]);
        }
        else {
            (0, chai_1.expect)(Object.keys(dictionary)).to.have.length(22);
        }
    }
}
function checkLanguageFilesUpdate(initialDictionary, expectedBaseContent, expectedLanguageContent, writeFileStub) {
    const mergedDictionary = {};
    Object.keys(initialDictionary).forEach(k => {
        const targetObj = k === 'base' ? expectedBaseContent : expectedLanguageContent;
        Object.keys(targetObj).forEach(dictionaryKey => {
            mergedDictionary[k] = {
                ...mergedDictionary[k],
                [dictionaryKey]: targetObj[dictionaryKey],
            };
        });
    });
    checkLanguageFiles(writeFileStub, mergedDictionary);
}
describe('Message Transformer', () => {
    describe('execution', () => {
        let visitorSpy;
        beforeEach(() => {
            visitorSpy = sinon.spy(mt, 'visitor');
        });
        afterEach(() => {
            visitorSpy.restore();
        });
        it('should include node files', () => {
            (0, test_helper_1.transpileFixture)(FIXTURE_NODE_INDEX, [mt.messageTransformer]);
            (0, test_helper_1.transpileFixture)(FIXTURE_NODE, [mt.messageTransformer]);
            (0, chai_1.expect)(visitorSpy.getCalls().length).to.be.equal(213);
        });
        it('should include page files', () => {
            (0, test_helper_1.transpileFixture)(FIXTURE_PAGE, [mt.messageTransformer]);
            (0, test_helper_1.transpileFixture)(FIXTURE_PAGE_INDEX, [mt.messageTransformer]);
            (0, chai_1.expect)(visitorSpy.getCalls().length).to.be.equal(463);
        });
    });
    xdescribe('file generation', () => {
        let dictionary;
        beforeEach(() => {
            readFileStub = (() => {
                const original = fs.readFileSync;
                return sinon
                    .stub(fs, 'readFileSync')
                    .callsFake((filePath, options) => {
                    if (typeof filePath !== 'string') {
                        return original(filePath, options);
                    }
                    const fileName = path.basename(filePath, '.json');
                    const result = dictionary && fileName in dictionary
                        ? JSON.stringify(dictionary[fileName])
                        : original(filePath, options);
                    return result;
                });
            })();
            writeFileStub = sinon.stub(fs, 'writeFileSync');
        });
        afterEach(() => {
            writeFileStub.restore();
            readFileStub.restore();
        });
        it('should create language files', () => {
            (0, test_helper_1.transpileFixture)(FIXTURE_PAGE_DECORATED, [mt.messageTransformer]);
            (0, chai_1.expect)(readFileStub.callCount).to.equal(14);
            (0, chai_1.expect)(writeFileStub.callCount).to.equal(12);
            checkLanguageFiles(writeFileStub);
        });
        let writeFileStub;
        let readFileStub;
        it('should update language files', () => {
            dictionary = xtrem_shared_1.supportedLocales.reduce((r, l) => {
                r[l] = {};
                return r;
            }, {});
            (0, test_helper_1.transpileFixture)(FIXTURE_PAGE_DECORATED, [mt.messageTransformer]);
            (0, chai_1.expect)(readFileStub.callCount).to.equal(14);
            (0, chai_1.expect)(writeFileStub.callCount).to.equal(12);
            const expectedLanguageFile = {
                '@sage/test-i18n-app/pages__action_buttons__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__decorator__onLoad': '',
                '@sage/test-i18n-app/pages__action_buttons__saveCrudAction__decorator__onClick': '',
                '@sage/test-i18n-app/pages__action_buttons__createCrudAction__decorator__onClick': '',
                '@sage/test-i18n-app/pages__action_buttons__createCrudAction__decorator__onClick__2': '',
                '@sage/test-i18n-app/pages__action_buttons__createCrudAction__decorator__onClick__3': '',
                '@sage/test-i18n-app/pages__action_buttons__deleteCrudAction__decorator__onClick': '',
                '@sage/test-i18n-app/pages__action_buttons__closeCrudAction__decorator__onClick': '',
                '@sage/test-i18n-app/pages__action_buttons__businessAction1__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__businessAction1__decorator__onClick': '',
                '@sage/test-i18n-app/pages__action_buttons__businessAction2__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__businessAction2__decorator__onClick': '',
                '@sage/test-i18n-app/pages__action_buttons__mainSection__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__mainBlock__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__resultField__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__controlSection__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__controlBlock__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__disableBusinessAction1__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__hideBusinessAction2__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__disableSaveCrudAction__decorator__title': '',
            };
            const expectedBaseFile = {
                '@sage/test-i18n-app/pages__action_buttons__businessAction1__decorator__onClick': 'Business action 1',
                '@sage/test-i18n-app/pages__action_buttons__businessAction1__decorator__title': 'Business action 1',
                '@sage/test-i18n-app/pages__action_buttons__businessAction2__decorator__onClick': 'Business action 2',
                '@sage/test-i18n-app/pages__action_buttons__businessAction2__decorator__title': 'Business action 2',
                '@sage/test-i18n-app/pages__action_buttons__closeCrudAction__decorator__onClick': 'Close CRUD Button',
                '@sage/test-i18n-app/pages__action_buttons__controlBlock__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__controlSection__decorator__title': 'Controlling Buttons',
                '@sage/test-i18n-app/pages__action_buttons__createCrudAction__decorator__onClick': 'Create CRUD Button',
                '@sage/test-i18n-app/pages__action_buttons__createCrudAction__decorator__onClick__2': 'Second string literal',
                '@sage/test-i18n-app/pages__action_buttons__createCrudAction__decorator__onClick__3': 'Third string literal',
                '@sage/test-i18n-app/pages__action_buttons__decorator__onLoad': 'This page show cases the CRUD and business actions.\\nThe actions are located outside the page body, on the title line and on the right-side panel.',
                '@sage/test-i18n-app/pages__action_buttons__decorator__title': 'Action Buttons',
                '@sage/test-i18n-app/pages__action_buttons__deleteCrudAction__decorator__onClick': 'Delete CRUD Button',
                '@sage/test-i18n-app/pages__action_buttons__disableBusinessAction1__decorator__title': 'Toggle Business Action 1 disabled',
                '@sage/test-i18n-app/pages__action_buttons__disableSaveCrudAction__decorator__title': 'Toggle Save CRUD action disabled',
                '@sage/test-i18n-app/pages__action_buttons__hideBusinessAction2__decorator__title': 'Toggle Business Action 1 hidden',
                '@sage/test-i18n-app/pages__action_buttons__mainBlock__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__mainSection__decorator__title': 'About this page',
                '@sage/test-i18n-app/pages__action_buttons__resultField__decorator__title': 'Selected action',
                '@sage/test-i18n-app/pages__action_buttons__saveCrudAction__decorator__onClick': 'Save CRUD Button',
            };
            checkLanguageFilesUpdate(dictionary, expectedBaseFile, expectedLanguageFile, writeFileStub);
        });
        it('should delete orphan keys from language files', () => {
            dictionary = xtrem_shared_1.supportedLocales.reduce((r, l) => {
                r[l] = l === 'base' ? {} : { hello: 'world' };
                return r;
            }, {});
            (0, test_helper_1.transpileFixture)(FIXTURE_PAGE_DECORATED, [mt.messageTransformer]);
            (0, chai_1.expect)(readFileStub.callCount).to.equal(14);
            (0, chai_1.expect)(writeFileStub.callCount).to.equal(12);
            const expectedLanguageFile = {
                '@sage/test-i18n-app/pages__action_buttons__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__decorator__onLoad': '',
                '@sage/test-i18n-app/pages__action_buttons__saveCrudAction__decorator__onClick': '',
                '@sage/test-i18n-app/pages__action_buttons__createCrudAction__decorator__onClick': '',
                '@sage/test-i18n-app/pages__action_buttons__createCrudAction__decorator__onClick__2': '',
                '@sage/test-i18n-app/pages__action_buttons__createCrudAction__decorator__onClick__3': '',
                '@sage/test-i18n-app/pages__action_buttons__deleteCrudAction__decorator__onClick': '',
                '@sage/test-i18n-app/pages__action_buttons__closeCrudAction__decorator__onClick': '',
                '@sage/test-i18n-app/pages__action_buttons__businessAction1__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__businessAction1__decorator__onClick': '',
                '@sage/test-i18n-app/pages__action_buttons__businessAction2__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__businessAction2__decorator__onClick': '',
                '@sage/test-i18n-app/pages__action_buttons__mainSection__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__mainBlock__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__resultField__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__controlSection__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__controlBlock__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__disableBusinessAction1__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__hideBusinessAction2__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__disableSaveCrudAction__decorator__title': '',
            };
            const expectedBaseFile = {
                '@sage/test-i18n-app/pages__action_buttons__businessAction1__decorator__onClick': 'Business action 1',
                '@sage/test-i18n-app/pages__action_buttons__businessAction1__decorator__title': 'Business action 1',
                '@sage/test-i18n-app/pages__action_buttons__businessAction2__decorator__onClick': 'Business action 2',
                '@sage/test-i18n-app/pages__action_buttons__businessAction2__decorator__title': 'Business action 2',
                '@sage/test-i18n-app/pages__action_buttons__closeCrudAction__decorator__onClick': 'Close CRUD Button',
                '@sage/test-i18n-app/pages__action_buttons__controlBlock__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__controlSection__decorator__title': 'Controlling Buttons',
                '@sage/test-i18n-app/pages__action_buttons__createCrudAction__decorator__onClick': 'Create CRUD Button',
                '@sage/test-i18n-app/pages__action_buttons__createCrudAction__decorator__onClick__2': 'Second string literal',
                '@sage/test-i18n-app/pages__action_buttons__createCrudAction__decorator__onClick__3': 'Third string literal',
                '@sage/test-i18n-app/pages__action_buttons__decorator__onLoad': 'This page show cases the CRUD and business actions.\\nThe actions are located outside the page body, on the title line and on the right-side panel.',
                '@sage/test-i18n-app/pages__action_buttons__decorator__title': 'Action Buttons',
                '@sage/test-i18n-app/pages__action_buttons__deleteCrudAction__decorator__onClick': 'Delete CRUD Button',
                '@sage/test-i18n-app/pages__action_buttons__disableBusinessAction1__decorator__title': 'Toggle Business Action 1 disabled',
                '@sage/test-i18n-app/pages__action_buttons__disableSaveCrudAction__decorator__title': 'Toggle Save CRUD action disabled',
                '@sage/test-i18n-app/pages__action_buttons__hideBusinessAction2__decorator__title': 'Toggle Business Action 1 hidden',
                '@sage/test-i18n-app/pages__action_buttons__mainBlock__decorator__title': '',
                '@sage/test-i18n-app/pages__action_buttons__mainSection__decorator__title': 'About this page',
                '@sage/test-i18n-app/pages__action_buttons__resultField__decorator__title': 'Selected action',
                '@sage/test-i18n-app/pages__action_buttons__saveCrudAction__decorator__onClick': 'Save CRUD Button',
            };
            checkLanguageFilesUpdate(dictionary, expectedBaseFile, expectedLanguageFile, writeFileStub);
        });
    });
});
//# sourceMappingURL=message-transformer-test.js.map