"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.nodePropertyNameExtractor = exports.getOperationDecoratorParameters = exports.visitor = void 0;
const xtrem_log_1 = require("@sage/xtrem-log");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const lodash_1 = require("lodash");
const ts = require("typescript");
const utils_1 = require("../utils");
const transformer_utils_1 = require("./transformer-utils");
const excludedAstTypes = new Set([
    ts.SyntaxKind.ImportDeclaration,
    ts.SyntaxKind.TypePredicate,
    ts.SyntaxKind.TypeReference,
    ts.SyntaxKind.PropertyAssignment,
]);
const includedOperations = ['mutation', 'asyncMutation', 'bulkMutation', 'query'];
const logger = xtrem_log_1.Logger.getLogger(__filename, 'i18n-resolver');
/**
 * Top-level TS visitor
 * @param prefix prefix for translation keys
 */
const visitor = (dictionary, packageName, dirName, filename, ctx, nodePrefix) => (node) => {
    if (excludedAstTypes.has(node.kind)) {
        return node;
    }
    let prefix = nodePrefix;
    if (ts.isClassDeclaration(node)) {
        const nodeName = node.name?.getText().trim();
        prefix = `${packageName}/${(0, transformer_utils_1.createDictionaryKey)(dirName, (0, lodash_1.snakeCase)(nodeName))}`;
        if (dirName === 'nodes')
            dictionary[`${prefix}__node_name`] = (0, xtrem_shared_1.titleCase)(nodeName);
        const decorator = ts.getDecorators(node)?.[0];
        if (decorator) {
            const bulkMutationNames = (0, utils_1.getCrudBulkMutationNames)(filename, decorator);
            bulkMutationNames.forEach(bulkMutationName => {
                dictionary[`${prefix}__bulkMutation__${bulkMutationName}`] = (0, xtrem_shared_1.titleCase)(bulkMutationName);
            });
            const isAbstract = (0, xtrem_shared_1.isAbstractNodeDecorator)(filename, decorator);
            const isSubNode = (0, xtrem_shared_1.isSubNodeDecorator)(decorator);
            const isNodeExtension = (0, xtrem_shared_1.isNodeExtensionDecorator)(decorator);
            const canExport = (0, xtrem_shared_1.canExportNodeDecorator)(filename, decorator);
            const storage = (0, xtrem_shared_1.getNodeStorageType)(filename, decorator);
            if (!isAbstract && canExport && !isNodeExtension && (storage === "'sql'" || isSubNode)) {
                dictionary[`${prefix}__asyncMutation__asyncExport`] = 'Export';
                dictionary[`${prefix}__asyncMutation__asyncExport__parameter__id`] = 'ID';
                dictionary[`${prefix}__asyncMutation__asyncExport__parameter__filter`] = 'Filter';
            }
        }
    }
    if (ts.isPropertyDeclaration(node) && !(0, lodash_1.isEmpty)(ts.getDecorators(node))) {
        const propertyName = node.name.getText().trim();
        (0, transformer_utils_1.addDictionaryEntry)(dictionary, `${prefix}__property__${propertyName}`, (0, xtrem_shared_1.titleCase)(propertyName));
        return node;
    }
    if (ts.isMethodDeclaration(node)) {
        const decorator = ts.getDecorators(node)?.[0];
        if (decorator?.expression &&
            ts.isCallExpression(decorator.expression) &&
            ts.isPropertyAccessExpression(decorator.expression?.expression)) {
            const decoratorName = decorator.expression.expression.name?.getText().trim();
            if (includedOperations.includes(decoratorName)) {
                const nodeName = node.name?.getText().trim();
                const operationEntry = `${prefix}__${decoratorName}__${nodeName}`;
                const parametersFromDecorator = getOperationDecoratorParameters(filename, decorator) || [];
                logger.verbose(() => `------Extract i18n entry from ${filename}`);
                logger.verbose(() => `   operationEntry ${operationEntry}`);
                logger.verbose(() => `       parametersFromDecorator ${parametersFromDecorator}`);
                const parameters = node.parameters
                    .map(parameter => {
                    return parameter.name?.getText().trim();
                })
                    .filter((_name, idx) => {
                    return (
                    // for bulkMutation, skip the second parameter
                    !(decoratorName === 'bulkMutation' && idx === 1) &&
                        // context parameter position (first position) is skipped
                        idx !== 0);
                });
                logger.verbose(() => `       parameters ${parameters}`);
                if (!(0, lodash_1.isEqual)(parametersFromDecorator, parameters)) {
                    let difference = (0, lodash_1.differenceWith)(parameters, parametersFromDecorator, lodash_1.isEqual);
                    if (difference.length === 0)
                        difference = (0, lodash_1.differenceWith)(parametersFromDecorator, parameters, lodash_1.isEqual);
                    throw new Error(`There is a difference between parameters in the decorator and parameters in the function declaration of '${nodeName}' operation: '${difference}'`);
                }
                (0, transformer_utils_1.addDictionaryEntry)(dictionary, operationEntry, (0, xtrem_shared_1.titleCase)(nodeName));
                parameters.forEach(parameter => {
                    (0, transformer_utils_1.addDictionaryEntry)(dictionary, `${operationEntry}__parameter__${parameter}`, (0, xtrem_shared_1.titleCase)(parameter));
                });
                return node;
            }
        }
    }
    return ts.visitEachChild(node, (0, exports.visitor)(dictionary, packageName, dirName, filename, ctx, prefix), ctx);
};
exports.visitor = visitor;
function getOperationDecoratorParameters(filename, node) {
    const decoratorArguments = (0, xtrem_shared_1.getDecoratorArguments)(filename, node);
    const parameters = decoratorArguments.find(argument => ts.isPropertyAssignment(argument) &&
        argument.initializer &&
        argument.initializer.kind === ts.SyntaxKind.ArrayLiteralExpression &&
        ts.isIdentifier(argument.name) &&
        argument.name.escapedText.toString() === 'parameters');
    if (parameters &&
        ts.isPropertyAssignment(parameters) &&
        parameters.initializer &&
        parameters.initializer.kind === ts.SyntaxKind.ArrayLiteralExpression &&
        ts.isArrayLiteralExpression(parameters.initializer))
        return parameters.initializer.elements.map(param => {
            const nameProperty = ts.isObjectLiteralExpression(param) && param.properties
                ? param.properties?.find(prop => prop.name?.getText() === 'name')
                : undefined;
            if (!nameProperty)
                logger.warn('"name" property undefined');
            return nameProperty && ts.isPropertyAssignment(nameProperty) && ts.isStringLiteral(nameProperty.initializer)
                ? (0, lodash_1.trim)(nameProperty.initializer.getText(), '"\'')
                : undefined;
        });
    return undefined;
}
exports.getOperationDecoratorParameters = getOperationDecoratorParameters;
/**
 * This transformer is meant to be run BEFORE the 'message-transformer' and its purpose is
 * to wrap some page/sticker decorator properties (see 'includedProperties') with a call to the 'ui.localize' function.
 *
 * @param ctx transformation context
 * @returns the transformed file
 */
function nodePropertyNameExtractor(ctx) {
    return (file) => {
        if (!(0, transformer_utils_1.isNodeLikeArtifactFile)(file) || (0, transformer_utils_1.isTestFile)(file)) {
            return file;
        }
        const filename = file.fileName;
        try {
            const dictionary = {};
            const dirName = (0, utils_1.getDirName)(file, true);
            const nameAndRoot = (0, utils_1.getPackageNameAndRoot)(filename);
            const packageName = nameAndRoot.name;
            const result = ts.visitNode(file, (0, exports.visitor)(dictionary, packageName, dirName, filename, ctx));
            const key = `${packageName}/package__name`;
            dictionary[key] = (0, xtrem_shared_1.titleCase)(packageName);
            (0, transformer_utils_1.writeStringLiteralsToBase)(dictionary, nameAndRoot.root);
            return result;
        }
        catch (err) {
            throw new Error(`${filename}: decorator transformer failed: ${err.message}`);
        }
    };
}
exports.nodePropertyNameExtractor = nodePropertyNameExtractor;
//# sourceMappingURL=node-property-name-extractor.js.map