import type { Editor } from '@ckeditor/ckeditor5-core';
import type { Conversion, Schema, ViewElement, BoxSides } from '@ckeditor/ckeditor5-engine';
/**
 * Conversion helper for upcasting attributes using normalized styles.
 *
 * @param options.modelAttribute The attribute to set.
 * @param options.styleName The style name to convert.
 * @param options.viewElement The view element name that should be converted.
 * @param options.defaultValue The default value for the specified `modelAttribute`.
 * @param options.shouldUpcast The function which returns `true` if style should be upcasted from this element.
 */
export declare function upcastStyleToAttribute(conversion: Conversion, options: {
    modelAttribute: string;
    styleName: string;
    viewElement: string | RegExp;
    defaultValue: string;
    reduceBoxSides?: boolean;
    shouldUpcast?: (viewElement: ViewElement) => boolean;
}): void;
export interface StyleValues {
    color: string;
    style: string;
    width: string;
}
/**
 * Conversion helper for upcasting border styles for view elements.
 *
 * @param defaultBorder The default border values.
 * @param defaultBorder.color The default `borderColor` value.
 * @param defaultBorder.style The default `borderStyle` value.
 * @param defaultBorder.width The default `borderWidth` value.
 */
export declare function upcastBorderStyles(conversion: Conversion, viewElementName: string, modelAttributes: StyleValues): void;
/**
 * Conversion helper for downcasting an attribute to a style.
 */
export declare function downcastAttributeToStyle(conversion: Conversion, options: {
    modelElement: string;
    modelAttribute: string;
    styleName: string;
}): void;
/**
 * Conversion helper for downcasting attributes from the model table to a view table (not to `<figure>`).
 */
export declare function downcastTableAttribute(conversion: Conversion, options: {
    modelAttribute: string;
    styleName: string;
}): void;
export declare function enableTableBorderProperties(schema: Schema, conversion: Conversion): void;
/**
 * Enables the `'tableCellBorderStyle'`, `'tableCellBorderColor'` and `'tableCellBorderWidth'` attributes for table cells.
 *
 * @param defaultBorder The default border values.
 * @param defaultBorder.color The default `tableCellBorderColor` value.
 * @param defaultBorder.style The default `tableCellBorderStyle` value.
 * @param defaultBorder.width The default `tableCellBorderWidth` value.
 */
export declare function enableTableCellBorderProperties(schema: Schema, conversion: Conversion): void;
/**
 * Enables the `'tableCellHorizontalAlignment'` attribute for table cells.
 *
 * @param defaultValue The default horizontal alignment value.
 */
export declare function enableHorizontalAlignmentProperty(schema: Schema, conversion: Conversion, defaultValue: string): void;
/**
 * Enables the `'verticalAlignment'` attribute for table cells.
 *
 * @param defaultValue The default vertical alignment value.
 */
export declare function enableVerticalAlignmentProperty(schema: Schema, conversion: Conversion, defaultValue: string): void;
/**
 * Enables conversion for an attribute for simple view-model mappings.
 *
 * @param options.defaultValue The default value for the specified `modelAttribute`.
 */
export declare function enableTableProperty(schema: Schema, conversion: Conversion, options: {
    modelAttribute: string;
    styleName: string;
    defaultValue: string;
}): void;
/**
 * Enables conversion for an attribute for simple view-model mappings.
 *
 * @param options.defaultValue The default value for the specified `modelAttribute`.
 */
export declare function enableCellProperty(schema: Schema, conversion: Conversion, options: {
    modelAttribute: string;
    styleName: string;
    defaultValue: string;
    reduceBoxSides?: boolean;
}): void;
/**
 * Returns a string if all four values of box sides are equal.
 *
 * If a string is passed, it is treated as a single value (pass-through).
 *
 * ```ts
 * // Returns 'foo':
 * getSingleValue( { top: 'foo', right: 'foo', bottom: 'foo', left: 'foo' } );
 * getSingleValue( 'foo' );
 *
 * // Returns undefined:
 * getSingleValue( { top: 'foo', right: 'foo', bottom: 'bar', left: 'foo' } );
 * getSingleValue( { top: 'foo', right: 'foo' } );
 * ```
 */
export declare function getSingleValue(objectOrString: BoxSides | string | undefined): string | undefined;
/**
 * Adds a unit to a value if the value is a number or a string representing a number.
 *
 * **Note**: It does nothing to non-numeric values.
 *
 * ```ts
 * getSingleValue( 25, 'px' ); // '25px'
 * getSingleValue( 25, 'em' ); // '25em'
 * getSingleValue( '25em', 'px' ); // '25em'
 * getSingleValue( 'foo', 'px' ); // 'foo'
 * ```
 *
 * @param defaultUnit A default unit added to a numeric value.
 */
export declare function addDefaultUnitToNumericValue(value: string | number | undefined, defaultUnit: string): string | number | undefined;
export interface NormalizedDefaultProperties {
    borderStyle: string;
    borderWidth: string;
    borderColor: string;
    backgroundColor: string;
    width: string;
    height: string;
    alignment?: string;
    padding?: string;
    verticalAlignment?: string;
    horizontalAlignment?: string;
}
/**
 * Returns the normalized configuration.
 *
 * @param options.includeAlignmentProperty Whether the "alignment" property should be added.
 * @param options.includePaddingProperty Whether the "padding" property should be added.
 * @param options.includeVerticalAlignmentProperty Whether the "verticalAlignment" property should be added.
 * @param options.includeHorizontalAlignmentProperty Whether the "horizontalAlignment" property should be added.
 * @param options.isRightToLeftContent Whether the content is right-to-left.
 */
export declare function getNormalizedDefaultProperties(config: Partial<NormalizedDefaultProperties> | undefined, options?: {
    includeAlignmentProperty?: boolean;
    includePaddingProperty?: boolean;
    includeVerticalAlignmentProperty?: boolean;
    includeHorizontalAlignmentProperty?: boolean;
    isRightToLeftContent?: boolean;
}): NormalizedDefaultProperties;
/**
 * Enables the `'alignment'` attribute for table.
 *
 * @param defaultValue The default alignment value.
 */
export declare function enableTableAlignmentProperty(schema: Schema, conversion: Conversion, defaultValue: string): void;
/**
 * Register table border and background attributes converters.
 */
export declare function downcastTableBorderAndBackgroundAttributes(editor: Editor): void;
/**
 * Enables the `'tableInsertPageBreak'` attribute for table row.
 *
 * @param defaultPageBreak The default page break value.
 */
export declare function enableTableRowProperties(schema: Schema, conversion: Conversion): void;
export declare function upcastPageBreak(conversion: Conversion, viewElementName: string, modelAttribute: string): void;
//# sourceMappingURL=query-table-editing-utils.d.ts.map