import { addDefaultUnitToNumericValue, getSingleValue } from '../query-table-editing-utils';
import QueryTableCellPropertyCommand from './query-table-cell-property-command';
/**
 * The table cell padding command.
 *
 * The command is registered by the {@link module:table/tablecellproperties/tablecellpropertiesediting~TableCellPropertiesEditing} as
 * the `'tableCellPadding'` editor command.
 *
 * To change the padding of selected cells, execute the command:
 *
 * ```ts
 * editor.execute( 'tableCellPadding', {
 *   value: '5px'
 * } );
 * ```
 *
 * **Note**: This command adds the default `'px'` unit to numeric values. Executing:
 *
 * ```ts
 * editor.execute( 'tableCellPadding', {
 *   value: '5'
 * } );
 * ```
 *
 * will set the `padding` attribute to `'5px'` in the model.
 */
export default class QueryTableCellPaddingCommand extends QueryTableCellPropertyCommand {
    /**
     * Creates a new `TableCellPaddingCommand` instance.
     *
     * @param editor An editor in which this command will be used.
     * @param defaultValue The default value of the attribute.
     */
    constructor(editor, defaultValue) {
        super(editor, 'queryTableCellPadding', defaultValue);
    }
    /**
     * @inheritDoc
     */
    _getAttribute(tableCell) {
        if (!tableCell) {
            return undefined;
        }
        return getSingleValue(tableCell.getAttribute(this.attributeName));
    }
    /**
     * @inheritDoc
     */
    _getValueToSet(value) {
        return addDefaultUnitToNumericValue(value, 'px');
    }
}
//# sourceMappingURL=query-table-cell-padding-command.js.map