/** @packageDocumentation @module date */
import { LocalizeLocale } from '@sage/xtrem-shared';
import * as datetime from './datetime';
import * as time from './time';
/** Integer type, to distinguish integers in the API documentation */
export type integer = number;
export declare const abbreviatedMonthNames: string[];
export declare const monthNames: string[];
/** Day of the week */
export declare enum WeekDay {
    sunday = 0,
    monday = 1,
    tuesday = 2,
    wednesday = 3,
    thursday = 4,
    friday = 5,
    saturday = 6
}
export declare const abbreviatedDayNames: string[];
export type WeekDayName = (typeof dayNames)[number];
export declare const dayNames: readonly ["Sunday", "Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday"];
export declare function isLeap(year: number): boolean;
export declare function daysInMonth(year: number, month: number): number;
export type DateDiff = {
    years?: number;
    months?: number;
    weeks?: number;
    days?: number;
};
/**
 * A `DateValue` represents a date.
 *
 * This class is called `DateValue` rather than Date to avoid confusions with JavaScript's built-in `Date` class.
 *
 * The main differences with JavaScript `Date` are:
 *
 * - `DateValue` is a pure date, without any time nor timezone information. Only year, month and day.
 * - `DateValue` is immutable.
 * - The `month` property takes values between 1 and 12, not 0 and 11.
 *
 * The API uses a simple chaining style. For example:
 *
 * ```ts
 * // the current date
 * const today = DateValue.today();
 * // the end of this month, as another date
 * const endOfThisMonth = today.endOfThisMonth();
 * // the week day of the first day of next month
 * const nextMonthWeekStart = today.begOfMonth().addMonth(1).weekDay;
 * ```
 */
export declare class DateValue {
    /**
     * The internal value for the date, as a number YYYYMMDD
     * This is *not* the number of milliseconds since 1970, but the year, month and day packed into a single number.
     */
    private readonly _value;
    constructor(value: number);
    get constructorName(): string;
    /**
     * The date's internal value.
     *
     * Note: this is *not* the UNIX timestamp but the year, month and day packed into a single number,
     * for example 20200514 for 2020-05-14
     */
    get value(): integer;
    /** Number of seconds since 1970-01-01 */
    get epoch(): integer;
    /** The year, between 0 and 9999. */
    get year(): integer;
    /** The month, between 1 and 12. */
    get month(): integer;
    /** The day of the month, between 1 and 31. */
    get day(): integer;
    /** The day of the week, between 0 (Sunday) and 6 (Saturday).  */
    get weekDay(): WeekDay;
    /** The day of the year, between 1 and 366. */
    get yearDay(): integer;
    /**
     * The week number, between 1 and 53, as defined by [ISO 8601](https://en.wikipedia.org/wiki/ISO_8601#Week_dates).
     *
     * Week 1 is the beginning of the week that contains January 4th.
     */
    get week(): integer;
    valueOf(): string;
    /**
     * Compares two dates.
     *
     * `date1.compare(date2)` returns a positive integer if `date1 > date2`,
     * a negative integer if `date1 < date2`, and 0 if the dates are equal.
     */
    compare(date: DateValue): integer;
    /** Tests date equality */
    equals(date: DateValue): boolean;
    /** Is the date between `begin` and `end`? (inclusive) */
    isBetween(begin: DateValue, end: DateValue): boolean;
    /** Is the date in a leap year? */
    isLeapYear(): boolean;
    /** Is the date a work day? (between Monday and Friday) */
    isWorkDay(): boolean;
    /** The number of days in the date's month, between 28 and 31. */
    daysInMonth(): number;
    /** The 1st of January of the date's year. */
    begOfYear(): DateValue;
    /** The 31st of December of the date's year. */
    endOfYear(): DateValue;
    /** The first day of the date's quarter. */
    begOfQuarter(): DateValue;
    /** The last day of the date's quarter. */
    endOfQuarter(): DateValue;
    /** The first day of the date's month. */
    begOfMonth(): DateValue;
    /** The last day of the date's month. */
    endOfMonth(): DateValue;
    /** Date for a given day in the same month as this date. */
    sameMonth(day: number): DateValue;
    /** The last occurence of a given day of the month before this date. */
    pastDay(day: number, includeThis?: boolean): DateValue;
    /** The next occurence of a given day of the month after this date. */
    futureDay(day: number, includeThis?: boolean): DateValue;
    /** The last occurence of the same day in a given month before this date. */
    pastMonth(month: number, includeThis?: boolean): DateValue;
    /** The next occurence of the same day in a given month after this date. */
    futureMonth(month: number, includeThis?: boolean): DateValue;
    /** The beginning of the week which contains this date. */
    begOfWeek(startDay?: WeekDay): DateValue;
    /** The end of the week which contains this date. */
    endOfWeek(startDay?: WeekDay): DateValue;
    /** The requested week day in the same week as this date. */
    sameWeek(weekDay: WeekDay, startDay?: WeekDay): DateValue;
    /** The last requested week day before this date. */
    pastWeekDay(weekDay: WeekDay, includeThis?: boolean): DateValue;
    /** The next requested week day after this date. */
    futureWeekDay(weekDay: WeekDay, includeThis?: boolean): DateValue;
    /** The week number between 0 and 53. */
    weekNumber(firstDayOfWeek: WeekDay): integer;
    /** Adds `years` years to the date. */
    addYears(years: integer): DateValue;
    /** Adds `months` months to the date. */
    addMonths(months: integer): DateValue;
    /** Adds `weeks` weeks to the date. */
    addWeeks(weeks: integer): DateValue;
    /** Adds `days` days to the date. */
    addDays(days: integer): DateValue;
    /** The number of days between two dates. */
    daysDiff(date: DateValue): integer;
    /**
     * Adds a delta which may combine years, months and days to the date.
     *
     * @example `d.add({ months: 2 })` adds 2 months to `d`.
     */
    add(delta: DateDiff): DateValue;
    toString(): string;
    /**
     * formats the date according to `format`
     * example: date.toString(format = 'YYYY-MM-DD')
     * @param locale LocalizeLocale
     * @param format string
     * @returns string
     */
    format(fmt: string, locale?: LocalizeLocale): string;
    at(t: time.Time, timeZone?: string, millisecond?: number): datetime.Datetime;
    toJsDate(timeZone?: string | undefined): Date;
    toJSON(): string;
    isNull(): boolean;
    static amDesignator(locale?: LocalizeLocale): string;
    static pmDesignator(locale?: LocalizeLocale): string;
    static monthName(month: number, locale?: LocalizeLocale, abbrev?: boolean): string;
    static monthFromName(name: string, locale?: LocalizeLocale): number;
    static weekDayName(weekDay: number, locale?: LocalizeLocale, abbrev?: boolean): string;
    static weekDayFromName(name: string, locale?: LocalizeLocale): number;
    static parse(str: string, locale?: LocalizeLocale, format?: string): DateValue;
    static today(timeZone?: string | undefined): DateValue;
    static fromJsDate(jsDate: Date, timeZone?: string | undefined): DateValue;
    static fromInternalValue(value: number): DateValue;
    static make(year: number, month: number, day: number): DateValue;
    static makeInWeek(year: number, week: number, wday: number): DateValue;
    static getWeekDayNames(startDay?: WeekDayName): readonly WeekDayName[];
    static getWeekDayNumbers(startDay?: WeekDay): readonly WeekDay[];
    static isDate(obj: any): obj is DateValue;
}
export declare const amDesignator: typeof DateValue.amDesignator;
export declare const pmDesignator: typeof DateValue.pmDesignator;
export declare const monthName: typeof DateValue.monthName;
export declare const monthFromName: typeof DateValue.monthFromName;
export declare const weekDayName: typeof DateValue.weekDayName;
export declare const weekDayFromName: typeof DateValue.weekDayFromName;
export declare const parse: typeof DateValue.parse;
export declare const today: typeof DateValue.today;
export declare const fromJsDate: typeof DateValue.fromJsDate;
export declare const fromInternalValue: typeof DateValue.fromInternalValue;
export declare const make: typeof DateValue.make;
export declare const makeInWeek: typeof DateValue.makeInWeek;
export declare const isDate: typeof DateValue.isDate;
export declare const getWeekDayNames: typeof DateValue.getWeekDayNames;
export declare const getWeekDayNumbers: typeof DateValue.getWeekDayNumbers;
//# sourceMappingURL=date.d.ts.map