import { Dict } from '@sage/xtrem-shared';
import { StaticThis } from '../decorators';
import { ActivityExtension } from './activity-extension';
import { Node } from './node';
/**
 * Grant the permissions on a set of activities
 */
export interface PermissionGrant {
    permissions: string[];
    on: (() => Activity)[];
}
/**
 * Grant the operations from a set of nodes
 */
export interface OperationGrant {
    operations: string[];
    on?: (() => {
        new (): Node;
    })[];
}
export declare class InternalBaseGrant {
    readonly type: string;
    constructor(type: string);
    /**
     * Key that uniquely identifies the grant
     */
    get key(): string;
    /**
     * This is an instance of a InternalOperationGrant
     * @returns
     */
    isOperationGrant(): this is InternalOperationGrant;
    /**
     * This is an instance of a InternalPermissionGrant
     * @returns
     */
    isPermissionGrant(): this is InternalPermissionGrant;
}
export declare class InternalOperationGrant extends InternalBaseGrant {
    readonly operation: string;
    readonly node: string;
    constructor(operation: string, node: string);
    /**
     * Key that uniquely identifies the grant
     */
    get key(): string;
}
export declare class InternalPermissionGrant extends InternalBaseGrant {
    readonly permission: string;
    readonly activity: string;
    constructor(permission: string, activity: string);
    /**
     * Key that uniquely identifies the grant
     */
    get key(): string;
}
/**
 * Represents the structure of a flattened permission
 * {read: ['SalesOrder','SalesOrderLine']}
 */
export type FlattenedPermission = Dict<string[]>;
/**
 * Base of the definition passed into the Activity and activity extension constructors
 */
export interface BaseActivityDefinition {
    permissions: string[];
    operationGrants?: Dict<OperationGrant[]>;
    permissionGrants?: Dict<PermissionGrant[]>;
}
/**
 * definition passed into the Activity constructor
 */
export interface ActivityDefinition extends BaseActivityDefinition {
    description: string;
    node: () => {
        new (): Node;
    };
    __filename: string;
}
export declare class Activity {
    #private;
    private readonly definition;
    constructor(definition: ActivityDefinition);
    /**
     * Description of activity
     */
    get description(): string;
    /**
     * Node related to the activity
     */
    get node(): StaticThis<Node>;
    /**
     * Activity name derived from the file name
     */
    get name(): string;
    /**
     * list of permission of the activity
     */
    get permissions(): string[];
    /**
     * list of operationGrants of the activity
     */
    get operationGrants(): Dict<OperationGrant[]> | undefined;
    /**
     * list of operationGrants of the activity
     */
    get permissionGrants(): Dict<PermissionGrant[]> | undefined;
    /**
     * The root package of the activity
     */
    get package(): string;
    /** Returns a flattened version of the permissions. For instance, in the following example:
     *  - lookup grants lookup on the current node, but also on Sales and Purchases.
     *  - read grants read on the current node, but also on Journal, and grants lookup on Journal and Sales.
     * {
     *       lookup: {
     *           lookup: [Sales, Purchases]
     *       },
     *       read: {
     *           read: [Journal],
     *           lookup: [Journal, Sales],
     *       }
     * },
     */
    get flattenedPermissions(): Dict<FlattenedPermission>;
    mergeExtension(activityExtension: ActivityExtension): void;
    get internalGrants(): Dict<Dict<InternalOperationGrant | InternalPermissionGrant>>;
    private addNodeOperation;
    private fillFactoryGrants;
    private addOperationGrants;
    /**
     * Adds grants from all children node on the vital graph
     * @param factory
     * @param grant
     * @param permission
     * @returns `true` if the grants were already added
     */
    private addExtendedVitalFactoriesGrants;
    getLocalizedTitleKey(): string;
    getPermissionLocalizedTitleKey(permission: string): string;
}
//# sourceMappingURL=activity.d.ts.map