import { Express } from 'express';
import { Application } from '../application';
import { StaticThis } from '../decorators';
import { Context } from '../runtime/context';
import { Node } from '../ts-api';
export interface DownloadTargetInfo {
    nodeName: string;
    _id: string;
}
export type FileStorageInfo = Pick<Context, 'application' | 'tenantId'>;
type DownloadHandler = (node: Node) => Promise<string>;
/**
 * Manages file storage operations and provides utility methods for handling file downloads.
 */
export declare abstract class FileStorageManager {
    private static _downloadHandlersByNodeName;
    /**
     * Adds authenticated routes to the Express app for handling file downloads.
     * @param app - The Express app.
     * @param application - The application instance.
     */
    static addAuthenticatedRoutes(app: Express, application: Application): void;
    /**
     * Adds unauthenticated routes to the Express app for handling the dev upload.
     * @param app - The Express app.
     * @param application - The application instance.
     */
    static addUnauthenticatedRoutes(app: Express, application: Application): void;
    /**
     * Registers a download handler for a specific node class.
     * @param clas - The node class.
     * @param handler - The download handler function.
     */
    static registerDownloadHandler(clas: StaticThis<Node>, handler: DownloadHandler): void;
    static getSandboxFilePath(storage: FileStorageInfo, objectKey: string): string;
    static clearSandbox(storage: FileStorageInfo): Promise<void>;
    /**
     * Generates an on-demand download URL for a given node.
     * @param node - The node to generate the download URL for.
     * @returns A Promise that resolves to the on-demand download URL.
     */
    static getOnDemandDownloadUrl(node: Node): Promise<string>;
    /**
     * Retrieves the download URL for a given target.
     * @param context - The context object.
     * @param target - The target string.
     * @returns A Promise that resolves to the download URL.
     * @throws {SecurityError|SystemError} If the target data is invalid.
     */
    static getTargetDownloadUrl(context: Context, target: string): Promise<string>;
}
export {};
//# sourceMappingURL=file-storage-manager.d.ts.map