import { Datetime } from '@sage/xtrem-date-time';
import { S3Bucket } from './s3-bucket';
export type S3ObjectInfo = {
    bucketName: string;
    folder: string;
    key: string;
    lastModified?: Datetime;
    eTag?: string;
    size?: number;
    storageClass?: string;
};
export declare abstract class S3Helper {
    /**
     * Ensures that the provided S3Bucket is compatible with S3 infos
     * @param s3Bucket
     * @param s3Infos
     */
    private static _checkS3Bucket;
    /**
     * Upload a local file to a S3 bucket and returns the S3 URI of the written object
     * @param sourceFilename the local filename of the file to copy
     * @param targetS3Info the bucket/folder/key of the object to write
     * @param s3Bucket the (optional) S3 bucket to use (to optimize chained calls)
     */
    static upload(sourceFilename: string, targetS3Info: S3ObjectInfo, s3Bucket?: S3Bucket): Promise<string>;
    /**
     * Download a file from a S3 bucket
     * @param sourceS3Info the bucket/folder/key of the object to copy
     * @param targetFilename the local file to write
     * @param s3Bucket the (optional) S3 bucket to use (to optimize chained calls)
     */
    static download(sourceS3Info: S3ObjectInfo, targetFilename: string, s3Bucket?: S3Bucket): Promise<void>;
    /**
     * Parse a S3 URI with the following format : s3://bucketName/xxx/yyy/key
     * @param s3Uri
     */
    static parseS3Uri(s3Uri: string): S3ObjectInfo;
    /**
     * Builds a valid S3 URI
     * @param s3
     */
    static buildS3Uri(s3: S3ObjectInfo): string;
    /**
     * Returns whether an object exists in a bucket
     * @param s3 the information about the object
     * @param s3Bucket the (optional) S3 bucket to use (to optimize chained calls)
     */
    static objectExists(s3: S3ObjectInfo): Promise<boolean>;
    /**
     * List all objects present in a bucket: this function supports AWS S3 pagination mechanism
     * @param s3 the information about the objects
     * @param s3Bucket the S3 bucket to use
     * @param params the parameters to use
     */
    private static _listObjectsV2;
    /**
     * List objects present in a bucket
     * @param s3 the information about the objects
     */
    static listObjects(s3: S3ObjectInfo): Promise<S3ObjectInfo[]>;
}
//# sourceMappingURL=s3-helper.d.ts.map