/// <reference types="node" />
/// <reference types="node" />
/** @packageDocumentation @module runtime */
import { AnyValue, AsyncResponse } from '@sage/xtrem-async-helper';
import { Dict, integer } from '@sage/xtrem-shared';
import { GraphQLSchema } from 'graphql';
import * as http from 'http';
import { StaticThis } from '../decorators';
import { NotificationTopic } from '../interop';
import { Context, ContextOptions, CreateAdminUserOptions, UserData } from '../runtime/context';
import { NodeFactory } from '../runtime/node-factory';
import { Node } from '../ts-api';
import { Activity } from '../ts-api/activity';
import { DataType, DataTypeOptions } from '../types';
import { WorkflowManagerInterface } from '../workflow/workflow-manager-interface';
import { ActivityManager } from './activity-manager';
import { ClientSettingsManager } from './client-settings-manager';
import { CsvChecksumManager } from './csv-checksum-manager';
import { HotUpgradeManager } from './hot-upgrade-manager';
import { NodeFactoriesManager } from './node-factories-manager';
import { NotificationManager } from './notification-manager';
import { Package, PackageCreateOptions, PackageJsonAbout, PackageJsonFile } from './package';
import { PackageManager } from './package-manager';
import { ServiceOption } from './service-option';
import { ServiceOptionManager } from './service-option-manager';
import './unhandled-error-monitor';
import EventEmitter = require('events');
export interface Dependency {
    name: string;
    version: string;
}
/**
 * Type of application:
 *                          +---------+-------+------+----------+-----+
 *                          | service | batch | test | dev-tool | job |
 * +------------------------+---------+-------+------+----------+-----+
 * | graphQL API            |    X    |   -   |   X  |    -     |  -  |
 * +------------------------+---------+-------+------+----------+-----+
 * | system broadcasts      |    X    |   X   |   X  |    -     |  X  |
 * +------------------------+---------+-------+------+----------+-----+
 * | applicative queues     |  in dev |   X   |   X  |    -     |  -  |
 * +------------------------+---------+-------+------+----------+-----+
 */
export type ApplicationType = 
/**
 * example: pnpm run start
 */
'service'
/**
 * example: xtrem test, ....
 */
 | 'test'
/**
 * example: xtrem compile, ....
 */
 | 'dev-tool'
/**
 * example: xtrem manage, schema, tenant, ....
 */
 | 'admin-tool'
/**
 * example: upgrade operations, ...
 */
 | 'job';
export interface ApplicationCreateOptions extends PackageCreateOptions {
    applicationType?: ApplicationType;
    schemaName?: string;
    testApplicationDir?: string;
    startOptions?: ApplicationStartServicesOptions;
}
export interface ApplicationConstructorOptions extends ApplicationCreateOptions {
    schemaName: string;
}
export declare const allowedStartChannels: readonly ["graphql", "listeners", "routing"];
export type StartChannel = (typeof allowedStartChannels)[number];
export interface ApplicationStartServicesOptions {
    services: string[];
    channels: StartChannel[];
    webSockets?: string[];
    /**
     * Names of queues to explicitly start
     * If not set, all the queues will be started
     */
    queues?: string[];
}
type WithContextFunction = <T extends AnyValue | void>(tenantId: string | null, body: (context: Context) => AsyncResponse<T>, options?: ContextOptions) => Promise<T>;
export interface WithContext {
    withCommittedContext: WithContextFunction;
    withUncommittedContext: WithContextFunction;
    withReadonlyContext: WithContextFunction;
}
export declare class Application {
    #private;
    readonly options: ApplicationConstructorOptions;
    readonly mainPackage: Package;
    private _rootAbout;
    private _about;
    private readonly packageLoader;
    readonly factoriesManager: NodeFactoriesManager;
    readonly activityManager: ActivityManager;
    readonly serviceOptionManager: ServiceOptionManager;
    readonly packageManager: PackageManager;
    readonly notificationManager: NotificationManager;
    readonly clientSettingsManager: ClientSettingsManager;
    readonly workflowManager: WorkflowManagerInterface;
    /** TODO: make private again */
    _sqlPackageFactories: NodeFactory[];
    private _pluginDependencies;
    static readonly emitter: EventEmitter<[never]>;
    readonly nodeAndEnumNames: Dict<'node' | 'enum'>;
    readonly dataTypes: Dict<DataType<AnyValue, DataTypeOptions>>;
    readonly notificationTopics: Dict<NotificationTopic>;
    private graphqlSchemaFunnel;
    private graphqlSchemas;
    readonly requestFunnel: import("@sage/xtrem-async-helper").Funnel;
    private static hasInstance;
    /** The external http server, set when the application is started with 'graphql' inside `startOptions.channels` */
    graphqlHttpServer?: http.Server;
    /**
     * Returns the test nodes API if the application is run from a test/fixtures directory
     * that contains a test-application.ts file.
     * Returns an empty object otherwise.
     */
    private tryLoadTestApi;
    registerDataType(pack: Package, name: string, dataType: DataType<AnyValue, DataTypeOptions>): void;
    private registerDefaultReferenceDataType;
    registerNotificationTopic(pack: Package, name: string, topic: NotificationTopic): void;
    findNotificationTopic(name: string): NotificationTopic;
    registerMiscArtifacts(): void;
    /**
     * Returns the manager in charge of hot-upgrades
     */
    get hotUpgradeManager(): HotUpgradeManager;
    get isReady(): boolean;
    setReady(): void;
    /**
     * The manager in charge of the checksums of loaded CSV files
     */
    get csvChecksumManager(): CsvChecksumManager;
    get applicationType(): ApplicationType | undefined;
    /**
     * Get the add-on package paths
     */
    get addOnPackagePaths(): string[];
    get schemaName(): string;
    get activities(): Dict<Activity>;
    get serviceOptionsByName(): Dict<ServiceOption>;
    findServiceOption(name: string): ServiceOption;
    /**
     * List of packages started in the current container.
     */
    private _servicePackagesStarted?;
    private _startOptions;
    get startOptions(): ApplicationStartServicesOptions;
    get servicePackagesStarted(): Package[];
    /**
     * emit events to start listeners
     * @param otherTopics  topics to emit other than the standard `listen` and `ready`, used for unit tests.
     */
    startListeners(otherTopics?: string[]): void;
    /**
     * Limits the length of an SQS queue name to 75 characters, 80 with the .fifo extension.
     * If the name is already 75 characters or less, it is returned as is.
     * If the name is longer than 75 characters, it is truncated and appended with "---" in the middle.
     * @param name - The original queue name.
     * @returns The modified queue name.
     */
    static limitSqsQueueName(name: string): string;
    /**
     * Generates the raw SQS queue name based on the provided name.
     *
     * For unit tests, the queue name will be the package name with the routing queue name appended.
     * In multi-app mode, the queue name is prefixed with the app name if it is not already prefixed.
     *
     * @param name - The name used to generate the queue name.
     * @returns The generated raw SQS queue name.
     */
    private rawSqsQueueName;
    /**
     * Returns the SQS queue name for the given name.
     * @param name - The name of the queue.
     * @returns The SQS queue name.
     */
    sqsQueueName(name: string): string;
    getPluginDependencies(): Dict<string>;
    /** @disabled_internal */
    /**
     * Returns the factories that have their own table in the database.
     * The returned factories are sorted so that referenced factories
     * are always before the factories that reference them with non nullable references.
     */
    getSqlPackageFactories(): NodeFactory[];
    get name(): string;
    /**
     * Returns the short name of an application without the sage prefix
     * for instance, will return 'xtrem-services-main' for the @sage/xtrem-services-main application
     */
    get shortName(): string;
    get packageName(): string;
    get dir(): string;
    get tmpDir(): string;
    get version(): string;
    get about(): PackageJsonAbout;
    get rootAbout(): PackageJsonAbout;
    checkIfPluginPackage(depPackageContent: PackageJsonFile | null): void;
    /**
     * Validate all Activities loaded from packages
     */
    validateActivities(): void;
    getPackages(): Package[];
    get packagesByName(): Dict<Package>;
    findPackage(name: string): Package;
    /**
     * Gets all platform packages required by the application packages
     */
    getPlatformPackages(): Dependency[];
    getAllFactories(): NodeFactory[];
    /** @disabled_internal */
    getFactoryByName(name: string): NodeFactory;
    /** @disabled_internal */
    tryToGetFactoryByName(name: string): NodeFactory | undefined;
    /** @disabled_internal */
    getFactoryByConstructor(nodeConstructor: StaticThis<Node>): NodeFactory;
    getFactoryByTableName(name: string): NodeFactory;
    tryGetFactoryByTableName(name: string): NodeFactory | undefined;
    findFactoriesUsingDatatype(dataType: DataType<AnyValue, unknown>): NodeFactory[];
    /**
     * Checks if a record is used in another node references
     * @param id _id of the record to be checked in other references
     * @param context current context
     * @param className className of the node of _id
     * @returns Promise
     */
    isRecordUsed(id: number, context: Context, className: string): Promise<boolean>;
    getAllNodes(): (typeof Node)[];
    isNodePublished(node: {
        new (): Node;
    }): boolean;
    /** @disabled_internal */
    getAllSortedFactories(): NodeFactory[];
    /**
     * Create the database schema, used in the cli as well platform tests
     */
    static dropDbSchema(schemaName: string): Promise<void>;
    /**
     * Rename a schema
     */
    static renameSchema(oldName: string, newName: string): Promise<void>;
    /**
     * Create the database schema, used in the cli as well platform tests
     */
    static createDbSchema(schemaName: string): Promise<void>;
    createAdminUser(tenantId: string, data: UserData, options?: CreateAdminUserOptions): Promise<void>;
    getGraphQLSchema(context?: Context): Promise<GraphQLSchema>;
    getPackageOfNodeConstructor(node: {
        new (): Node;
    }): Package;
    /**
     * Create a context bound to a specific tenant. This context will be committed

    * @param tenantId when set to null, the context can only be used to access to sharedTables or run custom SQL commands
     */
    withCommittedContext<T extends AnyValue | void>(tenantId: string | null, body: (context: Context) => AsyncResponse<T>, options?: ContextOptions): Promise<T>;
    /**
     * Create a context bound to a specific tenant. This context will not be committed
     *
     * @param tenantId when set to null, the context can only be used to access to sharedTables or run custom SQL commands
     */
    withUncommittedContext<T extends AnyValue | void>(tenantId: string | null, body: (context: Context) => AsyncResponse<T>, options?: ContextOptions): Promise<T>;
    /**
     * Create a context bound to a specific tenant. This context will be in read-only mode
     * @param tenantId when set to null, the context can only be used to access to sharedTables or run custom SQL commands
     */
    withReadonlyContext<T extends AnyValue | void>(tenantId: string | null, body: (context: Context) => AsyncResponse<T>, options?: ContextOptions): Promise<T>;
    /**
     * Gets a root frame for calling with*Context functions
     */
    get asRoot(): WithContext;
    /**
     * Create a context to perform DDL operations
     */
    createContextForDdl<T extends AnyValue | void>(body: (context: Context) => AsyncResponse<T>, options?: ContextOptions): Promise<T>;
    private initFactoryCacheLogs;
    clearGlobalCache(): void;
    invalidateGlobalCache(context: Context): Promise<void>;
    /** Used by xtrem-cop to verify that getValue / computeValue are correctly set` */
    verifySqlConversions(): Promise<integer>;
}
export declare function main(body: () => AsyncResponse<void>): void;
export {};
//# sourceMappingURL=application.d.ts.map