"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.reexportAll = exports.runBytecodeFile = exports.runBytecode = exports.compileFile = exports.compileAllFiles = exports.compileCode = exports.getSystemInfo = exports.sourceMapSetup = exports.shouldMinify = void 0;
const fs = require("fs");
const v8 = require("node:v8");
const path = require("path");
const semver = require("semver");
const source_map_support_1 = require("source-map-support");
const minifier = require("terser");
const vm = require('vm');
const zlib = require('zlib');
const Module = require('module');
const os = require('os');
const rootDir = path.resolve(__dirname, '../../../..');
v8.setFlagsFromString('--no-lazy');
if (Number.parseInt(process.versions.node.split('.')[0], 10) >= 12) {
    v8.setFlagsFromString('--no-flush-bytecode'); // Thanks to A-Parser (@a-parser)
}
const encoding = 'utf8';
const replacedSrcExtension = '.jsb';
const compressedReplacedSrcExt = '.jsbz';
const magicBytes = Buffer.from('#JSB#\n', encoding);
const separatorBytes = Buffer.from('#76ef331c-0587-4aad-adbb-d98eded8605c#\n', encoding);
function shouldMinify(packageJson) {
    // with node 22 we no longer get a compatible bytecode between linux and windows, so we always minify only
    return +process.versions.modules >= 127 || packageJson.type === 'module';
}
exports.shouldMinify = shouldMinify;
const sourceMapSupportOptions = {
    retrieveSourceMap(source) {
        // source map exists in the supplied source path
        if (fs.existsSync(`${source}.map`)) {
            return {
                url: source,
                map: fs.readFileSync(`${source}.map`, 'utf8'),
            };
        }
        // get file URL from @sage position
        //  e.g. source = /home/vsts/work/1/s/xtrem-platform/@sage/xtrem-cli/lib/cli.js
        //  fileUrl = @sage/xtrem-cli/lib/cli.js
        const fileUrl = source.substring(source.indexOf('@sage'));
        // filePath will be sth like /app/xtrem/xtrem-services/@sage/xtrem-sales/node_modules/@sage/xtrem-cli/lib/cli.js
        const filePath = path.join(process.cwd(), 'node_modules', fileUrl);
        // check if file path exists the cwd node_modules
        if (fs.existsSync(`${filePath}.map`)) {
            return {
                url: filePath,
                map: fs.readFileSync(`${filePath}.map`, 'utf8'),
            };
        }
        // rootFilePath will be sth like /app/xtrem/xtrem-services/node_modules/@sage/xtrem-cli/lib/cli.js
        const rootFilePath = path.join(process.cwd(), '..', '..', 'node_modules', fileUrl);
        // check if file path exists the root node_modules
        if (fs.existsSync(`${rootFilePath}.map`)) {
            return {
                url: rootFilePath,
                map: fs.readFileSync(`${rootFilePath}.map`, 'utf8'),
            };
        }
        // source map is not found anywhere
        return null;
    },
};
/**
 * install source-map-support with value for retrieveSourceMap function that will be used to retrieve the source map file content
 * @param options
 */
function sourceMapSetup(options = {}) {
    (0, source_map_support_1.install)({ ...sourceMapSupportOptions, ...options });
}
exports.sourceMapSetup = sourceMapSetup;
function getSystemInfo() {
    return { nodeVersion: process.version, arch: os.arch() };
}
exports.getSystemInfo = getSystemInfo;
// TODO: this is a temporary hack to avoid node.js crash when printing stack trace in async/await mode.
const keepSource = false;
function getReplacementSource(originalSource, preservedRangeList) {
    if (keepSource)
        return originalSource;
    let newSource = '';
    for (let index = 0; index < originalSource.length; index += 1) {
        const char = originalSource.charAt(index);
        if (char === '\n') {
            newSource = `${newSource}${char}`;
        }
        else if (preservedRangeList.find((r) => r[0] === index)) {
            const range = preservedRangeList.find((r) => r[0] === index);
            const preservedString = originalSource.substring(range[0], range[1] + 1);
            newSource = `${newSource}${preservedString}`;
            index = range[1];
        }
        else {
            newSource = `${newSource}${' '}`;
        }
    }
    return newSource;
}
/**
 * Generates v8 bytecode buffer.
 * @param   {string} javascriptCode JavaScript source that will be compiled to bytecode.
 * @returns {Buffer} The generated bytecode.
 */
function compileCode(javascriptCode, options) {
    const { filename } = options || {};
    const script = new vm.Script(javascriptCode, {
        filename,
        produceCachedData: true,
    });
    const systemInfo = getSystemInfo();
    if (semver.major(systemInfo.nodeVersion) < 14) {
        throw new Error(`Node version must be at least 14 to use xtrem-bytenode. Current version ${systemInfo.nodeVersion}`);
    }
    let bytecodeBuffer = script.createCachedData && typeof script.createCachedData === 'function'
        ? script.createCachedData()
        : script.cachedData;
    if (options && options.replacementSource) {
        bytecodeBuffer = Buffer.concat([
            magicBytes,
            Buffer.from(JSON.stringify(systemInfo), encoding),
            separatorBytes,
            options.replacementSource,
            separatorBytes,
            bytecodeBuffer, // binary code
        ]);
        if (options.compress) {
            // bytecodeBuffer = Buffer.from(zlib.deflateRawSync(bytecodeBuffer));
            bytecodeBuffer = zlib.deflateSync(bytecodeBuffer, (err) => {
                if (err) {
                    // eslint-disable-next-line no-console
                    console.error('An error occurred:', err);
                    process.exitCode = 1;
                }
            });
        }
    }
    return bytecodeBuffer;
}
exports.compileCode = compileCode;
async function minifyCode(javascriptCode, options) {
    const { filename } = options || {};
    // console.log(
    //     `Uglifying ${filename?.substring(rootDir.length + 1)} (${Buffer.byteLength(javascriptCode)} bytes)...`,
    // );
    let mapContent = '';
    const mapFile = `${filename}.map`;
    if (filename && fs.existsSync(mapFile)) {
        mapContent = fs.readFileSync(mapFile, 'utf-8');
    }
    const requiredNames = Array.from(javascriptCode.matchAll(/const ([a-zA-Z0-9_]+) = require\(/g)).map((match) => match[1]);
    const xtremRequires = requiredNames.filter((name) => name.startsWith('xtrem'));
    try {
        const result = await minifier.minify(javascriptCode, {
            // keep_fnames is required to prevent from having anonymous class
            keep_fnames: true,
            mangle: {
                // used in ts to sql functions, we may inspect the input code to allow mangling in other files
                // Context.getConfigurationValue is transformed to xtrem_core_1.Context.getConfigurationValue by tsc
                reserved: ['typesLib', ...xtremRequires],
            },
            compress: options?.compress
                ? {
                    // TO INVESTIGATE: boolean options create issues at runtime
                    // prevent replacing true and false by !0 and !1
                    booleans: false,
                    booleans_as_integers: false,
                    // prevent rewriting ternary operators
                    // ex: const a = b ? c.p : d.p; => const a = (b ? c : d).p;
                    conditionals: false,
                    // adjust compress options to prevent from breaking the code parsed by ts to sql
                    // do not reduce vars which can lead to const replaced by var
                    reduce_vars: false,
                    // The name of this option is confusing, we need to set it to false to prevent modifying 'if ... return x; return y;' to 'if ... return x else return y;'
                    if_return: false,
                }
                : false,
            output: {
                preamble: `/* Copyright (c) 2020-${new Date().getFullYear()} Sage. All Rights Reserved. */`,
            },
            sourceMap: {
                content: mapContent,
                url: path.basename(mapFile),
            },
            ecma: 2020,
        });
        return result;
    }
    catch (error) {
        throw new Error(`Error minifying ${filename?.substring(rootDir.length + 1)}: ${error instanceof Error ? error.message : error}`);
    }
}
async function compileAllFiles(packageDir, files, options) {
    // eslint-disable-next-line import/no-dynamic-require, global-require
    const packageFile = require(path.resolve(packageDir, 'package.json'));
    const minify = shouldMinify(packageFile);
    // if the package is not a module, we need to reexport all the exports from the main index file to make it compatible with esm loader
    if (!minify && files.length > 1) {
        const mainIndex = path.join(packageDir, 'build', 'index.js');
        if (fs.existsSync(mainIndex)) {
            const pos = files.indexOf(mainIndex);
            if (pos > -1) {
                // Remove main index file from the list of files to compile
                files.splice(pos, 1);
            }
            reexportAll(mainIndex);
        }
    }
    await Promise.all(files.map(async (filename) => {
        await compileFile({ ...options, minify, filename });
    }));
}
exports.compileAllFiles = compileAllFiles;
/**
 * Compiles JavaScript file to .jsb/.jsbz file.
 * @param   options
 * @returns The compiled filename
 */
async function compileFile(options) {
    const binaryExtension = (compressed = false) => (compressed ? compressedReplacedSrcExt : replacedSrcExtension);
    const { compress, verbose, deleteSource, filename, minify } = options;
    const compileAsModule = options.compileAsModule === undefined ? true : options.compileAsModule;
    const extension = minify ? '.js' : binaryExtension(compress);
    const getPreservedRanges = options.getPreservedRanges ?? (() => []);
    const compiledFilename = options.output || `${filename.slice(0, -3)}${extension}`;
    let scriptCode = fs.readFileSync(filename, encoding).replace(/^#!.*/, '');
    if (compileAsModule && !minify) {
        scriptCode = Module.wrap(scriptCode);
    }
    const codeOptions = {
        ...options,
        filename,
        compileAsModule,
        minify,
        replacementSource: minify
            ? undefined
            : Buffer.from(getReplacementSource(scriptCode, getPreservedRanges(scriptCode)), encoding),
    };
    const { code, map } = minify
        ? await minifyCode(scriptCode, codeOptions)
        : //   { code: scriptCode, map: undefined }
            { code: compileCode(scriptCode, codeOptions), map: undefined };
    if (code === undefined) {
        throw new Error(`Failed to compile ${filename}. No code returned from minifier.`);
    }
    if (verbose) {
        console.log(`Compiled ${filename?.substring(rootDir.length + 1)} (${Buffer.byteLength(scriptCode)} bytes) to ${compiledFilename?.substring(rootDir.length + 1)} (${Buffer.byteLength(code)} bytes) `);
    }
    fs.writeFileSync(compiledFilename, code, { encoding });
    if (map) {
        fs.writeFileSync(`${compiledFilename}.map`, typeof map === 'string' ? map : JSON.stringify(map), { encoding });
    }
    if (!minify && deleteSource) {
        fs.unlinkSync(filename);
    }
    return compiledFilename;
}
exports.compileFile = compileFile;
function invalidFileError(filename) {
    return new Error(`Invalid binary file content: ${filename}`);
}
// TODO: rewrite this function
function fixBytecode(bytecodeBuffer) {
    const dummyBytecode = compileCode('"ಠ_ಠ"');
    if (process.version.startsWith('v8.8') || process.version.startsWith('v8.9')) {
        // Node is v8.8.x or v8.9.x
        dummyBytecode.slice(16, 20).copy(bytecodeBuffer, 16);
        dummyBytecode.slice(20, 24).copy(bytecodeBuffer, 20);
    }
    else if (process.version.startsWith('v12') ||
        process.version.startsWith('v13') ||
        process.version.startsWith('v14') ||
        process.version.startsWith('v15') ||
        process.version.startsWith('v16') ||
        process.version.startsWith('v17') ||
        process.version.startsWith('v18') ||
        process.version.startsWith('v19') ||
        process.version.startsWith('v20')) {
        dummyBytecode.slice(12, 16).copy(bytecodeBuffer, 12);
    }
    else {
        dummyBytecode.slice(12, 16).copy(bytecodeBuffer, 12);
        dummyBytecode.slice(16, 20).copy(bytecodeBuffer, 16);
    }
}
function splitContent(fileContent, filename) {
    const magicPosition = fileContent.indexOf(magicBytes);
    const sepPosition = fileContent.indexOf(separatorBytes);
    if (magicPosition < 0 || sepPosition < 0 || magicPosition > sepPosition) {
        throw invalidFileError(filename);
    }
    const systemInfo = fileContent.subarray(magicPosition + magicBytes.length - 1, sepPosition);
    const nextSepPosition = fileContent.indexOf(separatorBytes, sepPosition + separatorBytes.length + 1);
    if (nextSepPosition < 0) {
        throw invalidFileError(filename);
    }
    const dummyCode = fileContent.subarray(sepPosition + separatorBytes.length, nextSepPosition);
    const bytecodeBuffer = fileContent.subarray(nextSepPosition + separatorBytes.length);
    fixBytecode(bytecodeBuffer);
    return { dummyCode, bytecodeBuffer, systemInfo };
}
/**
 * Runs v8 bytecode buffer and returns the result.
 * @param   {Buffer} bytecodeBuffer The buffer object that was created using compileCode function.
 * @returns {any}    The result of the very last statement executed in the script.
 */
function runBytecode(bytecodeBuffer, filename) {
    const content = splitContent(bytecodeBuffer, filename);
    const script = new vm.Script(content.dummyCode.toString(), {
        cachedData: content.bytecodeBuffer,
    });
    if (script.cachedDataRejected) {
        throw new Error(`Invalid or incompatible cached data (cachedDataRejected): ${filename}`);
    }
    return script.runInThisContext();
}
exports.runBytecode = runBytecode;
/**
 * Runs .jsb/.jsbz file and returns the result.
 * @param   {string} filename
 * @returns {any}    The result of the very last statement executed in the script.
 */
function runBytecodeFile(filename) {
    if (typeof filename !== 'string') {
        throw new Error(`filename must be a string. ${typeof filename} was given.`);
    }
    const bytecodeBuffer = Buffer.from(fs.readFileSync(filename, encoding), encoding);
    return runBytecode(bytecodeBuffer, filename);
}
exports.runBytecodeFile = runBytecodeFile;
function applyCode(module, filename, script) {
    /*
        This part is based on:
        https://github.com/zertosh/v8-compile-cache/blob/7182bd0e30ab6f6421365cee0a0c4a8679e9eb7c/v8-compile-cache.js#L158-L178
    */
    function require(id) {
        return module.require(id);
    }
    require.resolve = (request, options) => {
        return Module._resolveFilename(request, module, false, options);
    };
    require.main = process.mainModule;
    require.extensions = Module._extensions;
    require.cache = Module._cache;
    const compiledWrapper = script.runInThisContext({
        filename,
        lineOffset: 0,
        columnOffset: 0,
        displayErrors: true,
    });
    const dirname = path.dirname(filename);
    const args = [module.exports, require, module, filename, dirname, process, global];
    return compiledWrapper.apply(module.exports, args);
}
function verifySource(systemInfo, filename) {
    const fileSystemInfo = JSON.parse(Buffer.from(systemInfo).toString(encoding));
    if (!fileSystemInfo.arch || !fileSystemInfo.nodeVersion) {
        throw invalidFileError(filename);
    }
    const currentSystemInfo = getSystemInfo();
    if (currentSystemInfo.arch !== fileSystemInfo.arch) {
        throw new Error(`Architecture ${currentSystemInfo.arch} invalid for file ${filename} which was created in ${fileSystemInfo.arch} architecture.`);
    }
    const buildVersion = semver.parse(fileSystemInfo.nodeVersion);
    if (buildVersion == null) {
        throw new Error(`Invalid build version '${fileSystemInfo.nodeVersion}'.`);
    }
    const expectedMajorMinorVersion = `${buildVersion.major}.${buildVersion.minor}`;
    if (!semver.satisfies(currentSystemInfo.nodeVersion, expectedMajorMinorVersion)) {
        throw new Error(`Node version must satisfy major.minor version '${expectedMajorMinorVersion}'. Current version is '${currentSystemInfo.nodeVersion}'.`);
    }
}
function getReplacedSourceScript(filename, compressed = false) {
    let fileContent = fs.readFileSync(filename);
    if (compressed) {
        // fileContent = Buffer.from(zlib.inflateRawSync(fileContent));
        fileContent = zlib.inflateSync(fileContent, (err) => {
            if (err) {
                // eslint-disable-next-line no-console
                console.error('An error occurred:', err);
                process.exitCode = 1;
            }
        });
    }
    const content = splitContent(fileContent, filename);
    verifySource(content.systemInfo, filename);
    const script = new vm.Script(content.dummyCode.toString(), {
        filename: `${filename.substring(0, filename.lastIndexOf('.'))}.js`,
        lineOffset: 0,
        columnOffset: 0,
        displayErrors: true,
        cachedData: content.bytecodeBuffer,
    });
    if (script.cachedDataRejected) {
        throw new Error(`Invalid or incompatible cached data (cachedDataRejected) in ${filename}`);
    }
    return script;
}
function reexportAll(mainIndex) {
    const packDir = path.dirname(path.dirname(mainIndex));
    // eslint-disable-next-line global-require, import/extensions
    const { parse } = require('./lexer');
    const { code, modified } = traverseExports({ parse, packDir }, mainIndex);
    if (!modified)
        return;
    fs.writeFileSync(mainIndex, code);
}
exports.reexportAll = reexportAll;
/**
 * Travserse the exports of a module using a slightly modified version of the cjs lexer for the node.js esm loader
 * it will generate a new code with those exports defined at the toplevel without the __exportStar calls so that esm can detect all the exports.
 * This faster than requiring the module and get the exports
 * @param config
 * @param requiredModPath
 * @param reexports
 * @param requirePath
 * @returns
 * @see https://github.com/nodejs/node/blob/main/doc/api/esm.md#commonjs-namespaces
 * @see https://github.com/nodejs/cjs-module-lexer/blob/main/README.md#grammar
 */
function traverseExports(config, requiredModPath, reexports = [], requirePath = []) {
    const ext = path.extname(requiredModPath);
    const requiredFile = `${requiredModPath}${ext ? '' : '.js'}`;
    const indexFile = fs.existsSync(requiredFile) ? requiredFile : path.join(requiredModPath, 'index.js');
    const code = fs.readFileSync(indexFile, encoding);
    const dir = path.dirname(indexFile);
    const { parse, packDir } = config;
    let result;
    try {
        result = parse(code);
    }
    catch (e) {
        console.error(`Failed to parse ${indexFile}`);
        throw e;
    }
    const exports = result.exports.filter((exp) => exp !== '__esModule');
    const isMain = requirePath.length === 0;
    const exportsArray = [];
    if (exports.length > 0 && !isMain) {
        reexports.push(...exports);
    }
    if (result.reexports.length > 0) {
        result.reexports.forEach((exp) => {
            const file = exp.startsWith('.')
                ? path.join(dir, exp)
                : path.join(packDir, 'node_modules', exp, 'build', 'index.js');
            const names = isMain ? [] : reexports;
            traverseExports(config, file, names, [...requirePath, requiredModPath]);
            if (isMain)
                exportsArray.push(names);
        });
    }
    if (isMain && exportsArray.some((m) => m.length > 0)) {
        let begin = code.indexOf('Object.defineProperty(exports, "__esModule"', 0);
        if (begin < 0) {
            begin = code.indexOf("Object.defineProperty(exports, '__esModule'", 0);
        }
        const defines = Object.create(null);
        exportsArray.forEach((names, i) => {
            return names.forEach((name) => {
                // The same property may be exported from several require, we keep the last one which is most likely not undefined
                defines[name] =
                    `Object.defineProperty(exports, '${name}', { enumerable: true, get: function() { return _e${i + 1}.${name}; } });`;
            });
        });
        const reexportDefines = Object.values(defines).flat().join('\n');
        let j = 0;
        // remove the __exportStar calls and replace them with require calls and defineProperty to match the esm loader behavior
        const newCode = code
            .slice(begin)
            .replace(/__exportStar\(require\((["'][\w@/.-]+["'])\),\s*exports\);/g, (_m, p1) => {
            j += 1;
            return `const _e${j} = require(${p1});`;
        })
            .replace(/\/\/#\s*sourceMappingURL=[a-z0-9/-]+.js.map$/, '');
        return {
            code: `// Modified automatically to fix exports detection when loaded from an esm module\n${newCode}${reexportDefines}`,
            modified: true,
        };
    }
    return { code, modified: false };
}
Module._extensions[replacedSrcExtension] = (module, filename) => {
    const script = getReplacedSourceScript(filename);
    return applyCode(module, filename, script);
};
Module._extensions[compressedReplacedSrcExt] = (module, filename) => {
    const script = getReplacedSourceScript(filename, true);
    return applyCode(module, filename, script);
};
//# sourceMappingURL=bytenode.js.map