/** Interfaces and types for management code GS 1 */

import { DateValue } from '@sage/xtrem-date-time';
import { integer } from '@sage/xtrem-shared';
import { fields } from '@sage/xtrem-ui';
import { DataTitle } from '../shared-functions/index';

export const compositeCodePrefix = /^(\d{2,4}|\][A-Za-z]\d{3,5}|\x1d\d{2,4})/;

// Incoming page fields definition

export type BarcodeManagerSupportedFields =
    | fields.Reference
    | fields.Date
    | fields.Text
    | fields.Numeric
    | fields.FilterSelect;

export type ElementTypeSupported = string | DateValue | number | undefined;

export type AsyncVoidFunction = () => Promise<void>;

export type VoidFunction = () => void;

export type AsyncAndVoidFunction = AsyncVoidFunction | VoidFunction;

export interface FieldSupported {
    mainField: BarcodeManagerSupportedFields;
    onChangeMainField?: AsyncVoidFunction;
    unitField?: BarcodeManagerSupportedFields;
    onChangeUnitField?: AsyncVoidFunction;
}

export type DictionaryFieldSupported = Record<DataTitle, FieldSupported>;

// Internal page fields definition

/** @internal */
export interface FieldSegment {
    currentField: BarcodeManagerSupportedFields;
    onChange?: AsyncVoidFunction;
}

/**
 * This attribute is used for very order of segment into page...
 * when the value has not changed, this segment has unused
 */
/** @internal */
export interface SegmentForDictionary extends FieldSegment {
    sequence: integer;
}

/** @internal */
export type DictionarySegment = Record<string, SegmentForDictionary>;

// This expose only data during mapping operations
/** @internal */
export interface DataSegmentMapped extends FieldSegment {
    data: any;
}

// This expose only data during mapping operations

/** @internal */
export interface FieldDataType {
    DataTitle: DataTitle;
}

/**
 * This attribute is used for very order of field into page...
 * when the value has not changed, this field has unused
 * This interface store only minimum data..
 */
/** @internal */
export interface DataFieldMapped extends FieldDataType, FieldSupported {
    sequence: number;
    data?: ElementTypeSupported;
    unit?: ElementTypeSupported;
}

// Supported page fields definition without data

/**
 * This attribute is used for very order of field into page...
 * when the value has not changed, this field has unused
 */
/** @internal */
export interface ScreenFieldSupported extends FieldDataType {
    sequence: number;
    segments: FieldSegment[];
}

// Data composite

/**
 * This composite item is used in storage and service
 */
export type DataComposite = {
    DataTitle: DataTitle;
    data: ElementTypeSupported;
    unit?: ElementTypeSupported;
};

/**
 * All composite data have stored in dictionary.
 */
export type DictionaryDataComposite = Record<DataTitle, DataComposite>;

/**
 * This asynchronous function signature is used only for specifying
 * a call back to check composite data before continuous
 * @param dictionaryDataComposite : current dictionary  (may be empty)
 * @return true when dispatching is allowing to continue action
 *         false when dispatching must clear all composites and exiting
 */
export type AsyncCompositeAllowed = (dictionaryDataComposite: DictionaryDataComposite) => Promise<boolean>;
