import { DateValue } from '@sage/xtrem-date-time';
export interface GenericPrintReportParameters {
    parametersNames: string[];
    parametersValues: string[];
}

/** @internal */
export class GenericPrintPayloadMapping {
    /**
     * Parse xtrem variable such as boolean and date to X3 format
     * @param valueFromPayload
     * @return value or undefined when type object is not authorized.
     */
    private static xtremToLegacyDataFormatting(value: any): string | undefined {
        let valueFromPayload = value;

        /**
         * check if the variable is an instance of date
         */

        if (valueFromPayload instanceof DateValue) {
            valueFromPayload = valueFromPayload.value;
        } else if (typeof valueFromPayload === 'object' || typeof valueFromPayload === 'function') {
            /**
             * check if the variable is an unsupported object instance or function
             */

            return undefined;
        }

        /**
         * check if is in date format YYYY-MM-DD
         */
        const found: any = valueFromPayload?.toString().match(/([12]\d{3}-(0[1-9]|1[0-2])-(0[1-9]|[12]\d|3[01]))/g);

        if (found !== null) {
            return found?.toString().replace(/-/g, '');
        }

        /**
         * if boolean translate
         *  true -> 1 and false -> 0 to meet x3 variables
         */
        if (Boolean(valueFromPayload) === valueFromPayload) {
            return valueFromPayload ? '2' : '1';
        }

        return String(valueFromPayload);
    }

    /**
     *
     * @param content Format content entry
     * @returns JSON formatted objet
     */
    private static stringifyContent(content: any) {
        return JSON.stringify(
            content,
            (_key, value) => {
                return value ?? '';
            },
            4,
        );
    }

    /**
     * Convert a object to a string contain all properties/values
     * @param object Object to convert.
     * @returns String formatted.
     */
    private static convertObjectToParameters(object: any): GenericPrintReportParameters {
        const _parameters = <GenericPrintReportParameters>{
            parametersNames: [],
            parametersValues: [],
        };

        Object.entries(object).forEach(([key, value]) => {
            const valueFormatted = GenericPrintPayloadMapping.xtremToLegacyDataFormatting(value);
            if (valueFormatted !== undefined) {
                _parameters.parametersNames.push(key);
                _parameters.parametersValues.push(valueFormatted);
            }
        });

        return _parameters;
    }

    /**
     * Create a updated value of object ready to export (JSON) / specif format
     * @param payloadContent
     * @return updated payload
     */
    private static updatePrintPayload(payloadContent: any): any {
        const _UpdatedPayloadContent: any = {};
        const _payloadContent = JSON.parse(GenericPrintPayloadMapping.stringifyContent(payloadContent));

        // prepare object to export without processing any object : boolean, date,number, string only
        Object.entries(_payloadContent).forEach(([key, value]) => {
            const valueComputed = GenericPrintPayloadMapping.xtremToLegacyDataFormatting(value);
            if (valueComputed !== undefined) {
                _UpdatedPayloadContent[key] = valueComputed;
            }
        });

        return _UpdatedPayloadContent;
    }

    /**
     * Build the page payload from the graphql one
     * @param graphQLPayload
     * @return parameters values
     */
    static buildGraphQLPayload(graphQLPayload: any): GenericPrintReportParameters {
        const _parameters = <GenericPrintReportParameters>{
            parametersNames: [],
            parametersValues: [],
        };

        // Custom parameters has object
        if (graphQLPayload == null || typeof graphQLPayload !== 'object' || Object.keys(graphQLPayload).length === 0) {
            throw Error('Custom parameters is mandatory');
        }
        const updatedGraphQLPayload = GenericPrintPayloadMapping.updatePrintPayload(graphQLPayload);
        return GenericPrintPayloadMapping.convertObjectToParameters(updatedGraphQLPayload);
    }
}
