"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GenericPrintPayloadMapping = void 0;
const xtrem_decimal_1 = require("@sage/xtrem-decimal"), typesLib = xtrem_decimal_1;
const xtrem_date_time_1 = require("@sage/xtrem-date-time");
/** @internal */
class GenericPrintPayloadMapping {
    /**
     * Parse xtrem variable such as boolean and date to X3 format
     * @param valueFromPayload
     * @return value or undefined when type object is not authorized.
     */
    static xtremToLegacyDataFormatting(value) {
        let valueFromPayload = value;
        /**
         * check if the variable is an instance of date
         */
        if (valueFromPayload instanceof xtrem_date_time_1.DateValue) {
            valueFromPayload = valueFromPayload.value;
        }
        else if (typesLib.strictEq(typeof valueFromPayload, 'object') || typesLib.strictEq(typeof valueFromPayload, 'function')) {
            /**
             * check if the variable is an unsupported object instance or function
             */
            return undefined;
        }
        /**
         * check if is in date format YYYY-MM-DD
         */
        const found = valueFromPayload?.toString().match(/([12]\d{3}-(0[1-9]|1[0-2])-(0[1-9]|[12]\d|3[01]))/g);
        if (typesLib.strictNe(found, null)) {
            return found?.toString().replace(/-/g, '');
        }
        /**
         * if boolean translate
         *  true -> 1 and false -> 0 to meet x3 variables
         */
        if (typesLib.strictEq(Boolean(valueFromPayload), valueFromPayload)) {
            return valueFromPayload ? '2' : '1';
        }
        return String(valueFromPayload);
    }
    /**
     *
     * @param content Format content entry
     * @returns JSON formatted objet
     */
    static stringifyContent(content) {
        return JSON.stringify(content, (_key, value) => {
            return value ?? '';
        }, 4);
    }
    /**
     * Convert a object to a string contain all properties/values
     * @param object Object to convert.
     * @returns String formatted.
     */
    static convertObjectToParameters(object) {
        const _parameters = {
            parametersNames: [],
            parametersValues: [],
        };
        Object.entries(object).forEach(([key, value]) => {
            const valueFormatted = GenericPrintPayloadMapping.xtremToLegacyDataFormatting(value);
            if (typesLib.strictNe(valueFormatted, undefined)) {
                _parameters.parametersNames.push(key);
                _parameters.parametersValues.push(valueFormatted);
            }
        });
        return _parameters;
    }
    /**
     * Create a updated value of object ready to export (JSON) / specif format
     * @param payloadContent
     * @return updated payload
     */
    static updatePrintPayload(payloadContent) {
        const _UpdatedPayloadContent = {};
        const _payloadContent = JSON.parse(GenericPrintPayloadMapping.stringifyContent(payloadContent));
        // prepare object to export without processing any object : boolean, date,number, string only
        Object.entries(_payloadContent).forEach(([key, value]) => {
            const valueComputed = GenericPrintPayloadMapping.xtremToLegacyDataFormatting(value);
            if (typesLib.strictNe(valueComputed, undefined)) {
                _UpdatedPayloadContent[key] = valueComputed;
            }
        });
        return _UpdatedPayloadContent;
    }
    /**
     * Build the page payload from the graphql one
     * @param graphQLPayload
     * @return parameters values
     */
    static buildGraphQLPayload(graphQLPayload) {
        const _parameters = {
            parametersNames: [],
            parametersValues: [],
        };
        // Custom parameters has object
        if (typesLib.eq(graphQLPayload, null) || typesLib.strictNe(typeof graphQLPayload, 'object') || typesLib.strictEq(Object.keys(graphQLPayload).length, 0)) {
            throw Error('Custom parameters is mandatory');
        }
        const updatedGraphQLPayload = GenericPrintPayloadMapping.updatePrintPayload(graphQLPayload);
        return GenericPrintPayloadMapping.convertObjectToParameters(updatedGraphQLPayload);
    }
}
exports.GenericPrintPayloadMapping = GenericPrintPayloadMapping;
//# sourceMappingURL=generic-print-payload-mapping.js.map