"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readSerialNumberFromStockId = exports.handleFilterOnChange = exports.generateStockTableFilter = exports.removeFilters = exports.findSetting = exports.managePages = void 0;
const xtrem_decimal_1 = require("@sage/xtrem-decimal"), typesLib = xtrem_decimal_1;
const dialogs_1 = require("@sage/x3-master-data/lib/client-functions/dialogs");
const xtrem_client_1 = require("@sage/xtrem-client");
const xtrem_shared_1 = require("@sage/xtrem-shared");
const ui = require("@sage/xtrem-ui");
/**
 * Initialize manage page
 * @param pageInstance current page
 * @param stockSite stock site
 * @param productSite partial product site object
 * @param flowType type of flow
 * @param errorMessage error messages missing settings
 * @param stockSearchFilters mandatory array to return settings
 */
//todo:
function managePages(pageInstance, stockSite, productSite, flowType, errorMessage, stockSearchFilters) {
    // Initializing array when undefined
    stockSearchFilters ??= [];
    if (stockSearchFilters.length) {
        // remove all items in array
        stockSearchFilters.length = 0;
    }
    if (!pageInstance || !stockSite || !productSite || !flowType) {
        throw new xtrem_shared_1.SystemError('Invalid arguments');
    }
    const _selectedMobileSettings = JSON.parse(pageInstance.$.queryParameters?.mobileSettings);
    if (typesLib.strictEq(_selectedMobileSettings, null)) {
        throw new Error(errorMessage);
    }
    // Store value in array only when usable and return true when done
    const _pushSettings = (field) => {
        if (typesLib.strictNe(field, 'none')) {
            stockSearchFilters?.push(field);
            return true;
        }
        return false;
    };
    // The fields are memorized until they meet the none type.
    const _allPushed = _pushSettings(_selectedMobileSettings.stockField1) &&
        _pushSettings(_selectedMobileSettings.stockField2) &&
        _pushSettings(_selectedMobileSettings.stockField3) &&
        _pushSettings(_selectedMobileSettings.stockField4) &&
        _pushSettings(_selectedMobileSettings.stockField5) &&
        _pushSettings(_selectedMobileSettings.stockField6) &&
        _pushSettings(_selectedMobileSettings.stockField7) &&
        _pushSettings(_selectedMobileSettings.stockField8);
    if (pageInstance._pageMetadata.layoutFields?.location) {
        if (!findSetting('location', stockSearchFilters) || typesLib.strictNe(productSite.isLocationManaged, true)) {
            pageInstance['location'].isHidden = true;
        }
    }
    if (pageInstance._pageMetadata.layoutFields?.lot) {
        if (!findSetting('lot', stockSearchFilters) || typesLib.strictEq((productSite?.product?.lotManagementMode ?? 'notManaged'), 'notManaged')) {
            pageInstance['lot'].isHidden = true;
        }
    }
    if (pageInstance._pageMetadata.layoutFields?.sublot)
        if (!findSetting('sublot', stockSearchFilters) || typesLib.strictNe((productSite?.product?.lotManagementMode ?? ''), 'lotAndSublot')) {
            pageInstance['sublot'].isHidden = true;
        }
    if (pageInstance._pageMetadata.layoutFields?.serialNumber) {
        if (!findSetting('serial', stockSearchFilters) ||
            !['receivedIssued'].includes(productSite?.product?.serialNumberManagementMode ?? 'receivedIssued')) {
            pageInstance['serialNumber'].isHidden = true;
        }
    }
    if (pageInstance._pageMetadata.layoutFields?.status) {
        if (!findSetting('status', stockSearchFilters)) {
            pageInstance['status'].isHidden = true;
        }
    }
    if (pageInstance._pageMetadata.layoutFields?.packingUnit) {
        if (!findSetting('packingUnit', stockSearchFilters)) {
            pageInstance['packingUnit'].isHidden = true;
        }
    }
    if (pageInstance._pageMetadata.layoutFields?.packingUnitToStockUnitConversionFactor) {
        if (!findSetting('pacStkConversionFactor', stockSearchFilters)) {
            pageInstance['packingUnitToStockUnitConversionFactor'].isHidden = true;
        }
    }
    if (pageInstance._pageMetadata.layoutFields?.identifier1) {
        if (!findSetting('identifier1', stockSearchFilters)) {
            pageInstance['identifier1'].isHidden = true;
        }
    }
    if (pageInstance._pageMetadata.layoutFields?.identifier2) {
        if (!findSetting('identifier2', stockSearchFilters)) {
            pageInstance['identifier2'].isHidden = true;
        }
    }
    if (pageInstance._pageMetadata.layoutFields?.licensePlateNumber) {
        if (typesLib.strictNe(productSite.isLicensePlateNumberManaged, true) ||
            !findSetting('licensePlateNumber', stockSearchFilters)) {
            pageInstance['licensePlateNumber'].isHidden = true;
        }
    }
    if (pageInstance._pageMetadata.layoutFields?.stockCustomField1) {
        if (!findSetting('customField1', stockSearchFilters)) {
            pageInstance['stockCustomField1'].isHidden = true;
        }
    }
    if (pageInstance._pageMetadata.layoutFields?.stockCustomField2) {
        if (!findSetting('customField2', stockSearchFilters)) {
            pageInstance['stockCustomField2'].isHidden = true;
        }
    }
}
exports.managePages = managePages;
/**
 * Search if filter
 * @param field current search field
 * @param fields list of current filter
 * @returns true when found used filter parameter
 */
function findSetting(field, fields) {
    return typesLib.gte(fields.findIndex(fld => typesLib.strictEq(fld, field)), 0);
}
exports.findSetting = findSetting;
/**
 * Remove filter (asynchronous for avoiding anticipated onChange)
 * @param pageInstance current page
 */
async function removeFilters(pageInstance) {
    if (pageInstance._pageMetadata.layoutFields?.location)
        pageInstance['location'].value = null;
    if (pageInstance._pageMetadata.layoutFields?.lot)
        pageInstance['lot'].value = null;
    if (pageInstance._pageMetadata.layoutFields?.sublot)
        pageInstance['sublot'].value = null;
    if (pageInstance._pageMetadata.layoutFields?.serialNumber)
        pageInstance['serialNumber'].value = null;
    if (pageInstance._pageMetadata.layoutFields?.status)
        pageInstance['status'].value = null;
    if (pageInstance._pageMetadata.layoutFields?.packingUnit)
        pageInstance['packingUnit'].value = null;
    if (pageInstance._pageMetadata.layoutFields?.packingUnitToStockUnitConversionFactor)
        pageInstance['packingUnitToStockUnitConversionFactor'].value = null;
    if (pageInstance._pageMetadata.layoutFields?.identifier1)
        pageInstance['identifier1'].value = null;
    if (pageInstance._pageMetadata.layoutFields?.identifier2)
        pageInstance['identifier2'].value = null;
    if (pageInstance._pageMetadata.layoutFields?.licensePlateNumber)
        pageInstance['licensePlateNumber'].value = null;
    if (pageInstance._pageMetadata.layoutFields?.stockCustomField1)
        pageInstance['stockCustomField1'].value = null;
    if (pageInstance._pageMetadata.layoutFields?.stockCustomField2)
        pageInstance['stockCustomField2'].value = null;
}
exports.removeFilters = removeFilters;
function generateStockTableFilter(pageInstance) {
    const stockSite = pageInstance['_stockSite'];
    const stockFilter = {
        availableQuantity: { _gt: '0' },
        //      quantityInPackingUnit: { _gt: '0' },
        isBeingCounted: false,
        owner: stockSite?.code,
    };
    let parameterLocation = pageInstance.$.queryParameters?.location;
    if (!parameterLocation &&
        pageInstance._pageMetadata.layoutFields?.location &&
        pageInstance['location'].value) {
        parameterLocation = pageInstance['location'].value?.code;
    }
    const quantityToPick = pageInstance['_quantityToPick'] ?? null;
    const beginSerialNumber = pageInstance['serialNumber']?.value?.code ?? null;
    const endSerialNumber = beginSerialNumber && quantityToPick ? _calculateEndingSerialNumber(beginSerialNumber, quantityToPick) : null;
    const isFilterSerialNumber = pageInstance['_isFilterSerialNumber'] ?? false;
    const filterLicensePlateNumber = pageInstance._pageMetadata.layoutFields?.licensePlateNumber
        ? pageInstance['licensePlateNumber'].value?.code
            ? { licensePlateNumber: { code: pageInstance['licensePlateNumber'].value.code } }
            : {}
        : pageInstance.$.queryParameters?.licensePlateNumberOrigin
            ? { licensePlateNumber: { code: pageInstance.$.queryParameters.licensePlateNumberOrigin } }
            : { licensePlateNumber: { code: { _in: [undefined, ''] } } };
    return {
        ...stockFilter,
        ...(pageInstance._pageMetadata.layoutFields?.product &&
            pageInstance['product'].value?.code && {
            product: { product: { code: pageInstance['product'].value.code } },
        }),
        ...filterLicensePlateNumber,
        ...(pageInstance._pageMetadata.layoutFields?.location &&
            !pageInstance['location'].isHidden &&
            pageInstance['location'].value && {
            location: { code: parameterLocation },
        }),
        // lot and sublot are reference fields bound to the same Lot node
        // so first check if sublot field is enabled with a value. If so, filter for both lot and sublot
        // otherwise check if lot field is enabled with a value. If so filter for lot and additionally sublot, if applicable
        ...((pageInstance._pageMetadata.layoutFields?.lot &&
            !pageInstance['lot'].isHidden &&
            pageInstance['lot'].value?.lot && {
            lot: pageInstance['lot'].value.lot,
            ...(pageInstance['lot'].value.sublot && {
                sublot: pageInstance['lot'].value.sublot,
            }),
        }) ||
            (pageInstance._pageMetadata.layoutFields?.sublot &&
                !pageInstance['sublot'].isHidden &&
                pageInstance['sublot'].value?.sublot && {
                lot: pageInstance['sublot'].value.lot,
                sublot: pageInstance['sublot'].value.sublot,
            })),
        //...(this.sublot.value?.sublot && { sublot: this.sublot.value.sublot }),
        ...(pageInstance._pageMetadata.layoutFields?.serialNumber &&
            !isFilterSerialNumber &&
            !pageInstance['serialNumber'].isHidden &&
            pageInstance['serialNumber'].value?.code && {
            serialNumber: pageInstance['serialNumber'].value.code,
        }),
        ...(pageInstance._pageMetadata.layoutFields?.serialNumber &&
            isFilterSerialNumber &&
            beginSerialNumber &&
            endSerialNumber && {
            serialNumber: {
                _gte: beginSerialNumber,
                //                  _lte: endSerialNumber,
            },
        }),
        ...(pageInstance._pageMetadata.layoutFields?.status &&
            !pageInstance['status'].isHidden &&
            pageInstance['status'].value && { status: { code: pageInstance['status'].value } }),
        // ...(this.packingUnit.value && { packingUnit: this._unitMap.get(this.packingUnit.value).unit }),
        ...(pageInstance._pageMetadata.layoutFields?.packingUnit &&
            !pageInstance['packingUnit'].isHidden &&
            pageInstance['packingUnit'].value && {
            packingUnit: { code: pageInstance['packingUnit'].value },
        }),
        ...(pageInstance._pageMetadata.layoutFields?.packingUnitToStockUnitConversionFactor &&
            !pageInstance['packingUnitToStockUnitConversionFactor'].isHidden &&
            pageInstance['packingUnitToStockUnitConversionFactor'].value && {
            packingUnitToStockUnitConversionFactor: pageInstance['packingUnitToStockUnitConversionFactor']
                .value,
        }),
        ...(pageInstance._pageMetadata.layoutFields?.identifier1 &&
            !pageInstance['identifier1'].isHidden &&
            pageInstance['identifier1'].value && { identifier1: pageInstance['identifier1'].value }),
        ...(pageInstance._pageMetadata.layoutFields?.identifier2 &&
            !pageInstance['identifier2'].isHidden &&
            pageInstance['identifier2'].value && { identifier2: pageInstance['identifier2'].value }),
        ...(pageInstance._pageMetadata.layoutFields?.stockCustomField1 &&
            !pageInstance['stockCustomField1'].isHidden &&
            pageInstance['stockCustomField1'].value && {
            stockCustomField1: pageInstance['stockCustomField1'].value,
        }),
        ...(pageInstance._pageMetadata.layoutFields?.stockCustomField2 &&
            !pageInstance['stockCustomField2'].isHidden &&
            pageInstance['stockCustomField2'].value && {
            stockCustomField2: pageInstance['stockCustomField2'].value,
        }),
    };
}
exports.generateStockTableFilter = generateStockTableFilter;
function _calculateEndingSerialNumber(startingSerialNumber, quantity) {
    return startingSerialNumber.replace(/\d+$/, match => {
        const endingNumber = (typesLib.sub(typesLib.add(Number(match), quantity), 1)).toString();
        const lengthDiff = typesLib.max(typesLib.sub(endingNumber.length, match.length), 0);
        return endingNumber.padStart(typesLib.add(match.length, lengthDiff), '0');
    });
}
async function handleFilterOnChange(pageInstance, field, filterCriterion = null) {
    // if field value is cleared out during onChange, then delete the corresponding filter property in stock table's filter
    const stockKey = field.id;
    if (!stockKey) {
        //throw new SystemError('Invalid stock property to filter');
        return; // do nothing?
    }
    if (!pageInstance._pageMetadata.layoutFields?.stock) {
        //throw new SystemError('Invalid stock property to pageInstance');
        return; // do nothing?
    }
    const stock = pageInstance['stock'];
    const stockChangeLines = pageInstance['_stockChangeLines'] ?? null;
    const miscellaneousStockTransactionLines = pageInstance['_miscellaneousStockTransactionLines'] ?? null;
    const currentLine = pageInstance['_currentLine'];
    const currentStockChangeLine = stockChangeLines ? (stockChangeLines[currentLine] ?? null) : null;
    const currentMiscellaneousStockTransactionLine = miscellaneousStockTransactionLines
        ? (miscellaneousStockTransactionLines[currentLine] ?? null)
        : null;
    const quantityToPick = pageInstance['_quantityToPick'] ?? null;
    const beginSerialNumber = pageInstance['serialNumber']?.value?.code ?? null;
    const endSerialNumber = beginSerialNumber && quantityToPick ? _calculateEndingSerialNumber(beginSerialNumber, quantityToPick) : null;
    const isFilterSerialNumber = pageInstance['_isFilterSerialNumber'] ?? false;
    // When for the tuple the value is undefined, the key is removed from the filter.
    if (!field.value) {
        stock.filter = {
            ...stock.filter,
            ...(typesLib.strictEq(stockKey, 'location') && {
                ...(typesLib.strictEq(pageInstance.$.page.id, 'MobileSubcontractTransferDetails') && {
                    [stockKey]: { category: { _nin: ['subcontract', 'customer'] } },
                }),
                ...(typesLib.strictNe(pageInstance.$.page.id, 'MobileSubcontractTransferDetails') && { [stockKey]: undefined }),
            }),
            ...(!['location'].includes(stockKey) && { [stockKey]: undefined }),
        };
    }
    else {
        stock.filter = {
            ...stock.filter,
            ...(typesLib.strictEq(stockKey, 'packingUnit') &&
                pageInstance._pageMetadata.layoutFields?.packingUnit && {
                [stockKey]: {
                    code: pageInstance[stockKey].value,
                },
            }),
            ...(typesLib.strictEq(stockKey, 'status') && {
                [stockKey]: {
                    code: filterCriterion ??
                        (field instanceof ui.fields.Reference && field?.valueField
                            ? { [field.valueField]: field.value[field.valueField] }
                            : field.value),
                },
            }),
            ...(typesLib.strictEq(stockKey, 'serialNumber') &&
                isFilterSerialNumber &&
                beginSerialNumber &&
                endSerialNumber && {
                [stockKey]: {
                    _gte: beginSerialNumber,
                    //                      _lte: endSerialNumber,
                },
            }),
            ...(typesLib.strictEq(stockKey, 'serialNumber') &&
                !isFilterSerialNumber && {
                [stockKey]: {
                    _eq: beginSerialNumber,
                },
            }),
            // All other cases
            ...(!['packingUnit', 'status', 'serialNumber'].includes(stockKey) && {
                [stockKey]: filterCriterion ??
                    (field instanceof ui.fields.Reference && field?.valueField
                        ? { [field.valueField]: field.value[field.valueField] }
                        : field.value),
            }),
        };
        // stock.pageSize = 1000;
    }
    await _setStockQuantityInPackingUnit(pageInstance, stock, currentStockChangeLine ?? currentMiscellaneousStockTransactionLine);
    if (!field.value) {
        field.getNextField(true)?.focus();
    }
}
exports.handleFilterOnChange = handleFilterOnChange;
/**
 * _setStockQuantityInPackingUnit
 * @param pageInstance current page
 * @param stock stock table
 * @param currentStockLine stock line for lpnOperations or MiscellaneousIssue
 */
async function _setStockQuantityInPackingUnit(pageInstance, stock, currentStockLine) {
    const _stockDetails = currentStockLine?.stockDetails;
    if (_stockDetails) {
        stock.selectedRecords.forEach((rowId) => {
            const stockRecord = stock.getRecordValue(rowId);
            if (stockRecord) {
                const qtyTotal = _stockDetails.reduce((acc, curr) => {
                    return typesLib.add(acc, Number(curr.quantityInPackingUnit));
                }, 0);
                stockRecord.quantityInPackingUnit = String(qtyTotal);
                stock.setRecordValue(stockRecord);
            }
        });
        await pageInstance.$.commitValueAndPropertyChanges();
    }
}
/**
 * Read one serial number for a given stock id
 * @param pageInstance current page
 * @param stockId stock id
 * @param orderBy sort order
 * @returns serial number or null
 */
async function readSerialNumberFromStockId(pageInstance, stockId, orderBy) {
    if (stockId) {
        try {
            const serialNumber = (0, xtrem_client_1.extractEdges)(await pageInstance.$.graph
                .node('@sage/x3-stock-data/SerialNumber')
                .query(ui.queryUtils.edgesSelector({
                code: true,
            }, {
                filter: { stockId },
                orderBy: { code: orderBy },
            }))
                .execute());
            if (serialNumber[0]) {
                return serialNumber[0];
            }
        }
        catch (e) {
            await (0, dialogs_1.dialogMessage)(pageInstance, 'error', ui.localize('@sage/x3-stock/error-loading-serial-number-node', 'Error loading serial number node'), String(e));
        }
    }
    return null;
}
exports.readSerialNumberFromStockId = readSerialNumberFromStockId;
//# sourceMappingURL=manage-pages.js.map